﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/svc/svc_Kernel.h>
#include <nn/nn_BitTypes.h>
#include "kern_Platform.h"
#include "kern_Assert.h"
#include "kern_Kernel.h"
#include "kern_KDeviceAddressSpace.h"
#include "kern_KProcess.h"
#include "kern_PageTableSelect.h"
#include "kern_KMemoryManager.h"

namespace nn { namespace kern {

NN_AUTOOBJECT_DEFINE_TYPE_NAME(KDeviceAddressSpace);

KDeviceAddressSpace::KDeviceAddressSpace(): m_IsInitialized(false)
{
    NN_KERN_THIS_ASSERT();
}

KDeviceAddressSpace::~KDeviceAddressSpace()
{
    NN_KERN_THIS_ASSERT();
}

Result KDeviceAddressSpace::Initialize(uint64_t spaceAddress, uint64_t spaceSize)
{
    NN_KERN_THIS_ASSERT();

    Result result;

    result = m_Table.Initialize(spaceAddress, spaceSize);
    if (result.IsFailure())
    {
        return result;
    }

    m_SpaceAddress = spaceAddress;
    m_SpaceSize = spaceSize;
    m_IsInitialized = true;

    return result;
}

void KDeviceAddressSpace::Finalize()
{
    NN_KERN_THIS_ASSERT();

    m_Table.Finalize();

    KObjectAdaptor<KDeviceAddressSpace, KAutoObjectWithList>::Finalize();
}

Result KDeviceAddressSpace::Map(size_t* pMapSize, KProcessPageTable* pPageTable,
            KProcessAddress processAddress, size_t size,
            uint64_t deviceAddress,
            nn::svc::MemoryPermission devicePermission,
            bool isAligned, bool breakAllocate)
{
    NN_KERN_THIS_ASSERT();
    if (!(m_SpaceAddress <= deviceAddress && deviceAddress + size - 1 <= m_SpaceAddress + m_SpaceSize - 1))
    {
        return nn::svc::ResultInvalidRegion();
    }

    Result result;
    KScopedLightLock locker(&m_Lock);

    KScopedPageGroup pg(pPageTable->GetBlockInfoManager());

    result = pPageTable->LockForDeviceAddressSpace(&pg.GetPageGroup(), processAddress, size,
            ConvertToKMemoryPermission(devicePermission), isAligned);
    if (result.IsFailure())
    {
        return result;
    }

    result = m_Table.Map(pMapSize, pg.GetPageGroup(), deviceAddress, devicePermission, breakAllocate);
    if (result.IsFailure())
    {
        *pMapSize = 0;
    }

    if (*pMapSize != size)
    {
        Result resultCheck = pPageTable->UnlockForDeviceAddressSpace(processAddress + *pMapSize, size - *pMapSize);
        if (resultCheck.IsFailure())
        {
            NN_WARNING(false, "failed to unlock DeviceAddressSpace");
        }
    }

    pg.GetPageGroup().Close();

    return result;
}

Result KDeviceAddressSpace::Unmap(KProcessPageTable* pPageTable,
            KProcessAddress processAddress, size_t size,
            uint64_t deviceAddress)
{
    NN_KERN_THIS_ASSERT();
    if (!(m_SpaceAddress <= deviceAddress && deviceAddress + size - 1 <= m_SpaceAddress + m_SpaceSize - 1))
    {
        return nn::svc::ResultInvalidCurrentMemory();
    }

    Result result;
    KScopedLightLock locker(&m_Lock);

    KScopedPageGroup pg(pPageTable->GetBlockInfoManager());

    result = pPageTable->MakePageGroupContiguousAndOpen(&pg.GetPageGroup(), processAddress, size / NN_KERN_FINEST_PAGE_SIZE,
            KMemoryState_FlagsDeviceShared, KMemoryState_FlagsDeviceShared,
            0, 0,
            KMemoryAttribute_Locked | KMemoryAttribute_DeviceShared, KMemoryAttribute_DeviceShared);
    if (result.IsFailure())
    {
        return result;
    }

    result = m_Table.Unmap(pg.GetPageGroup(), deviceAddress);
    if (result.IsFailure())
    {
        pg.GetPageGroup().Close();
        return result;
    }

    result = pPageTable->UnlockForDeviceAddressSpace(processAddress, size);
    if (result.IsFailure())
    {
        NN_WARNING(false, "failed to unlock DeviceAddressSpace");
    }

    pg.GetPageGroup().Close();

    return result;
}

Result KDeviceAddressSpace::Attach(nn::svc::DeviceName deviceName)
{
    KScopedLightLock locker(&m_Lock);
    return m_Table.Attach(deviceName, m_SpaceAddress, m_SpaceSize);
}

Result KDeviceAddressSpace::Detach(nn::svc::DeviceName deviceName)
{
    KScopedLightLock locker(&m_Lock);
    return m_Table.Detach(deviceName);
}

void KDeviceAddressSpace::Initialize()
{
    KDevicePageTable::Initialize();
}


}}

