﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/*!
    @file

    @brief     カーネルケイパビリティの定義です。

*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_BitTypes.h>
#include <nn/util/util_BitPack.h>
#include <nn/svc/svc_Kernel.h>
#include <cstring>
#include "kern_PageTableSelect.h"
#include "svc/kern_KUserPointer.h"

namespace nn { namespace kern {
/*!
    @brief      ケイパビリティ クラスの定義です。

    ケイパビリティはセキュリティの仕組みです。
    各プロセスに対して、役目に応じた最小限の権限を与えることで
    セキュリティリスクから他のプロセスやカーネル自体を保護します。

*/
class KCapability
{
public: // TORIAEZU
    static const int SVC_CAPABILITY_BYTES       = 16;
    static const int INTERRUPT_CAPABILITY_BYTES = 128;

private:
    static const uint64_t PhysicalMapMask = ((1ull << 36) - 1);
    static const int PaddingInterruptNumber = 1023;

    Bit8                m_SvcCapability[SVC_CAPABILITY_BYTES];              // 128bit
    Bit8                m_InterruptCapability[INTERRUPT_CAPABILITY_BYTES];
    Bit64               m_CoreMask;
    Bit64               m_PriorityMask;
    util::BitPack32     m_OtherCapability;                                  // その他のケイパビリティ
    int32_t             m_HandleTableSize;
    util::BitPack32     m_IntentedKernelVersion;
    Bit32               m_ProgramType;

    struct OtherCapability
    {
        typedef nn::util::BitPack32::Field<0, 1, bool> PermitDebug;
        typedef nn::util::BitPack32::Field<PermitDebug::Next, 1, bool> DebugProcess;
    };

    enum FlagTypeLength
    {
        PriorityCoreNoTypeLength    = 3,
        SvcTypeLength               = 4,
        MapPhysicalTypeLength       = 6,
        MapIoTypeLength             = 7,
        InterruptTypeLength         = 11,
        MiscTypeLength              = 13,
        VersionTypeLength           = 14,
        HandleTypeLength            = 15,
        OtherTypeLength             = 16
    };

    enum FlagType
    {
        PriorityCoreNoType  = (1u << PriorityCoreNoTypeLength) - 1,
        SvcType             = (1u << SvcTypeLength) - 1,
        MapPhysicalType     = (1u << MapPhysicalTypeLength) - 1,
        MapIoType           = (1u << MapIoTypeLength) - 1,
        InterruptType       = (1u << InterruptTypeLength) - 1,
        MiscType            = (1u << MiscTypeLength) - 1,
        VersionType         = (1u << VersionTypeLength) - 1,
        HandleType          = (1u << HandleTypeLength) - 1,
        OtherType           = (1u << OtherTypeLength) - 1,
        PaddingType         = 0xffffffff
    };

    // 一度しか初期化できない Flags
    static const Bit32 InitializeOnceFlags =
        (1u << PriorityCoreNoTypeLength) |
        (1u << MiscTypeLength) |
        (1u << VersionTypeLength) |
        (1u << HandleTypeLength) |
        (1u << OtherTypeLength);

    Bit32 GetFlagType(const nn::util::BitPack32 flag)
    {
        Bit32 v = flag.storage;
        return (~v & (v + 1)) - 1;
    }

    struct InterruptFlags
    {
        typedef nn::util::BitPack32::Field<InterruptTypeLength + 1, 10, uint32_t> Intr0;
        typedef nn::util::BitPack32::Field<Intr0::Next, 10, uint32_t> Intr1;
    };

    struct MiscFlags
    {
        typedef nn::util::BitPack32::Field<MiscTypeLength + 1, 3, Bit32> ProgramType;
        typedef nn::util::BitPack32::Field<ProgramType::Next, 15, Bit32> Reserved;
    };

    struct VersionFlags
    {
        typedef nn::util::BitPack32::Field<VersionTypeLength + 1, 4, Bit32> Minor;
        typedef nn::util::BitPack32::Field<Minor::Next, 13, Bit32> Major;
    };

    struct PriorityCoreNoFlags
    {
        typedef nn::util::BitPack32::Field<PriorityCoreNoTypeLength + 1, 6, uint32_t> PriorityLowest;
        typedef nn::util::BitPack32::Field<PriorityLowest::Next, 6, uint32_t> PriorityHighest;
        typedef nn::util::BitPack32::Field<PriorityHighest::Next, 8, uint32_t> CoreNoMin;
        typedef nn::util::BitPack32::Field<CoreNoMin::Next, 8, uint32_t> CoreNoMax;
    };

    struct SvcFlags
    {
        typedef nn::util::BitPack32::Field<SvcTypeLength + 1, 24, Bit32> Flags;
        typedef nn::util::BitPack32::Field<Flags::Next, 3, Bit32> Index;
    };

    struct HandleFlags
    {
        typedef nn::util::BitPack32::Field<HandleTypeLength + 1, 10, int32_t> MaxHandles;
        typedef nn::util::BitPack32::Field<MaxHandles::Next, 6, Bit32> Reserved;
    };

    struct OtherFlags
    {
        typedef nn::util::BitPack32::Field<OtherTypeLength + 1, 1, bool> PermitDebug;
        typedef nn::util::BitPack32::Field<PermitDebug::Next, 1, bool> DebugProcess;
        typedef nn::util::BitPack32::Field<DebugProcess::Next, 13, Bit32> Reserved;
    };

    struct MapPhysicalStartFlags
    {
        typedef nn::util::BitPack32::Field<MapPhysicalTypeLength + 1, 24, uintptr_t> Pfn;
        typedef nn::util::BitPack32::Field<Pfn::Next, 1, bool> ReadOnly;
    };

    struct MapPhysicalSizeFlags
    {
        typedef nn::util::BitPack32::Field<MapPhysicalTypeLength + 1, 20, size_t> Pages;
        typedef nn::util::BitPack32::Field<Pages::Next, 4, Bit32> Reserved;
        typedef nn::util::BitPack32::Field<Reserved::Next, 1, bool> Normal;
    };

    struct MapIoFlags
    {
        typedef nn::util::BitPack32::Field<MapIoTypeLength + 1, 24, Bit32> Pfn;
    };

private:
    Result SetVersionCapability(nn::util::BitPack32 flag);
    Result SetPriorityCoreNoCapability(nn::util::BitPack32 flag);
    Result SetMiscCapability(nn::util::BitPack32 flag);
    //! 割り込み許可のケイパビリティを設定します。
    Result SetInterruptCapability(nn::util::BitPack32 flag);
    //! システムコールの使用許可ケイパビリティを設定します。
    Result SetSvcCapability(Bit32* pSetSvcIndex, nn::util::BitPack32 flag);
    //! Otherケイパビリティを設定します。
    Result SetOtherCapability(nn::util::BitPack32 flag);
    //! ケイパビリティに基づいたメモリ空間のマッピングをします。
    Result MapPages(nn::util::BitPack32 begin, nn::util::BitPack32 end, KProcessPageTable* pPageTable);
    //! ケイパビリティに基づいたレジスタ空間のマッピングをします。
    Result MapIoPages(nn::util::BitPack32 flag, KProcessPageTable* pPageTable);

    Result SetCapability(Bit32* pSetFlags, Bit32* pSetSvcIndex,
            const nn::util::BitPack32 flag, KProcessPageTable* pPageTable);

    Result SetCapabilities(const Bit32 flags[], int32_t num, KProcessPageTable* pPageTable);
    Result SetCapabilities(svc::KUserPointer<const Bit32*> flags, int32_t num, KProcessPageTable* pPageTable);

public:
    /*!
        @brief     コンストラクタ

        CHECK: メンバー変数の初期化場所統一を推奨します
    */
    KCapability()
    {}

    /*!
        @brief     ケイパビリティクラスを初期化します。

        @param[in]    flags     ケイパビリティリスト
        @param[in]    numFlags       ケイパビリティリストの要素数
        @param[in]    pageTable          当該プロセスのページテーブル

        @return       ステータスを返します。

    */
    Result  Initialize(const Bit32 flags[], int32_t num, KProcessPageTable* pPageTable);
    Result  Initialize(svc::KUserPointer<const Bit32*> flags, int32_t num, KProcessPageTable* pPageTable);

    /*!
        @brief     ハンドルの使用可能な上限を取得します。
    */
    int32_t GetHandleTableSize() const { return m_HandleTableSize; }

    /*!
        @brief     指定された割り込み番号は使用可能か?
    */
    bool IsPermittedInterruptNumber(uint32_t intNo) const
    {
        if (intNo < NN_BITSIZEOF(m_InterruptCapability))
        {
            return m_InterruptCapability[intNo / NN_BITSIZEOF(Bit8)] & (1u << (intNo % NN_BITSIZEOF(Bit8)));
        }
        else
        {
            return false;
        }
    }

    Bit64 GetCoreMask()     const { return m_CoreMask; }
    Bit64 GetPriorityMask() const { return m_PriorityMask; }
    Bit32 GetIntentedKernelMajorVersion() const { return m_IntentedKernelVersion.Get<VersionFlags::Major>(); }
    Bit32 GetIntentedKernelMinorVersion() const { return m_IntentedKernelVersion.Get<VersionFlags::Minor>(); }

    // OtherCapability
    /*!
        @brief     デバッガーのアタッチを受け付けるか?

    */
    bool    IsPermittedDebug()        const { return m_OtherCapability.Get<OtherCapability::PermitDebug>(); }

    /*!
        @brief     デバッグを許可していないプロセスを強制的にデバッグ可能か?

    */
    bool     CanForceDebug ()         const { return m_OtherCapability.Get<OtherCapability::DebugProcess>(); }

    void CopySvcPermisionBitsTo(void* p)
    {
        std::memcpy(p, m_SvcCapability, sizeof(m_SvcCapability));
    }

private:
    bool PermitInterrupt(uint32_t intNo)
    {
        if (intNo < NN_BITSIZEOF(m_InterruptCapability))
        {
            m_InterruptCapability[intNo / NN_BITSIZEOF(Bit8)] |= (1u << (intNo % NN_BITSIZEOF(Bit8)));
            return true;
        }
        else
        {
            return false;
        }
    }

    bool PermitSvc(uint32_t svcNo)
    {
        if (svcNo < NN_BITSIZEOF(m_SvcCapability))
        {
            m_SvcCapability[svcNo / NN_BITSIZEOF(Bit8)] |= (1u << (svcNo % NN_BITSIZEOF(Bit8)));
            return true;
        }
        else
        {
            return false;
        }
    }
};


}}

