﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/svc/svc_Kernel.h>
#include "kern_Platform.h"
#include "kern_InterlockedSelect.h"
#include "kern_KAddressArbiter.h"
#include "kern_KScheduler.h"
#include "kern_MemoryCopySelect.h"
#include "kern_KScopedSchedulingLock.h"
#include "kern_KScopedSchedulingLockAndSleep.h"
#include "kern_KProcess.h"
#include "kern_Current.h"

namespace nn { namespace kern {
namespace
{

inline bool ReadUserSpace(int32_t* p, KProcessAddress addr)
{
    return CopyMemoryFromUserOnly4(p, GetUntypedPointer(addr));
}

inline bool AccessOk(KProcessAddress addr)
{
    return KCPU::CanAtomicAccess(addr);
}

struct DecrementIfLessThan
{
    int32_t comparand;
    int32_t result;

    bool operator()(int32_t* x)
    {
        result = *x;
        if (result < comparand)
        {
            --*x;
            return true;
        }
        else
        {
            return false;
        }
    }
};

bool DecrementIfLessThanUserSpaceAtomic(int32_t* p, uintptr_t addr, int32_t comparand)
{
    KDisableInterrupt di;

    if (!AccessOk(addr))
    {
        return false;
    }

    DecrementIfLessThan updater;
    updater.comparand = comparand;

    Interlocked::AtomicUpdate(reinterpret_cast<int32_t*>(addr), &updater);
    *p = updater.result;

    return true;
}

struct UpdateIfEqual
{
    int32_t comparand;
    int32_t old;
    int32_t newValue;

    bool operator()(int32_t* x)
    {
        old = *x;
        if (old == comparand)
        {
            *x = newValue;
            return true;
        }
        else
        {
            return false;
        }
    }
};

bool UpdateIfEqualUserSpaceAtomic(int32_t* p, uintptr_t addr, int32_t comparand, int32_t newValue)
{
    KDisableInterrupt di;

    if (!AccessOk(addr))
    {
        return false;
    }

    UpdateIfEqual updater;
    updater.comparand = comparand;
    updater.newValue = newValue;

    Interlocked::AtomicUpdate(reinterpret_cast<int32_t*>(addr), &updater);
    *p = updater.old;

    return true;
}

KThread s_Comp;
}


Result KAddressArbiter::WaitIfLessThan(uintptr_t addr, int32_t value, bool dec, int64_t timeout)
{
    KThread* pThread = &GetCurrentThread();
    KHardwareTimer* pTimer;

    {
        KScopedSchedulingLockAndSleep sleep(&pTimer, pThread, timeout);
        if (pThread->IsTerminateRequested())
        {
            sleep.Cancel();
            return nn::svc::ResultTerminateRequested();
        }

        pThread->SetSyncedObject(nullptr, nn::svc::ResultTimeout());

        bool bSuccess;
        int32_t test;

        if (dec)
        {
            bSuccess = DecrementIfLessThanUserSpaceAtomic(&test, addr, value);
        }
        else
        {
            bSuccess = ReadUserSpace(&test, addr);
        }

        if (!bSuccess)
        {
            sleep.Cancel();
            return nn::svc::ResultInvalidCurrentMemory();
        }

        if (test >= value)
        {
            sleep.Cancel();
            return nn::svc::ResultInvalidState();
        }

        if (timeout == 0)
        {
            sleep.Cancel();
            return nn::svc::ResultTimeout();
        }

        pThread->SetAddressArbiter(addr);
        m_List.insert(*pThread);
        pThread->SetState(KThread::STATE_WAIT);
    }

    // タイムアウトで待ち解除されても、ここでSignalされることがある。
    if (pTimer != nullptr)
    {
        // タイマーのキャンセル
        pTimer->CancelTask(pThread);
    }

    {
        // タイムアウト、プロセス終了時などで、Signalされていないのに待ちが解除されることがある。
        KScopedSchedulingLock locker;
        if (pThread->InWaitingForAddressArbiter())
        {
            m_List.erase(m_List.iterator_to(*pThread));
            pThread->ClearAddressArbiter();
        }
    }

    KSynchronizationObject* dummy = nullptr;
    Result ret = pThread->GetWaitResult(&dummy);

    return ret;
}

Result KAddressArbiter::WaitIfEqual(uintptr_t addr, int32_t value, int64_t timeout)
{
    KThread* pThread = &GetCurrentThread();
    KHardwareTimer* pTimer;

    {
        KScopedSchedulingLockAndSleep sleep(&pTimer, pThread, timeout);
        if (pThread->IsTerminateRequested())
        {
            sleep.Cancel();
            return nn::svc::ResultTerminateRequested();
        }

        pThread->SetSyncedObject(nullptr, nn::svc::ResultTimeout());

        bool bSuccess;
        int32_t test;

        bSuccess = ReadUserSpace(&test, addr);

        if (!bSuccess)
        {
            sleep.Cancel();
            return nn::svc::ResultInvalidCurrentMemory();
        }

        if (test != value)
        {
            sleep.Cancel();
            return nn::svc::ResultInvalidState();
        }

        if (timeout == 0)
        {
            sleep.Cancel();
            return nn::svc::ResultTimeout();
        }

        pThread->SetAddressArbiter(addr);
        m_List.insert(*pThread);
        pThread->SetState(KThread::STATE_WAIT);
    }

    // タイムアウトで待ち解除されても、ここでSignalされることがある。
    if (pTimer != nullptr)
    {
        // タイマーのキャンセル
        pTimer->CancelTask(pThread);
    }

    {
        // タイムアウト、プロセス終了時などで、Signalされていないのに待ちが解除されることがある。
        KScopedSchedulingLock locker;
        if (pThread->InWaitingForAddressArbiter())
        {
            m_List.erase(m_List.iterator_to(*pThread));
            pThread->ClearAddressArbiter();
        }
    }

    KSynchronizationObject* dummy = nullptr;
    Result ret = pThread->GetWaitResult(&dummy);

    return ret;
}

Result KAddressArbiter::Signal(uintptr_t addr, int32_t num)
{
    int numWaiter = 0;
    {
        KScopedSchedulingLock locker;
        s_Comp.SetupForAddressArbiterCompare(addr, -1);
        AddressArbiterList::iterator it = m_List.nfind(s_Comp);

        while ((it != m_List.end()) && (num <= 0 || numWaiter < num) && (it->GetAddressKey() == addr))
        {
            it->SetSyncedObject(nullptr, ResultSuccess());
            NN_KERN_ASSERT(it->InWaitingForAddressArbiter());
            it->WakeUp();
            it->ClearAddressArbiter();
            it = m_List.erase(it);
            ++numWaiter;
        }
    }
    return ResultSuccess();
}

// For Manual Clear
Result KAddressArbiter::SignalIncrementIfEqual(uintptr_t addr, int32_t value, int32_t num)
{
    int numWaiter = 0;
    {
        KScopedSchedulingLock locker;
        s_Comp.SetupForAddressArbiterCompare(addr, -1);
        AddressArbiterList::iterator it = m_List.nfind(s_Comp);

        int32_t test;
        bool bSuccess;

        bSuccess = UpdateIfEqualUserSpaceAtomic(&test, addr, value, value + 1);
        if (!bSuccess)
        {
            return nn::svc::ResultInvalidCurrentMemory();
        }
        if (test != value)
        {
            return nn::svc::ResultInvalidState();
        }

        while ((it != m_List.end()) && (num <= 0 || numWaiter < num) && (it->GetAddressKey() == addr))
        {
            it->SetSyncedObject(nullptr, ResultSuccess());
            NN_KERN_ASSERT(it->InWaitingForAddressArbiter());
            it->WakeUp();
            it->ClearAddressArbiter();
            it = m_List.erase(it);
            ++numWaiter;
        }
    }
    return ResultSuccess();
}

// For Auto Clear
Result KAddressArbiter::SignalUpdateByCountIfEqual(uintptr_t addr, int32_t value, int32_t num)
{
    int numWaiter = 0;
    {
        KScopedSchedulingLock locker;
        s_Comp.SetupForAddressArbiterCompare(addr, -1);
        AddressArbiterList::iterator it = m_List.nfind(s_Comp);

        {
            int32_t newValue;
            if (num <= 0)
            {
                if ((it != m_List.end()) && (it->GetAddressKey() == addr))
                {
                    newValue = value - 1;
                }
                else
                {
                    newValue = value + 1;
                }
            }
            else
            {
                AddressArbiterList::iterator tmp = it;
                int n = 0;
                while ((tmp != m_List.end()) && (tmp->GetAddressKey() == addr) && n < num + 1)
                {
                    tmp++;
                    n++;
                }

                if (n == 0)
                {
                    newValue = value + 1;
                }
                else if (n <= num)
                {
                    newValue = value - 1;
                }
                else
                {
                    newValue = value;
                }
            }

            int32_t test;
            bool bSuccess;
            if (value != newValue)
            {
                bSuccess = UpdateIfEqualUserSpaceAtomic(&test, addr, value, newValue);
            }
            else
            {
                bSuccess = ReadUserSpace(&test, addr);
            }
            if (!bSuccess)
            {
                return nn::svc::ResultInvalidCurrentMemory();
            }
            if (test != value)
            {
                return nn::svc::ResultInvalidState();
            }
        }

        while ((it != m_List.end()) && (num <= 0 || numWaiter < num) && (it->GetAddressKey() == addr))
        {
            it->SetSyncedObject(nullptr, ResultSuccess());
            NN_KERN_ASSERT(it->InWaitingForAddressArbiter());
            it->WakeUp();
            it->ClearAddressArbiter();
            it = m_List.erase(it);
            ++numWaiter;
        }
    }
    return ResultSuccess();
}

}}
