﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/*!
    @file

    @brief ASSERT に関する API の宣言

*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/svc/svc_Kernel.h>
#include "kern_DebugString.h"
#include "kern_Panic.h"
#include "kern_Platform.h"
#include "kern_MemoryMap.h"
#include "kern_Macro.h"

/* ------------------------------------------------------------------------
        assert
   ------------------------------------------------------------------------ */

#ifndef NN_SWITCH_DISABLE_ASSERT_WARNING
#define NN_KERN_ASSERTMSG(exp, ...)                                                          \
    (void) (NN_LIKELY(exp) || ( NN_LOG("Failed assertion at %s:%d\n  ", __FILE__, __LINE__), \
                NN_LOG(__VA_ARGS__),                                                         \
                NN_LOG("\n"),                                                                \
                ::nn::dbg::Break(nn::svc::BreakReason_Assert),                               \
                0))
#define NN_KERN_ASSERTMSG_WITH_RESULT(exp, result, ...)  NN_KERN_ASSERTMSG(exp, __VA_ARGS__)
#else
    // assert 無効
    #define NN_KERN_ASSERTMSG(exp, ...)                      ((void)0)
    #define NN_KERN_ASSERTMSG_WITH_RESULT(exp, result, ...)  ((void)0)
#endif // NN_SWITCH_DISABLE_ASSERT_WARNING

#define NN_ANY_TO_UPTR(ptr)                     ((uintptr_t)((const void*)(ptr)))
#define NN_KERN_ASSERT(  exp)                   NN_KERN_ASSERTMSG(  (exp), "%s", #exp)
#define NN_KERN_RESULT_ASSERT(  exp)            NN_KERN_ASSERTMSG_WITH_RESULT(  (exp).IsSuccess(), (exp), "\"%s\" is Failure.", #exp)
#define NN_KERN_ASSERT_WITH_RESULT(exp, result) NN_KERN_ASSERTMSG_WITH_RESULT(   (exp), (result), "%s", #exp )
#define NN_KERN_NULL_ASSERT(  exp)              NN_KERN_ASSERTMSG(   (exp) != NULL, "%s must not be NULL", #exp )
#define NN_KERN_ALIGN_ASSERT(  exp, align)      NN_KERN_ASSERTMSG(   ((uintptr_t)(exp)) % (align) == 0, "%s(=0x%08x) must be %d byte aligned.", #exp, (exp), align )
#define NN_KERN_MIN_ASSERT(  exp, min)          NN_KERN_ASSERTMSG(   (exp) >= (min), "%s(=%d) must be >= %s(=%d).", #exp, (exp), #min, min )
#define NN_KERN_MAX_ASSERT(  exp, max)          NN_KERN_ASSERTMSG(   (exp) <= (max), "%s(=%d) must be <= %s(=%d).", #exp, (exp), #max, max )
#define NN_KERN_EQUAL_ASSERT(  exp, equ)        NN_KERN_ASSERTMSG(   (exp) == (equ), "%s(=%d) must be == %s(=%d).", #exp, (exp), #equ, equ )
#define NN_KERN_NOT_EQUAL_ASSERT(  exp, equ)    NN_KERN_ASSERTMSG(   (exp) != (equ), "%s(=%d) must be != %s(=%d).", #exp, (exp), #equ, equ )


#define NN_KERN_ABORT()                 NN_KERNEL_PANIC("panic at %s:%d", __FILE__, __LINE__)
#define NN_KERN_ABORT_UNLESS(exp)       (void) (NN_LIKELY(exp) || (NN_KERNEL_PANIC("%s at %s:%d", #exp, __FILE__, __LINE__), 0))
#define NN_KERN_ABORT_IF_FAILED(exp)    (void) (NN_LIKELY((exp).IsSuccess()) || (NN_KERNEL_PANIC("%x at %s:%d", (exp).GetInnerValueForDebug(), __FILE__, __LINE__), 0))


/* ------------------------------------------------------------------------
        warning
   ------------------------------------------------------------------------ */

#ifndef NN_SWITCH_DISABLE_ASSERT_WARNING
    #ifndef NN_SWITCH_DISABLE_DEBUG_PRINT
        #define NN_WARNING(exp, ...) \
            (void) (NN_LIKELY(exp) || (nndbgPrintWarning_(__FILE__, __LINE__, __VA_ARGS__), 0))
    #else
        #define NN_WARNING(exp, ...)  ((void)0)
    #endif
#else
    #define NN_WARNING(exp, ...)  ((void)0)
#endif // NN_SWITCH_DISABLE_ASSERT_WARNING

/*!
    @def       NN_KERN_IS_VALID_POINTER

    @brief     ポインタがカーネル内メモリ空間をさしているか判定します。

    @param[in]    ptr    有効かどうかを判定するポインタ。

*/
#define NN_KERN_IS_VALID_POINTER(ptr)       ((NN_KERN_V_ADDR_KERNEL <= NN_ANY_TO_UPTR(ptr)) && \
                                             (NN_ANY_TO_UPTR(ptr) < NN_KERN_V_ADDR_KERNEL_END))

/*!
    @def       NN_KERNEL_PAGE_ALIGN_ASSERT

    @brief     ページサイズでアライメントされているかどうかを判定します。

    @param[in]    exp    アライメントされているかをを判定する式。

*/
#define NN_KERNEL_PAGE_ALIGN_ASSERT(exp)    NN_KERN_ALIGN_ASSERT(exp, NN_KERN_FINEST_PAGE_SIZE)

/*!
    @def       NN_KERN_POINTER_ASSERT

    @brief     ポインタがカーネル内メモリ空間を指し示していなければメッセージを出力して停止します。

    @param[in]    p      有効かどうかを判定するポインタ。

*/
#define NN_KERN_POINTER_ASSERT(p)           NN_KERN_ASSERTMSG(NN_KERN_IS_VALID_POINTER(p), "%s(=%p) is invalid pointer", #p, (p))

/*!
    @def       NN_KERN_THIS_ASSERT

    @brief     thisポインタがカーネル内メモリ空間を指し示していなければメッセージを出力して停止します。

*/
#define NN_KERN_THIS_ASSERT()               NN_KERN_POINTER_ASSERT(this)

