﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/TargetConfigs/build_Base.h>
#include <nn/nn_Common.h>
#include <nn/nn_BitTypes.h>
#include "kern_MemoryMap.h"
#include "../../kern_ContextSelect.h"
#include "../../kern_InterlockedSelect.h"
#include "../ARM64/kern_RegisterAccess.h"
#include "../../kern_InterruptControllerSelect.h"

namespace nn { namespace kern {

class KThread;

namespace ARMv8A {

class KCPU
{
public:
#if defined NN_BUILD_CONFIG_CPU_CORTEX_A53_AARCH64
    static const size_t INSTRUCTION_CACHE_LINE_SIZE   = 64;
    static const size_t DATA_CACHE_LINE_SIZE          = 64;
    static const int    NUM_PERFORMANCE_COUNTER       = 6;
#elif defined NN_BUILD_CONFIG_CPU_CORTEX_A57_AARCH64
    static const size_t INSTRUCTION_CACHE_LINE_SIZE   = 64;
    static const size_t DATA_CACHE_LINE_SIZE          = 64;
    static const int    NUM_PERFORMANCE_COUNTER       = 6;
#else
    #error NN_BUILD_CONFIG_CPU_ is not defined
#endif


#if defined(NN_BUILD_CONFIG_HARDWARE_SMMA53)
    static const int32_t    NUM_CORE = 2;
#elif defined(NN_BUILD_CONFIG_HARDWARE_JUNO)
#if defined NN_BUILD_CONFIG_CPU_CORTEX_A57_AARCH64
    static const int32_t    NUM_CORE = 2;
#elif defined NN_BUILD_CONFIG_CPU_CORTEX_A53_AARCH64
    static const int32_t    NUM_CORE = 4;
#endif
#elif defined(NN_BUILD_CONFIG_HARDWARE_MTBVP)
    static const int32_t    NUM_CORE = 4;
#elif defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK2) || defined(NN_BUILD_CONFIG_HARDWARE_NX)
    static const int32_t    NUM_CORE = 4;
#elif defined(NN_BUILD_CONFIG_HARDWARE_JETSONTX2)
    static const int32_t    NUM_CORE = 4;
#else
    #error NN_BUILD_CONFIG_HARDWARE_ is not defined
#endif

public:
    static Result InvalidateDataCache(void* addr, size_t size);
    static Result StoreDataCache(const void* addr, size_t size);
    static Result FlushDataCache(const void* addr, size_t size);
    static Result InvalidateInstructionCache(void* addr, size_t size);

    static void InvalidateEntireDataCache();
    static void StoreEntireDataCache();
    static void FlushEntireDataCache();
    static void InvalidateEntireInstructionCache();

    static void FlushEntireDataCacheLocal();
    static void InvalidateEntireInstructionCacheLocal();

    static void SwithcToHyp();

    static void SynchronizeAllCore();
    static void Initialize0(int32_t coreNo);
    static void Initialize1(int32_t coreNo);

    static void SleepCore()
    {
        asm volatile("wfi");
    }
    static int32_t  GetCurrentCoreNo()
    {
        Bit64 cpuId;
        HW_GET_MPIDR_EL1(cpuId);
        return (cpuId & HW_MPIDR_EL1_AFF0_MASK);
    }
    static void OnThreadCoreChanged(KThread* pThread)
    {
        KContext::OnThreadCoreChanged(pThread);
    }

    static void SwitchThreadLocalRegion(uintptr_t tlr)
    {
        HW_SET_THREAD_ID_USER_READ_ONLY(tlr);
    }

    //-----------------------------------------------------
    // CPU 固有

    static void WaitDataCacheOperation()
    {
        asm volatile("dsb sy");
    }
    static void CareInstructionConsistency()
    {
        DataSynchronizationBarrier();
        InstructionMemoryBarrier();
    }
    static Bit64 GetCurrentTcr()
    {
        Bit64 tcr;
        HW_GET_TCR_EL1(tcr);
        return tcr;
    }
    static void SwitchProcess(Bit64 tcr, Bit64 ttbrValue, Bit64 id)
    {
        // ttbrValue が 0の場合は、TTBR0 の変換を禁止する。
        HW_SET_TCR_EL1(tcr);
        HW_SET_TTBR0_EL1(ttbrValue);
        HW_SET_CONTEXTIDR_EL1(id);
        InstructionMemoryBarrier();
    }
    static Bit64 GetSctlr()
    {
        Bit64 sctlr;
        HW_GET_SCTLR_EL1(sctlr);
        return sctlr;
    }
    static void SetSctlr(Bit64 sctlr)
    {
        HW_SET_SCTLR_EL1(sctlr);
    }
    static uint16_t GetCurrentAsid()
    {
        Bit64 ttbr0;
        HW_GET_TTBR0_EL1(ttbr0);
        return static_cast<uint16_t>((ttbr0 >> 48) & 0xFFFF);
    }
    static void InvalidateEntireTlb()
    {
        asm volatile("TLBI VMALLE1IS":::"memory");
        CareInstructionConsistency();
    }
    static void InvalidateEntireTlbNoCodeModification()
    {
        asm volatile("TLBI VMALLE1IS":::"memory");
        DataSynchronizationBarrier();
        // コード変更がないのを前提に命令メモリバリアを使用しない
    }
    static void InvalidateTlbByMvaNoCodeModification(KProcessAddress va)
    {
        // 全ASIDに対してTLBを無効化している。
        // カーネルテーブルのみなら最適化できる
        asm volatile("TLBI VAAE1IS, %0"::"r"((GetAsInteger(va) >> 12) & 0xFFFFFFFFFFFul):"memory");
        DataSynchronizationBarrier();
        // コード変更がないのを前提に命令メモリバリアを使用しない
    }
    static void InvalidateTlbByAsid(Bit32 asid)
    {
        asm volatile("TLBI ASIDE1IS, %0"::"r"(static_cast<Bit64>(asid)<<48):"memory");
        CareInstructionConsistency();
    }
    static void InvalidateTlbByVaAsid(Bit32 asid, KProcessAddress va)
    {
        asm volatile("TLBI VALE1IS, %0"::"r"((static_cast<Bit64>(asid)<<48) | ((GetAsInteger(va) >> 12) & 0xFFFFFFFFFFFul)):"memory");
        CareInstructionConsistency();
    }
    static Bit32 GetFaultAddress()
    {
        Bit64 value;
        HW_GET_FAR_EL1(value);
        return value;
    }
    static Bit32 GetExceptionSyndrome()
    {
        Bit64 value;
        HW_GET_ESR_EL1(value);
        return value;
    }
    static bool GetPhysicalAddressOfCurrentSpace(KPhysicalAddress* pPhysicalAddress, KVirtualAddress va, bool privilege = false)
    {
        Bit64 pa;
        Bit64 frac = (GetAsInteger(va) &  0x00000FFFull);

        if (privilege)
        {
            // Address Translate Stage 1 EL1 Write
            asm volatile("AT S1E1W, %0"::"r"(GetAsInteger(va)):"memory");
        }
        else
        {
            // Address Translate Stage 1 EL0 Write
            asm volatile("AT S1E0W, %0"::"r"(GetAsInteger(va)):"memory");
        }
        InstructionMemoryBarrier();
        HW_GET_PAR_EL1(pa);

        if ((pa & 0x1) == 0)
        {
            *pPhysicalAddress = KPhysicalAddress((pa & 0x0000FFFFFFFFF000ull) | frac);
            return true;
        }
        else
        {
            return false;
        }
    }

    static bool CanAtomicAccess(KProcessAddress va, bool privilege = false)
    {
        Bit64 pa;
        if (privilege)
        {
            // Address Translate Stage 1 EL1 Write
            asm volatile("AT S1E1W, %0"::"r"(GetAsInteger(va)):"memory");
        }
        else
        {
            // Address Translate Stage 1 EL0 Write
            asm volatile("AT S1E0W, %0"::"r"(GetAsInteger(va)):"memory");
        }
        InstructionMemoryBarrier();
        HW_GET_PAR_EL1(pa);

        if ((pa & 0x1) == 0)
        {
            return ((pa >> 56) == HW_MAIR_ATTR_NORMAL_ICRW_OCRW);
        }
        else
        {
            return false;
        }
    }

    static void DataSynchronizationBarrier()
    {
        asm volatile("dsb sy");
    }

    static void DataMemoryBarrier()
    {
        asm volatile("dmb sy");
    }

    static void InstructionMemoryBarrier()
    {
        asm volatile("isb");
    }

    static uint64_t GetCycleCounter()
    {
        uint64_t x;
        HW_GET_PMCCNTR_EL0(x);
        return x;
    }

    static uint32_t GetPerformanceCounter(int n)
    {
        uint64_t x = 0;
        if (n < NUM_PERFORMANCE_COUNTER)
        {
            switch (n)
            {
            case 5:
                {
                    HW_GET_PMEVCNTR_EL0(5, x);
                }
                break;
            case 4:
                {
                    HW_GET_PMEVCNTR_EL0(4, x);
                }
                break;
            case 3:
                {
                    HW_GET_PMEVCNTR_EL0(3, x);
                }
                break;
            case 2:
                {
                    HW_GET_PMEVCNTR_EL0(2, x);
                }
                break;
            case 1:
                {
                    HW_GET_PMEVCNTR_EL0(1, x);
                }
                break;
            case 0:
                {
                    HW_GET_PMEVCNTR_EL0(0, x);
                }
            default:
                break;
            }
        }
        return static_cast<uint32_t>(x);
    }

    static void EnablePerformanceCounter()
    {
        HW_SET_PMCR_EL0(0ul);
        DataSynchronizationBarrier();
        InstructionMemoryBarrier();
        switch (NUM_PERFORMANCE_COUNTER)
        {
        default:
        case 6:
            {
                HW_SET_PMEVTYPER_EL0(5, HW_PMXEVTYPER(1, 0, HW_PM_TYPE_L2C_REFILL));
                HW_SET_PMCNTENSET_EL0(1ul << 5);
            } NN_FALL_THROUGH;
        case 5:
            {
                HW_SET_PMEVTYPER_EL0(4, HW_PMXEVTYPER(1, 1, HW_PM_TYPE_L2C_REFILL));
                HW_SET_PMCNTENSET_EL0(1ul << 4);
            } NN_FALL_THROUGH;
        case 4:
            {
                HW_SET_PMEVTYPER_EL0(3, HW_PMXEVTYPER(1, 0, HW_PM_TYPE_INST_RETIRED));
                HW_SET_PMCNTENSET_EL0(1ul << 3);
            } NN_FALL_THROUGH;
        case 3:
            {
                HW_SET_PMEVTYPER_EL0(2, HW_PMXEVTYPER(1, 1, HW_PM_TYPE_INST_RETIRED));
                HW_SET_PMCNTENSET_EL0(1ul << 2);
            } NN_FALL_THROUGH;
        case 2:
            {
                HW_SET_PMEVTYPER_EL0(1, HW_PMXEVTYPER(1, 0, HW_PM_TYPE_CPU_CYCLES));
                HW_SET_PMCNTENSET_EL0(1ul << 1);
            } NN_FALL_THROUGH;
        case 1:
            {
                HW_SET_PMEVTYPER_EL0(0, HW_PMXEVTYPER(1, 1, HW_PM_TYPE_CPU_CYCLES));
                HW_SET_PMCNTENSET_EL0(1ul << 0);
            } NN_FALL_THROUGH;
        case 0:
            {
                HW_SET_PMCCFILTR_EL0(HW_PMCCFILTR(1, 1));
                HW_SET_PMCNTENSET_EL0(1ul << 31);
            }
            break;
        }
        DataSynchronizationBarrier();
        InstructionMemoryBarrier();
        HW_SET_PMCR_EL0((1ul << 2) | (1ul << 1) | (1ul << 0));
    }
    static void GetCycleCounter(Bit64* pOut);
    static void GetPerformanceCounter(Bit32* pOut, int n);
public:
    typedef InterlockedVariable<KThread*> InterlockedThreadPtr;

    static uintptr_t GetCurrentValue()
    {
        register Bit64 x18 asm ("x18");
        asm volatile ("":"=r"(x18));
        return x18;
    }
    static void SetCurrentValue(uintptr_t value)
    {
        register Bit64 x18 asm ("x18") = value;
        asm volatile (""::"r"(x18));
        HW_SET_TPIDR_EL1(value);
    }
};

}
}}

