﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/TargetConfigs/build_Cpu.h>
#include <nn/nn_Common.h>
#include "../../kern_CPUSelect.h"
#include "../../kern_KHardwareTimerBase.h"

#if defined NN_BUILD_CONFIG_CPU_CORTEX_A9
#error not supported on Cortex-A9
#endif

namespace nn { namespace kern {

    namespace ARMv7A {


class KHardwareTimer
    : public KHardwareTimerBase
{
public:
    void    Initialize(int32_t coreNo);
    void    Finalize(int32_t coreNo);

    static int64_t GetTick()
    {
        return GetTimerCount();
    }

    virtual void  DoInterruptTask();

    static uint32_t GetRawTick()
    {
        return GetTimerLowCount();
    }

    void RegisterTaskAbsolute(KTimerTask* pTask, int64_t tick)
    {
        KScopedDisableDispatch dd;
        KScopedSimpleLock lock(&GetLockMutex());
        if (RegisterTaskAbsoluteImpl(pTask, tick))
        {
            SetTimerCount(tick);
            EnableHardwareTimerInterrupt();
        }
    }

private:
    enum {
#if defined NN_GUEST_OS || defined NN_BUILD_CONFIG_HARDWARE_PANDA5
        INTERRUPT_ID_TIMER = 30, // Non-Secure Timer
#else
        INTERRUPT_ID_TIMER = 29,
#endif
    };

    static void InitGlobalTimer(int32_t coreNo)
    {
        NN_UNUSED(coreNo);

        // コアごとに初期化が必要
        HW_SET_CP15_PL1_CNTKCTL(HW_CNTKCTL_PL0PCTEN);
        HW_SET_CP15_PL1_PHYSICAL_TIMER_COMPARE_VALUE(0xffffffff,0xffffffff);
        HW_SET_CP15_PL1_PHYSICAL_TIMER_CONTROL(0x00000003);
    }

    static void SetTimerCount(int64_t timerValue)
    {
        HW_SET_CP15_PL1_PHYSICAL_TIMER_COMPARE_VALUE(
            static_cast<Bit32>(timerValue),
            static_cast<Bit32>(timerValue >> 32) );
    }

    static int64_t GetTimerCount()
    {
        Bit32 hi;
        Bit32 lo;

        HW_GET_CP15_PHYSICAL_COUNT(lo, hi);

        return (static_cast<int64_t>(hi) << 32) | lo;
    }

    static uint32_t GetTimerLowCount()
    {
        Bit32 hi;
        Bit32 lo;

        HW_GET_CP15_PHYSICAL_COUNT(lo, hi);

        return lo;
    }

    static void EnableHardwareTimerInterrupt()
    {
        HW_SET_CP15_PL1_PHYSICAL_TIMER_CONTROL(0x00000001);
    }

    static void StopHardwareTimer()
    {
        HW_SET_CP15_PL1_PHYSICAL_TIMER_COMPARE_VALUE(0xffffffff,0xffffffff);
        HW_SET_CP15_PL1_PHYSICAL_TIMER_CONTROL(0x00000000);
    }

    static void DisableHardwareTimerInterrupt()
    {
        HW_SET_CP15_PL1_PHYSICAL_TIMER_CONTROL(0x00000003);
    }
};

    }
}}

