﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/TargetConfigs/build_Cpu.h>
#include <nn/nn_Common.h>
#include <nn/nn_BitTypes.h>
#include "../../kern_Platform.h"
#include "kern_GenericInterruptController.h"
#include "kern_MemoryMap.h"
#include "../../kern_Assert.h"
#include "../../kern_Current.h"

namespace nn { namespace kern { namespace ARM {
GicDistributer*  KInterruptController::s_GicDistributerBase;
GicCpuInterface* KInterruptController::s_GicCpuInterfaceBase;
Bit32 KInterruptController::s_GicMask[8];

KPhysicalAddress KInterruptController::GetRegisterAddress(int index)
{
    switch (index)
    {
    case 0: return NN_KERN_P_ADDR_GIC_DISTRIBUTER;
    case 1: return NN_KERN_P_ADDR_CPU_INTERFACE;
    default:
            NN_KERN_ABORT();
            break;
    }
    return Null<KPhysicalAddress>();
}

size_t KInterruptController::GetRegisterSize(int index)
{
    switch (index)
    {
    case 0: return NN_KERN_P_ADDR_GIC_DISTRIBUTER_SIZE;
    case 1: return NN_KERN_P_ADDR_CPU_INTERFACE_SIZE;
    default:
            NN_KERN_ABORT();
            break;
    }
    return 0;
}


void KInterruptController::PrepareInitialize(const KProcessAddress registerBase[NumMaps])
{
    s_GicDistributerBase = GetTypedPointer<GicDistributer>(registerBase[0]);
    s_GicCpuInterfaceBase = GetTypedPointer<GicCpuInterface>(registerBase[1]);
}

void KInterruptController::Initialize(int coreNo)
{
    NN_KERN_ABORT_UNLESS(s_GicDistributerBase != nullptr);
    NN_KERN_ABORT_UNLESS(s_GicCpuInterfaceBase != nullptr);
    m_GicDistributerBase = s_GicDistributerBase;
    m_GicCpuInterfaceBase = s_GicCpuInterfaceBase;

    m_GicCpuInterfaceBase->ctlr = 0;
    KCPU::SynchronizeAllCore();

    if (coreNo == 0)
    {
        m_GicDistributerBase->ctlr = 0;
    }

    KCPU::SynchronizeAllCore();

    m_GicCpuInterfaceBase->pmr = 0;
    m_GicCpuInterfaceBase->bpr = 7; // No preemption

    const int numIntLines = ((coreNo == 0)? ((m_GicDistributerBase->typer & 0x1f) * 32 + 32): NumLocalInterrupts);
    for (int i = 0; i < numIntLines / 32; ++i)
    {
        m_GicDistributerBase->icenabler[i] = 0xffffffffu;
        m_GicDistributerBase->icpendr[i]   = 0xffffffffu;
        m_GicDistributerBase->icactiver[i] = 0xffffffffu;
        m_GicDistributerBase->igroupr[i]   = 0x00000000u;
    }
    for (int i = 0; i < numIntLines; ++i)
    {
        m_GicDistributerBase->ipriorityr.bytes[i] = 0xFF;
        m_GicDistributerBase->itargetsr.bytes[i] = 0;
    }
    for (int i = 0; i < numIntLines / 16; ++i)
    {
        m_GicDistributerBase->icfgr[i] = 0x00000000u;
    }
    KCPU::SynchronizeAllCore();

    if (coreNo == 0)
    {
        m_GicDistributerBase->ctlr = 1;
    }
    KCPU::SynchronizeAllCore();
    m_GicCpuInterfaceBase->ctlr = 1;

    s_GicMask[coreNo] = GetOwnMask();
    SetPriorityLevel(PriorityLevel_Low);
}

void KInterruptController::DisableCoreLocalAll(KInterruptController::LocalState* pBuffer) const
{
    for (size_t i = 0; i < sizeof(pBuffer->isenabler) / sizeof(pBuffer->isenabler[0]); ++i)
    {
        pBuffer->isenabler[i] = m_GicDistributerBase->isenabler[i];
        m_GicDistributerBase->icenabler[i] = 0xffffffff;
    }

    for (size_t i = 0; i < sizeof(pBuffer->ipriorityr) / sizeof(pBuffer->ipriorityr[0]); ++i)
    {
        pBuffer->ipriorityr[i] = m_GicDistributerBase->ipriorityr.words[i];
        m_GicDistributerBase->ipriorityr.words[i] = 0xffffffff;
    }

    for (size_t i = 0; i < sizeof(pBuffer->itargetsr) / sizeof(pBuffer->itargetsr[0]); ++i)
    {
        pBuffer->itargetsr[i] = m_GicDistributerBase->itargetsr.words[i];
    }

    for (size_t i = 0; i < sizeof(pBuffer->icfgr) / sizeof(pBuffer->icfgr[0]); ++i)
    {
        pBuffer->icfgr[i] = m_GicDistributerBase->icfgr[i];
    }
}

void KInterruptController::DisableGlobalAll(KInterruptController::GlobalState* pBuffer) const
{
    for (size_t i = 0; i < sizeof(pBuffer->isenabler) / sizeof(pBuffer->isenabler[0]); ++i)
    {
        size_t offset = sizeof(KInterruptController::LocalState::isenabler) / sizeof(KInterruptController::LocalState::isenabler[0]);
        pBuffer->isenabler[i] = m_GicDistributerBase->isenabler[i + offset];
        m_GicDistributerBase->icenabler[i + offset] = 0xffffffff;
    }

    for (size_t i = 0; i < sizeof(pBuffer->ipriorityr) / sizeof(pBuffer->ipriorityr[0]); ++i)
    {
        size_t offset = sizeof(KInterruptController::LocalState::ipriorityr) / sizeof(KInterruptController::LocalState::ipriorityr[0]);
        pBuffer->ipriorityr[i] = m_GicDistributerBase->ipriorityr.words[i + offset];
        m_GicDistributerBase->ipriorityr.words[i + offset] = 0xffffffff;
    }

    for (size_t i = 0; i < sizeof(pBuffer->itargetsr) / sizeof(pBuffer->itargetsr[0]); ++i)
    {
        size_t offset = sizeof(KInterruptController::LocalState::itargetsr) / sizeof(KInterruptController::LocalState::itargetsr[0]);
        pBuffer->itargetsr[i] = m_GicDistributerBase->itargetsr.words[i + offset];
    }

    for (size_t i = 0; i < sizeof(pBuffer->icfgr) / sizeof(pBuffer->icfgr[0]); ++i)
    {
        size_t offset = sizeof(KInterruptController::LocalState::icfgr) / sizeof(KInterruptController::LocalState::icfgr[0]);
        pBuffer->icfgr[i] = m_GicDistributerBase->icfgr[i + offset];
    }
}

void KInterruptController::RestoreCoreLocal(const KInterruptController::LocalState& state) const
{
    for (size_t i = 0; i < sizeof(state.ipriorityr) / sizeof(state.ipriorityr[0]); ++i)
    {
        m_GicDistributerBase->ipriorityr.words[i] = state.ipriorityr[i];
    }

    for (size_t i = 0; i < sizeof(state.itargetsr) / sizeof(state.itargetsr[0]); ++i)
    {
        m_GicDistributerBase->itargetsr.words[i] = state.itargetsr[i];
    }

    for (size_t i = 0; i < sizeof(state.icfgr) / sizeof(state.icfgr[0]); ++i)
    {
        m_GicDistributerBase->icfgr[i] = state.icfgr[i];
    }

    for (size_t i = 0; i < sizeof(state.isenabler) / sizeof(state.isenabler[0]); ++i)
    {
        m_GicDistributerBase->icenabler[i] = 0xffffffff;
        m_GicDistributerBase->isenabler[i] = state.isenabler[i];
    }
}

void KInterruptController::RestoreGlobal(const KInterruptController::GlobalState& state) const
{
    for (size_t i = 0; i < sizeof(state.ipriorityr) / sizeof(state.ipriorityr[0]); ++i)
    {
        size_t offset = sizeof(KInterruptController::LocalState::ipriorityr) / sizeof(KInterruptController::LocalState::ipriorityr[0]);
        m_GicDistributerBase->ipriorityr.words[i + offset] = state.ipriorityr[i];
    }

    for (size_t i = 0; i < sizeof(state.itargetsr) / sizeof(state.itargetsr[0]); ++i)
    {
        size_t offset = sizeof(KInterruptController::LocalState::itargetsr) / sizeof(KInterruptController::LocalState::itargetsr[0]);
        m_GicDistributerBase->itargetsr.words[i + offset] = state.itargetsr[i];
    }

    for (size_t i = 0; i < sizeof(state.icfgr) / sizeof(state.icfgr[0]); ++i)
    {
        size_t offset = sizeof(KInterruptController::LocalState::icfgr) / sizeof(KInterruptController::LocalState::icfgr[0]);
        m_GicDistributerBase->icfgr[i + offset] = state.icfgr[i];
    }

    for (size_t i = 0; i < sizeof(state.isenabler) / sizeof(state.isenabler[0]); ++i)
    {
        size_t offset = sizeof(KInterruptController::LocalState::isenabler) / sizeof(KInterruptController::LocalState::isenabler[0]);
        m_GicDistributerBase->icenabler[i + offset] = 0xffffffff;
        m_GicDistributerBase->isenabler[i + offset] = state.isenabler[i];
    }
}

void KInterruptController::Finalize(int coreNo)
{
    // 分散割り込みコントローラ無効
    if (coreNo == 0)
    {
        m_GicDistributerBase->ctlr = 0;
    }
    m_GicCpuInterfaceBase->ctlr = 0;

    SetPriorityLevel(PriorityLevel_High);

    const int numIntLines = ((coreNo == 0)? ((m_GicDistributerBase->typer & 0x1f) * 32 + 32): NumLocalInterrupts);
    for (int i = 0; i < numIntLines / 32; ++i)
    {
        m_GicDistributerBase->icenabler[i] = 0xffffffffu;
        m_GicDistributerBase->icpendr[i]   = 0xffffffffu;
        m_GicDistributerBase->icactiver[i] = 0xffffffffu;
        m_GicDistributerBase->igroupr[i]   = 0x00000000u;
    }
    for (int i = 0; i < numIntLines; ++i)
    {
        m_GicDistributerBase->ipriorityr.bytes[i] = 0xFF;
        m_GicDistributerBase->itargetsr.bytes[i] = 0;
    }
    for (int i = 0; i < numIntLines / 16; ++i)
    {
        m_GicDistributerBase->icfgr[i] = 0x00000000u;
    }
    m_GicDistributerBase = nullptr;
    m_GicCpuInterfaceBase = nullptr;
}

}}}
