﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/TargetConfigs/build_Base.h>
#include <nn/nn_Common.h>
#include <nn/svc/svc_Kernel.h>
#include "../../kern_Platform.h"
#include "kern_SystemControlBase.h"
#include "../../kern_Result.h"
#include "../../kern_Kernel.h"
#include "../../kern_CPUSelect.h"
#include "../../kern_Main.h"
#include "../../kern_KProcess.h"
#include "kern_MemoryMap.h"
#include "../../kern_PageTableSelect.h"
#include "../../kern_InterruptNameSelect.h"
#include "../../kern_InterruptManagerSelect.h"
#include "../../kern_KLightMutex.h"
#include "../../kern_InitialProcess.h"
#include "../../kern_KThread.h"
#include "../../kern_KTrace.h"
#include "../../kern_KSimpleLock.h"
#include "../../kern_Trace.h"
#include "../../kern_Utility.h"
#include "../../kern_KMemoryLayout.h"
#include "../../kern_KHeapArrange.h"

#include <cstring>
#include <random>

namespace nn { namespace kern {
namespace {
std::mt19937 g_Random;
KSimpleLock  g_RandomLock;
bool g_RandInitialized = false;

uint64_t GetPhysicalMemorySize()
{
    return NN_KERN_P_ADDR_MAIN_MEMORY_SIZE;
}

size_t GetHeapSize()
{
    return KMemoryLayout::GetHeapRegionSize();
}
}

bool KTargetSystemBase::m_IsDevelopmentHardware         = false;
bool KTargetSystemBase::m_IsPrintEnabled                = false;
bool KTargetSystemBase::m_IsUserExceptionHandlerEnabled = false;
bool KTargetSystemBase::m_IsNonZeroFillMemoryEnabled    = false;
bool KTargetSystemBase::m_IsUserPmuAccessEnabled        = false;
bool KTargetSystemBase::m_IsInternalDebugEnabled        = false;


uint64_t KSystemControlBase::GetRandomValue(uint64_t randMin, uint64_t randMax)
{
    std::uniform_int_distribution<uint64_t> rand(randMin, randMax);
    KDisableInterrupt di;
    KScopedSimpleLock lock(&g_RandomLock);
    if (NN_UNLIKELY(!g_RandInitialized))
    {
        g_Random.seed(static_cast<uint64_t>(KHardwareTimer::GetTick()));
        g_RandInitialized = true;
    }
    return rand(g_Random);
}

void KSystemControlBase::GenerateRandom(Bit64* pBuffer, size_t num)
{
    for (size_t i = 0; i < num; i++)
    {
        pBuffer[i] = GetRandomValue(0, 0xfffffffffffffffful);
    }
}

void KSystemControlBase::GetHeapArrange(KHeapArrange* p)
{
    size_t managementSize = KMemoryManager::CalcManagementAreaSize(GetHeapSize());
    managementSize += KMemoryManager::CalcManagementAreaSize(NN_KERN_FINEST_PAGE_SIZE);
    managementSize += KMemoryManager::CalcManagementAreaSize(NN_KERN_FINEST_PAGE_SIZE);
    managementSize += KMemoryManager::CalcManagementAreaSize(NN_KERN_FINEST_PAGE_SIZE);
    {
        p->managementAddr = KMemoryLayout::GetHeapRegionBegin();
        p->managementSize = managementSize;
        p->secureSystemAddr = p->managementAddr + p->managementSize;
        p->secureSystemSize = GetHeapSize() - managementSize;
        p->secureSystemBackAddr = p->secureSystemAddr + p->secureSystemSize;
        p->nonSecureSystemAddr = 0;
        p->nonSecureSystemSize = 0;
        p->nonSecureSystemBackAddr = 0;
        p->applicationAddr = 0;
        p->applicationSize = 0;
        p->applicationBackAddr = 0;
        p->appletAddr = 0;
        p->appletSize = 0;
        p->appletBackAddr = 0;
    }
}

KMemoryManager::Region KSystemControlBase::GetCreateProcessMemoryRegion()
{
    return KMemoryManager::Region_SecureSystem;
}

void KSystemControlBase::SystemInitialize0()
{
    KTargetSystemBase::SetDevelopmentHardware(true);
    KTargetSystemBase::EnablePrint(true);
    KTargetSystemBase::EnableUserExceptionHandler(true);
    KTargetSystemBase::EnableNonZeroFillMemory(true);
    KTargetSystemBase::EnableUserPmuAccess(true);
    KTargetSystemBase::EnableInternalDebug(true);

    KResourceLimit& resourceLimit = Kernel::GetSystemResourceLimit();
    KAutoObject::Create(&resourceLimit);
    resourceLimit.Initialize();
    NN_KERN_ABORT_UNLESS(resourceLimit.SetLimitValue(nn::svc::LimitableResource_PhysicalMemoryMax,      GetPhysicalMemorySize()).IsSuccess());
    NN_KERN_ABORT_UNLESS(resourceLimit.SetLimitValue(nn::svc::LimitableResource_ThreadCountMax,         NN_KERN_SLAB_OBJ_NUM_THREAD).IsSuccess());
    NN_KERN_ABORT_UNLESS(resourceLimit.SetLimitValue(nn::svc::LimitableResource_EventCountMax,          NN_KERN_SLAB_OBJ_NUM_EVENT).IsSuccess());
    NN_KERN_ABORT_UNLESS(resourceLimit.SetLimitValue(nn::svc::LimitableResource_TransferMemoryCountMax, NN_KERN_SLAB_OBJ_NUM_TRANSFER_MEMORY).IsSuccess());
    NN_KERN_ABORT_UNLESS(resourceLimit.SetLimitValue(nn::svc::LimitableResource_SessionCountMax,        NN_KERN_SLAB_OBJ_NUM_SESSION).IsSuccess());

    NN_KERN_ABORT_UNLESS(resourceLimit.TestLimit(nn::svc::LimitableResource_PhysicalMemoryMax, KMemoryLayout::GetKernelRegionPhysicalSize()));
    if (NN_KERN_P_ADDR_RESERVED_LO_SIZE)
    {
        NN_KERN_ABORT_UNLESS(resourceLimit.TestLimit(nn::svc::LimitableResource_PhysicalMemoryMax, NN_KERN_P_ADDR_RESERVED_LO_SIZE));
    }
    if (NN_KERN_P_ADDR_RESERVED_HI_SIZE)
    {
        NN_KERN_ABORT_UNLESS(resourceLimit.TestLimit(nn::svc::LimitableResource_PhysicalMemoryMax, NN_KERN_P_ADDR_RESERVED_HI_SIZE));
    }
}

void KSystemControlBase::SystemInitialize1()
{
#if defined NN_KERN_ENABLE_KERNEL_TRACE
    InitializeTrace();
#endif
#if defined NN_KERN_ENABLE_KTRACE
    KTrace::Initialize(KMemoryLayout::ToLinearVirtualAddress(KMemoryLayout::GetDebugRegionPhysicalBegin()), NN_KERN_KTRACE_BUFFER_SIZE);
#endif
}

void KSystemControlBase::CallSecureMonitor64(Bit64* pBuf)
{
    NN_UNUSED(pBuf);
}
void KSystemControlBase::CallSecureMonitor32(Bit32* pBuf)
{
    NN_UNUSED(pBuf);
}

void KSystemControlBase::SleepSystem()
{
}

Result KSystemControlBase::ReadWriteRegister(Bit32* pOut, nn::svc::PhysicalAddress address, Bit32 mask, Bit32 value)
{
    NN_UNUSED(pOut);
    NN_UNUSED(address);
    NN_UNUSED(mask);
    NN_UNUSED(value);
    return nn::svc::ResultNotImplemented();
}

void KSystemControlBase::StopSystem()
{
    for (;;) {}
}

bool KSystemControlBase::IsIoMappablePhysicalAddress(KPhysicalAddress physAddr, size_t size)
{
    NN_UNUSED(physAddr);
    NN_UNUSED(size);
    return true;
}

bool KSystemControlBase::IsStaticMappablePhysicalAddress(KPhysicalAddress physAddr, size_t size)
{
    NN_UNUSED(physAddr);
    NN_UNUSED(size);
    return true;
}

size_t KSystemControlBase::ComputeMemoryRequirementForSecureMemory(size_t size, KMemoryManager::Region region)
{
    NN_UNUSED(region);

    return size;
}

Result KSystemControlBase::AllocateSecureMemory(KVirtualAddress* pAddr, size_t size, KMemoryManager::Region region)
{
    size_t align = KMemoryManager::PageSize;

    if ((size & (align - 1)) != 0)
    {
        return nn::svc::ResultInvalidSize();
    }

    Bit32 allocateOption = KMemoryManager::MakeAllocateOption(region, KMemoryManager::From_Front);
    KVirtualAddress va = Kernel::GetKernelHeapManager().AllocateContinuous(size / KMemoryManager::PageSize, align / KMemoryManager::PageSize, allocateOption);
    if (va == Null<KVirtualAddress>())
    {
        return nn::svc::ResultOutOfMemory();
    }

    Kernel::GetKernelHeapManager().Open(va, size / KMemoryManager::PageSize);
    *pAddr = va;

    return ResultSuccess();
}

void KSystemControlBase::FreeSecureMemory(KVirtualAddress addr, size_t size, KMemoryManager::Region region)
{
    NN_UNUSED(region);
    size_t align = KMemoryManager::PageSize;

    NN_KERN_ABORT_UNLESS((GetAsInteger(addr) & (align - 1)) == 0);
    NN_KERN_ABORT_UNLESS(((size) & (align - 1)) == 0);

    Kernel::GetKernelHeapManager().Close(addr, size / KMemoryManager::PageSize);
}

}}

