﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/TargetConfigs/build_Base.h>
#include <nn/nn_Common.h>
#include <nn/svc/svc_Kernel.h>
#include <cstring>
#include "../../kern_Platform.h"
#include "../../kern_KSimpleLock.h"
#include "../../kern_InterruptManagerSelect.h"
#include "kern_SecureMonitorCall.h"

namespace nn { namespace kern { namespace NX {
namespace smc {
namespace {
enum SmcFunction
{
    SmcFucntion_CpuSuspend              = 0xC4000001,
    SmcFucntion_CpuOff                  = 0xC4000002,
    SmcFucntion_CpuOn                   = 0xC4000003,
    SmcFucntion_GetConfig               = 0xC3000004,
    SmcFucntion_GenerateRandom          = 0xC3000005,
    SmcFucntion_ShowError               = 0xC3000006,
    SmcFucntion_SetKernelCarveoutRegion = 0xC3000007,
    SmcFucntion_ReadWriteRegister       = 0xC3000008,
};

void SecureMonitorCallFromKernel(Bit64* pBuf)
{
    register Bit64 x0 asm("x0") = pBuf[0];
    register Bit64 x1 asm("x1") = pBuf[1];
    register Bit64 x2 asm("x2") = pBuf[2];
    register Bit64 x3 asm("x3") = pBuf[3];
    register Bit64 x4 asm("x4") = pBuf[4];
    register Bit64 x5 asm("x5") = pBuf[5];
    register Bit64 x6 asm("x6") = pBuf[6];
    register Bit64 x7 asm("x7") = pBuf[7];

    {
        KDisableInterrupt di;
        asm volatile ("smc #1"
                : "+r"(x0), "+r"(x1), "+r"(x2), "+r"(x3), "+r"(x4), "+r"(x5), "+r"(x6), "+r"(x7)
                :
                : "x8", "x9", "x10", "x11", "x12",
                "x13", "x14", "x15", "x16", "x17", "x18", "cc", "memory");
        Bit64 tpidr;
        HW_GET_TPIDR_EL1(tpidr);
        register Bit64 x18 asm ("x18") = tpidr;
        asm volatile (""::"r"(x18));
    }

    pBuf[0] = x0;
    pBuf[1] = x1;
    pBuf[2] = x2;
    pBuf[3] = x3;
    pBuf[4] = x4;
    pBuf[5] = x5;
    pBuf[6] = x6;
    pBuf[7] = x7;
}

void SecureMonitorCallFromUser(Bit64* pBuf)
{
    register Bit64 x0 asm("x0") = pBuf[0];
    register Bit64 x1 asm("x1") = pBuf[1];
    register Bit64 x2 asm("x2") = pBuf[2];
    register Bit64 x3 asm("x3") = pBuf[3];
    register Bit64 x4 asm("x4") = pBuf[4];
    register Bit64 x5 asm("x5") = pBuf[5];
    register Bit64 x6 asm("x6") = pBuf[6];
    register Bit64 x7 asm("x7") = pBuf[7];

    {
        KDisableInterrupt di;
        asm volatile ("smc #0"
                : "+r"(x0), "+r"(x1), "+r"(x2), "+r"(x3), "+r"(x4), "+r"(x5), "+r"(x6), "+r"(x7)
                :
                : "x8", "x9", "x10", "x11", "x12",
                "x13", "x14", "x15", "x16", "x17", "x18", "cc", "memory");
        Bit64 tpidr;
        HW_GET_TPIDR_EL1(tpidr);
        register Bit64 x18 asm ("x18") = tpidr;
        asm volatile (""::"r"(x18));
    }

    pBuf[0] = x0;
    pBuf[1] = x1;
    pBuf[2] = x2;
    pBuf[3] = x3;
    pBuf[4] = x4;
    pBuf[5] = x5;
    pBuf[6] = x6;
    pBuf[7] = x7;
}
}

void CpuOn(Bit64 cpuId, Bit64 entryPoint, Bit64 contextId)
{
    Bit64 buffer[8] = { SmcFucntion_CpuOn, cpuId, entryPoint, contextId};
    SecureMonitorCallFromKernel(buffer);
    NN_KERN_ABORT_UNLESS(buffer[0] == 0);
}

void SecureMonitorCall(Bit64* pBuf)
{
    SecureMonitorCallFromUser(pBuf);
}

void GenerateRandomBytes(void* pBuffer, size_t size)
{
    static KSimpleLock g_SmcLock;
    NN_KERN_ABORT_UNLESS(0 < size && size <= 56);

    Bit64 buffer[8] = { SmcFucntion_GenerateRandom, size};
    {
        KDisableInterrupt di;
        KScopedSimpleLock lock(&g_SmcLock);
        SecureMonitorCallFromKernel(buffer);
    }
    NN_KERN_ABORT_UNLESS(buffer[0] == 0);

    std::memcpy(pBuffer, buffer + 1, size);
}

void GetConfig(ConfigItemData* pData, ConfigItem item)
{
    Bit64 buffer[8] = { SmcFucntion_GetConfig, item};
    SecureMonitorCallFromKernel(buffer);
    NN_KERN_ABORT_UNLESS(buffer[0] == 0);
    pData->data[0] = buffer[1];
    pData->data[1] = buffer[2];
    pData->data[2] = buffer[3];
    pData->data[3] = buffer[4];
}

void SetKernelCarveoutRegion(int index, uintptr_t address, size_t size)
{
    Bit64 buffer[8] = { SmcFucntion_SetKernelCarveoutRegion, static_cast<Bit64>(index), static_cast<Bit64>(address), static_cast<Bit64>(size) };
    SecureMonitorCallFromKernel(buffer);
    NN_KERN_ABORT_UNLESS(buffer[0] == 0);
}

void ShowError(Bit32 color)
{
    Bit64 buffer[8] = { SmcFucntion_ShowError, color};
    SecureMonitorCallFromKernel(buffer);
    NN_KERN_ABORT_UNLESS(buffer[0] == 0);
}

bool ReadWriteRegister(Bit32 *pOut, uint64_t address, Bit32 mask, Bit32 value)
{
    Bit64 buffer[8] = { SmcFucntion_ReadWriteRegister, address, mask, value };
    SecureMonitorCallFromKernel(buffer);
    *pOut = buffer[1];
    return (buffer[0] == 0);
}

}
}}}
