﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   スレッドに関する C 言語用 API の宣言
 */

#pragma once

#include <nn/os/os_Config.h>
#include <nnc/nn_Result.h>

#include <nnc/nn_Common.h>
#include <nnc/os/os_Macro.h>
#include <nnc/os/os_ThreadCommon.h>
#include <nnc/os/os_ThreadTypes.h>
#include <nnc/os/os_MultipleWaitTypes.h>
#include <nnc/os/detail/os_MacroImpl.h>

//--------------------------------------------------------------------------
//  C 向けの宣言
//--------------------------------------------------------------------------

//! @name スレッド関連 C 言語用 API
//! @{

//--------------------------------------------------------------------------
/**
 * @brief   スレッドを生成します。
 *
 * @details
 *  詳細は nn::os::CreateThread(ThreadType* thread, ThreadFunction function, void* argument, void* stack, size_t stackSize, int priority) を参照して下さい。
 *
 */
NN_OS_EXTERN_C nnResult nnosCreateThread(nnosThreadType* thread, nnosThreadFunction function, void* argument, void* stack, size_t stackSize, int priority);


//--------------------------------------------------------------------------
/**
 * @brief   スレッドを生成します。
 *
 * @details
 *  詳細は nn::os::CreateThread(ThreadType* thread, ThreadFunction function, void* argument, void* stack, size_t stackSize, int priority, int coreNumber) を参照して下さい。
 *
 */
NN_OS_EXTERN_C nnResult nnosCreateThreadWithCoreNumber(nnosThreadType* thread, nnosThreadFunction function, void* argument, void* stack, size_t stackSize, int priority, int coreNumber);


//--------------------------------------------------------------------------
/**
 * @brief   スレッドを削除します。
 *
 * @details
 *  詳細は nn::os::DestroyThread() を参照して下さい。
 *
 */
NN_OS_EXTERN_C void nnosDestroyThread(nnosThreadType* thread);


//--------------------------------------------------------------------------
/**
 * @brief   スレッドの実行を開始します。
 *
 * @details
 *  詳細は nn::os::StartThread() を参照して下さい。
 *
 */
NN_OS_EXTERN_C void nnosStartThread(nnosThreadType* thread);


//--------------------------------------------------------------------------
/**
 * @brief   自スレッドの ThreadType オブジェクトを取得します。
 *
 * @details
 *  詳細は nn::os::GetCurrentThread() を参照して下さい。
 *
 */
NN_OS_EXTERN_C nnosThreadType* nnosGetCurrentThread(void);


//--------------------------------------------------------------------------
/**
 * @brief   対象スレッドが終了するのを待ちます。
 *
 * @details
 *  詳細は nn::os::WaitThread() を参照して下さい。
 *
 */
NN_OS_EXTERN_C void nnosWaitThread(nnosThreadType* thread);


//--------------------------------------------------------------------------
/**
 * @brief   同一優先度の他のスレッドに実行機会を譲ります。
 *
 * @details
 *  詳細は nn::os::YieldThread() を参照して下さい。
 *
 */
NN_OS_EXTERN_C void nnosYieldThread(void);


//--------------------------------------------------------------------------
/**
 * @brief   自スレッドを指定された時間だけ休止します。
 *
 * @details
 *  詳細は nn::os::SleepThread() を参照して下さい。@n
 *  本 API では、休止する時間をナノ秒単位で指定します。
 *
 */
NN_OS_EXTERN_C void nnosSleepThread(int64_t nanoseconds);


//--------------------------------------------------------------------------
/**
 * @brief   対象スレッドの優先度を変更します。
 *
 * @details
 *  詳細は nn::os::ChangeThreadPriority() を参照して下さい。
 *
 */
NN_OS_EXTERN_C int nnosChangeThreadPriority(nnosThreadType* thread, int priority);


//--------------------------------------------------------------------------
/**
 * @brief   対象スレッドの本来の優先度を取得します。
 *
 * @details
 *  詳細は nn::os::GetThreadPriority() を参照して下さい。
 *
 */
NN_OS_EXTERN_C int nnosGetThreadPriority(const nnosThreadType* thread);


//--------------------------------------------------------------------------
/**
 * @brief   対象スレッドの現在の優先度を取得します。
 *
 * @details
 *  詳細は nn::os::GetThreadCurrentPriority() を参照して下さい。
 *
 */
NN_OS_EXTERN_C int nnosGetThreadCurrentPriority(const nnosThreadType* thread);


//--------------------------------------------------------------------------
/**
 * @brief   対象スレッドのスレッド名を設定します。
 *
 * @details
 *  詳細は nn::os::SetThreadName() を参照して下さい。
 *
 */
NN_OS_EXTERN_C void nnosSetThreadName(nnosThreadType* thread, const char* threadName);


//--------------------------------------------------------------------------
/**
 * @brief   対象スレッドのスレッド名が格納されているアドレスを設定します。
 *
 * @details
 *  詳細は nn::os::SetThreadNamePointer() を参照して下さい。
 *
 */
NN_OS_EXTERN_C void nnosSetThreadNamePointer(nnosThreadType* thread, const char* threadName);


//--------------------------------------------------------------------------
/**
 * @brief   対象スレッドのスレッド名が格納されているアドレスを取得します。
 *
 * @details
 *  詳細は nn::os::GetThreadNamePointer() を参照して下さい。
 *
 */
NN_OS_EXTERN_C const char* nnosGetThreadNamePointer(const nnosThreadType* thread);


//--------------------------------------------------------------------------
/**
 * @brief   現在自スレッドが動作しているコア番号を取得します。
 *
 * @deprecated
 *
 * @details
 *  nnosGetCurrentProcessorNumber() は削除予定です。@n
 *  詳細は nn::os::GetCurrentCoreNumber() を参照して下さい。
 *
 */
NN_OS_EXTERN_C int nnosGetCurrentProcessorNumber(void);


//--------------------------------------------------------------------------
/**
 * @brief   現在自スレッドが動作しているコア番号を取得します。
 *
 * @details
 *  詳細は nn::os::GetCurrentCoreNumber() を参照して下さい。
 *
 */
NN_OS_EXTERN_C int nnosGetCurrentCoreNumber(void);

//! @}


//! @name 多重待ち関連 API
//! @{

//--------------------------------------------------------------------------
/**
 * @brief   multiWaitHolder を初期化し、thread と関連付けます。
 *
 * @details
 *  詳細は nn::os::InitializeMultiWaitHolder(MultiWaitHolderType* multiWaitHolder, ThreadType* thread) を参照して下さい。
 *
 */
NN_OS_EXTERN_C void nnosInitializeMultiWaitHolderForThread(nnosMultiWaitHolderType* multiWaitHolder, nnosThreadType* thread);

//! @}

