﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   条件変数に関する C 言語用 API の宣言
 */

#pragma once

#include <nn/os/os_Config.h>
#include <nnc/nn_Common.h>
#include <nnc/os/os_Macro.h>
#include <nnc/os/os_MutexTypes.h>
#include <nnc/os/os_ConditionVariableCommon.h>
#include <nnc/os/os_ConditionVariableTypes.h>
#include <nnc/os/detail/os_MacroImpl.h>

#if defined(NN_BUILD_CONFIG_OS_WIN32)
    #include <nnc/os/detail/os_ConditionVariableApi-os.win32.h>
#elif defined(NN_BUILD_CONFIG_OS_HORIZON)
    #include <nnc/os/detail/os_ConditionVariableApi-os.horizon.h>
#endif

//--------------------------------------------------------------------------
//  C 向けの宣言
//--------------------------------------------------------------------------

//! @name 条件変数関連 C 言語用 API
//! @{

//--------------------------------------------------------------------------
/**
 * @brief   nnosConditionVariableType オブジェクトを初期化します。
 *
 * @details
 *  詳細は nn::os::InitializeConditionVariable() を参照して下さい。
 *
 */
NN_OS_EXTERN_C void nnosInitializeConditionVariable(nnosConditionVariableType* conditionVariable);


//--------------------------------------------------------------------------
/**
 * @brief   nnosConditionVariableType オブジェクトを Finalize します。
 *
 * @details
 *  詳細は nn::os::FinalizeConditionVariable() を参照して下さい。
 *
 */
NN_OS_EXTERN_C void nnosFinalizeConditionVariable(nnosConditionVariableType* conditionVariable);


//--------------------------------------------------------------------------
/**
 * @brief   nnosConditionVariableType にシグナルを通知します。
 *
 * @details
 *  詳細は nn::os::SignalConditionVariable() を参照して下さい。
 *
 */
NN_OS_EXTERN_C void nnosSignalConditionVariable(nnosConditionVariableType* conditionVariable);


//--------------------------------------------------------------------------
/**
 * @brief   nnosConditionVariableType にブロードキャスト通知を行ないます。
 *
 * @details
 *  詳細は nn::os::BroadcastConditionVariable() を参照して下さい。
 *
 */
NN_OS_EXTERN_C void nnosBroadcastConditionVariable(nnosConditionVariableType* conditionVariable);


//--------------------------------------------------------------------------
/**
 * @brief   nnosMutexType のアンロックと条件変数の待機をアトミックに行ないます。
 *
 * @details
 *  詳細は nn::os::WaitConditionVariable() を参照して下さい。
 *
 */
NN_OS_EXTERN_C void nnosWaitConditionVariable(nnosConditionVariableType* conditionVariable, nnosMutexType* mutex);


//--------------------------------------------------------------------------
/**
 * @brief   MutexType のアンロックと条件変数の待機をアトミックに行ないます。
 *
 * @return  条件変数待ちが解除された要因を返します
 *
 * @details
 *  詳細は nn::os::TimedWaitConditionVariable() を参照して下さい。
 *  本 API では、タイムアウト時間をナノ秒単位で指定します。
 *
 */
NN_OS_EXTERN_C nnosConditionVariableStatus nnosTimedWaitConditionVariable(nnosConditionVariableType* conditionVariable, nnosMutexType* mutex, int64_t timeout);


#if defined(NN_BUILD_FOR_DOCUMENT_GENERATION)
//--------------------------------------------------------------------------
/**
 * @brief   nnosConditionVariableType オブジェクトを静的に初期化します。
 *
 * @details
 *  nnosConditionVariableType オブジェクトを静的に初期化するための
 *  初期化リストに展開されるマクロです。
 *
 *  条件変数の詳細は nnosInitializeConditionVariable() を参照して下さい。
 *
 *  本マクロは下記例のように、静的な nnosConditionVariableType オブジェクト
 *  の初期化にのみ使用できます。こうして初期化したオブジェクトは
 *  nnosFinalizeConditionVariable() 呼び出しによる明示的な破棄も不要です。
 *
 *  使用例：
 *      @code
 *      nnosConditionVariableType g_StaticCondVar = NN_OS_CONDITION_VARIABLE_INITIALIZER();
 *      @endcode
 *
 */
#define NN_OS_CONDITION_VARIABLE_INITIALIZER()
#endif

