﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <cstdlib>
#include <stdint.h>
#include <nn/htcs/htcs_Types.h>

//==============================================================================

namespace nn { namespace tma { namespace htcs {

    //==============================================================================
    void Initialize() NN_NOEXCEPT;

    //==============================================================================
    void Finalize() NN_NOEXCEPT;

    //==============================================================================
    // Returns a new socket, or -1 if failed (call GetLastError() for more detail)
    int Socket() NN_NOEXCEPT;

    //==============================================================================
    // Closes the passed socket.  Returns 0 on success, -1 if failed
    int Close( int descriptor ) NN_NOEXCEPT;

    //==============================================================================
    // Connects to the service on the remote host.  Returns 0 on success, -1 if failed.
    int Connect( int descriptor, const nn::htcs::SockAddrHtcs* address ) NN_NOEXCEPT;

    //==============================================================================
    // Bind a socket to a local port number and IP address.  Returns 0 on success, -1 if failed.
    int Bind( int descriptor, const nn::htcs::SockAddrHtcs* address ) NN_NOEXCEPT;

    //==============================================================================
    // Place a socket in a state where it is listening for an incoming connection.
    // Returns 0 on success, -1 if failed.
    int Listen( int descriptor, int backlogCount ) NN_NOEXCEPT;

    //==============================================================================
    // Accept a connection from any remote host.
    // Returns 0 on success, -1 if failed.
    int Accept( int descriptor, nn::htcs::SockAddrHtcs* address ) NN_NOEXCEPT;

    //==============================================================================
    // Receive data at a socket.
    // Returns number of bytes received
    // Returns 0 if the remote host has terminated transmission of data
    // Returns -1 on failure.
    nn::htcs::ssize_t Recv( int descriptor, void* buffer, size_t bufferByteSize, int flags ) NN_NOEXCEPT;

    //==============================================================================
    // Send data to a connected socket.
    // Returns number of bytes sent
    // Returns -1 on failure.
    nn::htcs::ssize_t Send( int descriptor, const void* buffer, size_t bufferByteSize, int flags ) NN_NOEXCEPT;

    //==============================================================================
    // Disable receives and/or sends on a socket.
    // Returns 0 on success, -1 if failed.
    int Shutdown( int descriptor, int how ) NN_NOEXCEPT;

    //==============================================================================
    // Controls the operation setting of the socket.
    // Returns -1 if failed.
    int Fcntl( int descriptor, int command, int value ) NN_NOEXCEPT;

    //==============================================================================
    // Returns the socket error enumeration value that represents the details of the last error.
    int GetLastError() NN_NOEXCEPT;

    //==============================================================================
    // When Bind, you can use if you want to sepcify your own HtcsPeerName.  It
    // applies to INADDR_ANY in the socket.
    const nn::htcs::HtcsPeerName GetPeerNameAny() NN_NOEXCEPT;

    //==============================================================================
    // During connect, you can use if you want to specify a default host PC to HtcsPeerName.
    const nn::htcs::HtcsPeerName GetDefaultHostName() NN_NOEXCEPT;

    //==============================================================================
    // Types of errors
    enum SocketError
    {
        HTCS_EIPC             = 80,       // Internal IPC error
    };

//==============================================================================
}}} // namespace nn::tma::htcs
//==============================================================================
