﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_TimeSpan.h>
#include <nn/svc/svc_HardwareParamsSelect.h>

#ifdef __cplusplus

namespace nn { namespace svc {

/*!
    :category   時間

    @brief システムチックを扱う為のクラスです。

           システムチックは、CPU の 1 クロックの時間を表します。
           変換コンストラクタ・変換演算子を使うことで、実際の時間を表す @ref TimeSpan との相互変換ができます。
           システムが立ち上がってからのチック数を取得するには @ref GetSystemCurrent を使います。

           システムチックは個体差があり、温度により大きく変動して、おおよそ月差±300秒程度の精度を持ちます。
           これは製品として保証する値ではありません。

           @ref nn::fnd::DateTime::GetNow() で得られる現在日時とは進む速度が異なるため、混在して使用してはいけません。
           @ref nn::os::Tick::ToTimeSpan() を使用することにより、同じ単位の値が得られますが、同時に時間を計っても同じ値は得られません。

           また、多くの場合、内部で時間を扱うライブラリ（例：サウンドや動画のストリーミング再生）で扱われる時間とも進む速度が異なります。
           例えば、アプリの演出とサウンド再生の同期を取りたい場合には、個々のライブラリの仕様を把握し、速度の一致する時間を使用してください。
*/
class Tick
{
public:
    /*!
        @brief 1秒間のチック数を表す定数です。
    */
    static const int64_t TICKS_PER_SECOND  = NN_HW_TICKS_PER_SECOND;
    static int64_t GetTicksPerSecond() { return TICKS_PER_SECOND; }
public:

//! @name コンストラクタ
//! @{

    /*!
        @brief チック数を取ってオブジェクトを初期化するコンストラクタです。

        @param[in] tick チック数
    */
    explicit Tick(int64_t tick = 0) : m_Tick(tick) {}

    /*!
        @brief TimeSpan を取ってオブジェクトを初期化するコンストラクタです。

        @param[in] span 初期化する時間の値
    */
    explicit Tick(TimeSpan span);

//! @}

//! @name 変換
//! @{

    /*!
        @brief 64bit のチック値に変換します。
    */
    NN_IMPLICIT operator int64_t() const { return m_Tick; }

//! @}

//! @name 和差
//! @{

    //----------------------------------------------------------------------
    //! @brief  チック値を引きます。
    //!
    //! @param[in]  rhs     差し引くチック値
    //! @return     自身へのリファレンスを返します。
    //----------------------------------------------------------------------
    Tick& operator-=(Tick rhs);

    //----------------------------------------------------------------------
    //! @brief  チック値の差を求めます。
    //!
    //! @param[in]  rhs     差を取るチック値
    //! @return     自身の値と rhs の値の差と同じチック値を持つ Tick を返します。
    //----------------------------------------------------------------------
    Tick operator-(Tick rhs) const;

    //----------------------------------------------------------------------
    //! @brief  チック値を加えます。
    //!
    //! @param[in]  rhs     加えるチック値
    //! @return     自身へのリファレンスを返します。
    //----------------------------------------------------------------------
    Tick& operator+=(Tick rhs);

    //----------------------------------------------------------------------
    //! @brief  チック値の和を求めます。
    //!
    //! @param[in]  rhs     和を取るチック値
    //! @return     自身の値と rhs の値の和と同じチック値を持つ Tick を返します。
    //----------------------------------------------------------------------
    Tick operator+(Tick rhs) const;

    //----------------------------------------------------------------------
    //! @brief  TimeSpan を加えます。
    //!
    //! @param[in]  rhs     加える TimeSpan
    //! @return     自身へのリファレンスを返します。
    //----------------------------------------------------------------------
    Tick& operator+=(TimeSpan rhs);

    //----------------------------------------------------------------------
    //! @brief  TimeSpan との和を求めます。
    //!
    //! @param[in]  rhs     和を取る TimeSpan
    //! @return     自身の値と rhs をチックに変換した値との和と同じチック値を持つ Tick を返します。
    //----------------------------------------------------------------------
    Tick operator+(TimeSpan rhs) const;

private:
    static void DivideNanoSeconds(int64_t* pQuotient, const int64_t value)
    {
#if defined (NN_BUILD_CONFIG_ABI_ILP32)
        const int64_t magic = 0x112e0be826d694b3LL;
        const int rightShift = 26;

        int64_t n = MultiplyRightShift(value, magic);
        n >>= rightShift;
        *pQuotient = n + (static_cast<uint64_t>(value) >> 63);
#elif defined (NN_BUILD_CONFIG_ABI_LP64)
        *pQuotient = value / 1000000000;
#else
#error not supported
#endif
    }

    static void DivideNanoSeconds(int64_t* pQuotient, int64_t* pMod, const int64_t value)
    {
        DivideNanoSeconds(pQuotient, value);
        int64_t mod = value - *pQuotient * 1000000000;
        *pMod = mod;
    }

#if !defined (NN_BUILD_CONFIG_ABI_LP64)
    static int64_t MultiplyRightShift(const int64_t x, const int64_t y)
    {
        const uint64_t x_lo = x & 0xffffffff;
        const int64_t x_hi = x >> 32;

        const uint64_t y_lo = y & 0xffffffff;
        const int64_t y_hi = y >> 32;

        const int64_t z = x_hi * y_lo + ((x_lo * y_lo) >> 32);
        const int64_t z_lo = z & 0xffffffff;
        const int64_t z_hi = z >> 32;

        return x_hi * y_hi + z_hi + (static_cast<int64_t>(x_lo * y_hi + z_lo) >> 32);
    }
#endif

private:
    int64_t m_Tick;
};


inline Tick::Tick(TimeSpan span)
{
    int64_t ns0;
    int64_t ns1;

    int64_t tick0;
    int64_t tick1;
    DivideNanoSeconds(&ns0, &ns1, span.GetNanoSeconds());
    DivideNanoSeconds(&tick0, &tick1, GetTicksPerSecond());

    int64_t x;
    DivideNanoSeconds(&x, ns1 * tick1 + 1000000000ll - 1);

    m_Tick = (ns0 * tick0) * 1000000000ll + (ns0 * tick1 + tick0 * ns1) + x;
}

inline Tick& Tick::operator+=(TimeSpan rhs)
{
    Tick tick(rhs);
    this->m_Tick += tick.m_Tick;
    return *this;
}

inline Tick& Tick::operator-=(Tick rhs)
{
    this->m_Tick -= rhs.m_Tick;
    return *this;
}
inline Tick  Tick::operator-(Tick rhs) const
{
    Tick ret(*this);
    return ret -= rhs;
}
inline Tick& Tick::operator+=(Tick rhs)
{
    this->m_Tick += rhs.m_Tick;
    return *this;
}
inline Tick  Tick::operator+(Tick rhs) const
{
    Tick ret(*this);
    return ret += rhs;
}
inline Tick  Tick::operator+(TimeSpan rhs) const
{
    Tick ret(*this);
    return ret += rhs;
}

}}

#endif // __cplusplus

