﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <mutex>

#include <nn/nn_Macro.h>
#include <nn/socket/socket_Statistics.h>
#include <nn/socket/socket_Errno.h>
#include <nn/socket/socket_Types.h>
#include <nn/socket/socket_TypesPrivate.h>
#include <nn/socket/sys/socket.h>

namespace nn      {
namespace socket  {
namespace detail  {

#if defined(NN_BUILD_CONFIG_OS_WIN32)

static const int        WinsockHandleTableSize = MAX_SOCKETS_PER_CLIENT;

class PosixWinSockConverter
{
public:

    PosixWinSockConverter()
    {
        std::fill_n(g_WinsockHandleTable, WinsockHandleTableSize, INVALID_SOCKET);
    }

    int WinsockToPosixSocket(SOCKET winsockSocket) NN_NOEXCEPT;
    SOCKET PosixToWinsockSocket(int posixSocket) NN_NOEXCEPT;
    void ReleasePosixHandle(int posixSocket) NN_NOEXCEPT;
    int AcquirePosixHandle(SOCKET winsockSocket) NN_NOEXCEPT;

private:
    std::mutex        g_DescriptorTableLock;
    SOCKET            g_WinsockHandleTable[WinsockHandleTableSize];
};

int32_t MapProtocolValue(Protocol protocol);
Protocol MapProtocolValue(int32_t protocol);
int32_t MapTypeValue(Type type);
Type MapTypeValue(int32_t type);
int8_t MapFamilyValue(Family family);
Family MapFamilyValue(int8_t family);
int32_t MapMsgFlagValue(MsgFlag flag);
MsgFlag MapMsgFlagValue(int32_t flag);
int16_t MapPollEventValue(PollEvent poll);
PollEvent MapPollEventValue(int16_t poll);
uint32_t MapAddrInfoFlagValue(AddrInfoFlag aiFlag);
AddrInfoFlag MapAddrInfoFlagValue(uint32_t aiFlag);
uint32_t MapNameInfoFlagValue(NameInfoFlag niFlag);
NameInfoFlag MapNameInfoFlagValue(uint32_t niFlag);
uint32_t MapShutdownMethodValue(ShutdownMethod how);
ShutdownMethod MapShutdownMethodValue(uint32_t how);
uint32_t MapIoctlCommandValue(IoctlCommand command);
IoctlCommand MapIoctlCommandValue(uint32_t command);
uint32_t MapFcntlCommandValue(FcntlCommand command);
FcntlCommand MapFcntlCommandValue(uint32_t command);
uint32_t MapFcntlFlagValue(FcntlFlag flag);
FcntlFlag MapFcntlFlagValue(uint32_t flag);
int32_t MapLevelValue(Level level);
Level MapLevelValue(int32_t level);
int32_t MapOptionValue(Level level, Option option);
Option MapOptionValue(int32_t level, int32_t option);
int32_t MapErrnoValue(Errno error);
Errno MapErrnoValue(int32_t error);
int32_t MapErrnoValue(Errno error);
Errno MapErrnoValue(int32_t error);
int32_t MapHErrnoValue(HErrno error);
HErrno MapHErrnoValue(int32_t error);
int32_t MapAiErrnoValue(AiErrno error);
AiErrno MapAiErrnoValue(int32_t error);
int MapTcpInfoOptionValue(TcpInfoOption option);
TcpInfoOption MapTcpInfoOptionValue(int option);

#endif

void CopyToPlatform(sockaddr_in* pDest, const SockAddrIn* pSrc) NN_NOEXCEPT;
void CopyFromPlatform(SockAddrIn* pDest, const sockaddr_in* pSrc) NN_NOEXCEPT;
void CopyToPlatform(in_addr* pDest, const InAddr* pSrc) NN_NOEXCEPT;
void CopyFromPlatform(InAddr* pDest, const in_addr* pSrc) NN_NOEXCEPT;
bool CopyFromPlatform(AddrInfo** pDest, const addrinfo* pSrc) NN_NOEXCEPT;
void FreeCopiedAddrInfo(AddrInfo *pAddrInfo) NN_NOEXCEPT;
void CopyToPlatform(hostent* pDest, const HostEnt* pSrc) NN_NOEXCEPT;
void CopyFromPlatform(HostEnt* pDest, const hostent* pSrc) NN_NOEXCEPT;
void CopyToPlatform(timeval* pDest, const TimeVal* pSrc) NN_NOEXCEPT;
void CopyFromPlatform(TimeVal* pDest, const timeval* pSrc) NN_NOEXCEPT;
void CopyToPlatform(pollfd* pDest, const PollFd* pSrc) NN_NOEXCEPT;
void CopyFromPlatform(PollFd* pDest, const pollfd* pSrc) NN_NOEXCEPT;
void CopyToPlatform(fd_set* pDest, const FdSet* pSrc) NN_NOEXCEPT;
void CopyFromPlatform(FdSet* pDest, const fd_set* pSrc) NN_NOEXCEPT;
void CopyToPlatform(linger* pDest, const Linger* pSrc) NN_NOEXCEPT;
void CopyFromPlatform(Linger* pDest, const linger* pSrc) NN_NOEXCEPT;

#if !defined(NN_BUILD_CONFIG_OS_WIN32)
void CopyToPlatform(tcp_info* pDest, const TcpInfo* pSrc) NN_NOEXCEPT;
void CopyFromPlatform(TcpInfo* pDest, const tcp_info* pSrc) NN_NOEXCEPT;
#endif

}}} /* nn::socket::detail */
