﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/**
    @file
    @brief サービスフレームワークで使用されるネイティブハンドルの型を定義します。
*/

#include <nn/nn_Common.h>
#include <type_traits>
#include <utility>
#include <nn/sf/sf_Out.h>
#include <nn/os/os_NativeHandle.h>
#include <nn/util/util_Exchange.h>


namespace nn { namespace sf {

/**
    @brief サービスフレームワークでネイティブハンドルを扱う際に使用する型です。

    @details
     このクラスでは、ネイティブハンドル(os::NativeHandle)の値を保持するとともに、ネイティブハンドルの寿命管理も行います。
     寿命管理を行うかどうかは初期化時等に指定することができ、 IsManaged() によって管理が行われているかどうかを取得することが可能です。

     管理が行われている状態でこのクラスのインスタンスが破棄された場合には、デストラクタ内で os::CloseNativeHandle が呼ばれ、ネイティブハンドルが閉じられます。

     このクラスは寿命管理を行うため、コピー不可であり、基本的にムーブで扱う必要があります。
*/
class NativeHandle
{
    NN_DISALLOW_COPY(NativeHandle);
private:

    os::NativeHandle m_InnerValue;
    bool m_Managed;

public:

    /**
        @brief デフォルトコンストラクタ: 無効なネイティブハンドルを保持するものとして、初期化します。

        @post this->GetOsHandle() == os::InvalidNativeHandle
        @post this->IsManaged() == false
    */
    NativeHandle() NN_NOEXCEPT
        : m_InnerValue(os::InvalidNativeHandle)
        , m_Managed(false)
    {
    }

    /**
        @brief ムーブコンストラクタ: 管理権も含めてムーブして初期化します。

        @param[in] other ムーブ対象のオブジェクトを指定します。

        @post this->GetOsHandle() == (ムーブ前の other).GetOsHandle()
        @post this->IsManaged() == (ムーブ前の other).IsManaged()
        @post other.GetOsHandle() == os::InvalidNativeHandle
        @post other.IsManaged() == false
    */
    NativeHandle(NativeHandle&& other) NN_NOEXCEPT
        : m_InnerValue(other.m_InnerValue)
        , m_Managed(other.m_Managed)
    {
        other.m_InnerValue = os::InvalidNativeHandle;
        other.m_Managed = false;
    }

    /**
        @brief コンストラクタ: os::NativeHandle を用いて初期化します。

        @param[in] osHandle os ライブラリのネイティブハンドル値を指定します。
        @param[in] managed 指定された os ハンドルを管理するかどうかを指定します。

        @post this->GetOsHandle() == osHandle
        @post this->IsManaged() == managed
    */
    NativeHandle(os::NativeHandle osHandle, bool managed) NN_NOEXCEPT
        : m_InnerValue(osHandle)
        , m_Managed(managed)
    {
    }

    /**
        @brief デストラクタ: オブジェクトを解放し、ハンドルを管理している場合にはハンドルを閉じます。

        @details
         this->IsManaged() == true で呼ばれた場合には、内部で this->GetOsHandle() を引数として os::CloseNativeHandle() が呼ばれます。
    */
    ~NativeHandle() NN_NOEXCEPT
    {
        if (m_Managed)
        {
            os::CloseNativeHandle(m_InnerValue);
        }
    }

    /**
        @brief 管理権を放棄します。

        @post this->GetOsHandle() == os::InvalidNativeHandle
        @post this->IsManaged() == false

        @details
         os::NativeHandle を受け取る関数等に this->GetOsHandle() を引数として渡し、
         ネイティブハンドルの管理を移管した場合、この関数を呼んで管理を放棄してください。
    */
    void Detach() NN_NOEXCEPT
    {
        this->m_Managed = false;
        this->m_InnerValue = os::InvalidNativeHandle;
    }

    /**
        @brief 対象のオブジェクトとスワップします。

        @param[in] other スワップ対象のオブジェクトを指定します。

        @post this->GetOsHandle() == (スワップ前の other).GetOsHandle()
        @post other.GetOsHandle() == (スワップ前の this)->GetOsHandle()
        @post this->IsManaged() == (スワップ前の other).IsManaged()
        @post other.IsManaged() == (スワップ前の this)->IsManaged()
    */
    void swap(NativeHandle& other) NN_NOEXCEPT
    {
        using std::swap;
        swap(m_InnerValue, other.m_InnerValue);
        swap(m_Managed, other.m_Managed);
    }

    /**
        @brief ムーブ代入演算子: 指定したオブジェクトからムーブ代入を行います。

        @param[in] other ムーブ対象のオブジェクトを指定します。

        @return *this を返します。

        @post this->GetOsHandle() == (ムーブ前の other).GetOsHandle()
        @post this->IsManaged() == (ムーブ前の other).IsManaged()
        @post other.GetOsHandle() == os::InvalidNativeHandle
        @post other.IsManaged() == false

        @details
         ネイティブハンドルを管理するかどうかも含めて other からムーブ代入を行います。
         既に *this が管理を行うネイティブハンドを保持していた場合には、そのネイティブハンドルに対して os::CloseNativeHandle() が呼ばれます。
    */
    NativeHandle& operator=(NativeHandle&& other) NN_NOEXCEPT
    {
        NativeHandle(std::move(other)).swap(*this);
        return *this;
    }

    /**
        @brief 同じネイティブハンドルを保持し、管理を行わないオブジェクトを返します。

        @return 同じネイティブハンドルを保持し、管理を行わないオブジェクトを返します。

        @post (返り値).GetOsHandle() == this->GetOsHandle()
        @post (返り値).IsManaged() == false
    */
    NativeHandle GetShared() const NN_NOEXCEPT
    {
        return NativeHandle(m_InnerValue, false);
    }

    /**
        @brief 保持しているネイティブハンドルを返します。

        @return 保持しているネイティブハンドルを返します。
    */
    os::NativeHandle GetOsHandle() const NN_NOEXCEPT
    {
        return m_InnerValue;
    }

    /**
        @brief 保持しているネイティブハンドルを管理しているかどうかを返します。

        @return 管理しているなら true 返し、管理していないなら false を返します。
    */
    bool IsManaged() const NN_NOEXCEPT
    {
        return m_Managed;
    }

    /**
        @brief オブジェクトをリセットします。

        @post this->GetOsHandle() == os::InvalidNativeHandle
        @post this->IsManaged() == false

        @details
         *this = NativeHandle() と同等です。
    */
    void Reset() NN_NOEXCEPT
    {
        NativeHandle().swap(*this);
    }

};

//! @name NativeHandle 外部関数
//! @{

/**
    @brief swap: 二つの NativeHandle を swap します。

    @param x swap する一つ目の NativeHandle を指定します。
    @param y swap する二つ目の NativeHandle を指定します。

    @post x.GetOsHandle() == (スワップ前の y).GetOsHandle()
    @post y.GetOsHandle() == (スワップ前の x).GetOsHandle()
    @post x.IsManaged() == (スワップ前の y).IsManaged()
    @post y.IsManaged() == (スワップ前の x).IsManaged()
*/
inline void swap(NativeHandle& x, NativeHandle& y) NN_NOEXCEPT
{
    x.swap(y);
}

//! @}

/**
    @brief ネイティブハンドルに対する Out の特殊化です。
*/
template <>
class Out<NativeHandle>
{
private:

    NativeHandle* m_P;

public:

    /**
        @brief コンストラクタ: ネイティブハンドルのポインタを使用して初期化します。
    */
    NN_IMPLICIT Out(NativeHandle* p) NN_NOEXCEPT
        : m_P(p)
    {
    }

    /**
        @brief ネイティブハンドルをセットします。
    */
    void Set(NativeHandle x) const NN_NOEXCEPT
    {
        *this->m_P = std::move(x);
    }

    /**
        @brief 逆参照演算子: 内部に設定されているポインタの逆参照を取得します。

        @return 内部に設定されているポインタの逆参照を返します。
    */
    NativeHandle& operator*() const NN_NOEXCEPT
    {
        return *m_P;
    }

};

}}
