﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/**
    @file
    @brief サービスフレームワークで HIPC を使用する際に、セッション数を管理するプロキシオブジェクト生成機能を提供します。
    @details
     このヘッダで提供される関数の仕様は、今後変更される(破壊的な変更も含む)可能性があります。

     このファイルは <nn/sf/sf_HipcClient.h> をインクルードすることでインクルードされます。

     このファイルをインクルードすることで、以下のファイルで定義されているものを間接的にインクルードします。

     - <nn/sf/sf_HipcPortCommon.h>
*/

#include <nn/sf/hipc/client/sf_HipcSimpleClientSessionManagerBase.h>

namespace nn { namespace sf {

/**
    @brief HipcSimpleClientSessionManager のインスタンスで管理できるセッション数の最大値を表す定数です。
*/
const int HipcSimpleClientSessionManagerSessionCountMax = 16;

/**
    @brief オブジェクトへの呼び出しを、特定の 1 つ以上の HIPC セッションを共通で利用して呼び出すためのクラスです。

    @tparam SessionResourceManager IPC セッションの利用をカスタマイズする際に指定します。

    @details
     このクラスは初期化(Initialize 系関数呼び出し)時にオブジェクトを HIPC を使って取得し、
     この際に取得されたセッションをインスタンス内部に保持します。
     さらに、サーバ内にサブドメインを新規作成し、取得したオブジェクトをサブドメイン内オブジェクトとして登録します。
     初期化時に返されるオブジェクト参照は、このサブドメイン内オブジェクトへの参照です。
     初期化時の引数は nn::sf::CreateHipcProxyByName() や nn::sf::CreateHipcProxyByHandle() と同じです。

     このクラスを使ってオブジェクト参照を得る場合には、対象のサーバがサブドメインに対応している必要があります。
     例えば HipcSimpleAllInOneServerManager を使用している場合には、型引数 Option に、
     SubDomainCountMax と ObjectInSubDomainCountMax を適切に指定する必要があります。

     サブドメイン内オブジェクトへのアクセスは nn::sf::CreateHipcProxyByName() などによって得られたものと基本的には同様に行うことができます。
     サブドメイン内オブジェクトに対する呼び出しにおいて、新たにオブジェクトが返ったような場合には、
     そのオブジェクトもサブドメインに登録され、
     新たなオブジェクトへのアクセスも元と同じセッションが使用されます。

     IncreaseSession() や SetSessionCount() によって、
     セッション数(クライアントがサーバに同時に呼び出すことができる数)を変更することができます。
     これらの関数は、上記の関数による初期化後でないと使用することができません。

     SessionResourceManager を指定すると、IPC セッションの利用をカスタマイズすることができます。
     カスタマイズが不要な場合には、本クラステンプレートではなく HipcSimpleClientSessionManager を使用してください。

     SessionResourceManager クラスは以下のメンバ関数を持つ必要があります。

     - bool OnAcquire()
     - void OnRelease()

     SessionResourceManager::OnAcquire() 関数は、IPC 呼び出しを行う際に必要なセッションを割り当てようとした際に呼ばれます。
     この関数が true を返した際には、セッションが割り当てられ IPC 呼び出しが行われます。
     false を返すとセッションは割り当てられず、スレッドは待機状態になります。
     一方、 SessionResourceManager::OnRelease() 関数は、IPC 呼び出しが終わりセッションの割り当てが開放される際に呼ばれます。
     この呼び出しの後に、前述の待機状態のスレッドがいた場合、そのスレッドで再度 SessionResourceManager::OnAcquire() の呼び出しが行われます。
     また、この 2 関数の呼び出しは排他されており、同時に呼び出されることはありません。
     両関数から共有する変数に対してのアクセスを排他する必要はありません。

     SessionResourceManager::OnAcquire() および SessionResourceManager::OnRelease() は、
     その時にコンテキスト(TLS など)から取得できる情報を使用し、セッション数を管理することを想定しています。

    @see nn::sf::CreateHipcProxyByName(), nn::sf::CreateHipcProxyByHandle()
*/
template <typename SessionResourceManager>
class HipcCustomClientSessionManager
    : private hipc::client::HipcSimpleClientSessionManagerBaseN<HipcSimpleClientSessionManagerSessionCountMax, SessionResourceManager>
{
private:

    typedef hipc::client::HipcSimpleClientSessionManagerBaseN<HipcSimpleClientSessionManagerSessionCountMax, SessionResourceManager> Base;

public:

    //! @name コンストラクタとデストラクタ
    //! @{

    /**
        @brief (コンストラクタ) クライアントセッションマネージャを初期化します。

        @param[in] pSessionResourceManager SessionResourceManager のインスタンスを指定します。

        @post インスタンスが未初期化状態である
    */
    NN_IMPLICIT HipcCustomClientSessionManager(SessionResourceManager* pSessionResourceManager) NN_NOEXCEPT
        : Base(pSessionResourceManager)
    {
    }

    /**
        @brief (デストラクタ) インスタンスを破棄します。

        @details
         関連付けられているセッションがある場合には、セッションをすべて閉じます。
         この際、使用中のセッションがある場合には、使用が完了するまで待機します。
    */
    ~HipcCustomClientSessionManager() NN_NOEXCEPT
    {
    }

    //! @}

    //! @name 初期化処理と終了処理
    //! @{

    /**
        @brief オブジェクトを取得し、クライアントセッションマネージャに紐づけます。

        @tparam Interface サービスの提供するインターフェイス型を指定します。
        @tparam AllocationPolicy 状態なしのアロケーションポリシーを指定します。

        @param[out] pOut プロキシへの共有ポインタを格納するバッファのへポインタを指定します。
        @param[in] handle セッションハンドルを指定します。

        @retresult
        @endretresult

        @pre インスタンスが未初期化状態である
        @pre handle で表されるセッションが Interface を実装するオブジェクトに関連付けられている

        @post インスタンスが初期化済み状態である
        @post GetSessionCount() == 1

        @details
         引数の詳細は nn::sf::CreateHipcProxyByHandle() を参照してください。

        @see nn::sf::CreateHipcProxyByHandle()
    */
    template <typename Interface, typename AllocationPolicy>
    Result InitializeByHandle(SharedPointer<Interface>* pOut, HipcClientSessionHandle handle) NN_NOEXCEPT;


    /**
        @brief オブジェクトを取得し、クライアントセッションマネージャに紐づけます。

        @tparam Interface サービスの提供するインターフェイス型を指定します。
        @tparam AllocationPolicy 状態なしのアロケーションポリシーを指定します。

        @param[out] pOut プロキシへの共有ポインタを格納するバッファのへポインタを指定します。
        @param[in] pAllocator アロケータへのポインタを指定します。
        @param[in] handle セッションハンドルを指定します。

        @retresult
        @endretresult

        @pre インスタンスが未初期化状態である
        @pre handle で表されるセッションが Interface を実装するオブジェクトに関連付けられている

        @post インスタンスが初期化済み状態である
        @post GetSessionCount() == 1

        @details
         引数の詳細は nn::sf::CreateHipcProxyByHandle() を参照してください。

        @see nn::sf::CreateHipcProxyByHandle()
    */
    template <typename Interface, typename AllocationPolicy>
    Result InitializeByHandle(SharedPointer<Interface>* pOut, typename AllocationPolicy::Allocator* pAllocator, HipcClientSessionHandle handle) NN_NOEXCEPT;

    /**
        @brief オブジェクトを取得し、クライアントセッションマネージャに紐づけます。

        @tparam Interface サービスの提供するインターフェイス型を指定します。
        @tparam AllocationPolicy 状態無しのアロケーションポリシーを指定します。

        @param[out] pOut プロキシへの共有ポインタを格納するバッファのへポインタを指定します。
        @param[in] serviceName サービス名を指定します。

        @retresult
        @endretresult

        @pre インスタンスが未初期化状態である
        @pre std::strlen(serviceName) > 0
        @pre std::strlen(serviceName) <= nn::sf::HipcServiceNameLengthMax

        @post インスタンスが初期化済み状態である
        @post GetSessionCount() == 1

        @details
         引数の詳細は nn::sf::CreateHipcProxyByName() を参照してください。

        @see nn::sf::CreateHipcProxyByName()
    */
    template <typename Interface, typename AllocationPolicy>
    Result InitializeByName(SharedPointer<Interface>* pOut, const char* serviceName) NN_NOEXCEPT;


    /**
        @brief オブジェクトを取得し、クライアントセッションマネージャに紐づけます。

        @tparam Interface サービスの提供するインターフェイス型を指定します。
        @tparam AllocationPolicy 状態無しのアロケーションポリシーを指定します。

        @param[out] pOut プロキシへの共有ポインタを格納するバッファのへポインタを指定します。
        @param[in] pAllocator アロケータへのポインタを指定します。
        @param[in] serviceName サービス名を指定します。

        @retresult
        @endretresult

        @pre インスタンスが未初期化状態である
        @pre std::strlen(serviceName) > 0
        @pre std::strlen(serviceName) <= nn::sf::HipcServiceNameLengthMax

        @post インスタンスが初期化済み状態である
        @post GetSessionCount() == 1

        @details
         引数の詳細は nn::sf::CreateHipcProxyByName() を参照してください。

        @see nn::sf::CreateHipcProxyByName()
    */
    template <typename Interface, typename AllocationPolicy>
    Result InitializeByName(SharedPointer<Interface>* pOut, typename AllocationPolicy::Allocator* pAllocator, const char* serviceName) NN_NOEXCEPT;

    /**
        @brief インスタンスを未初期化状態にします。

        @pre インスタンスが初期化済み状態である
        @post インスタンスが未初期化状態である

        @details
         関連付けられているセッションがある場合には、セッションをすべて閉じます。
         この際、使用中のセッションがある場合には、使用が完了するまで待機します。
    */
    void Finalize() NN_NOEXCEPT;

    //! @}

    //! @name セッション数管理
    //! @{

    /**
        @brief セッション数を 1 増加します。

        @retresult
        @endretresult

        @pre GetSessionCount() < nn::sf::HipcSimpleClientSessionManagerSessionCountMax
        @pre インスタンスが初期化済み状態である

        @post GetSessionCount() == (呼び出し前の GetsessionCount()) + 1

        @details
         セッションを新たに作成し、セッション数を 1 増加します。
         作成されたセッションは以降のアクセスに使用されます。

         この関数は内部で既存のセッションに対するサーバ呼び出しを行います。
         サーバ側で処理がされないような場合には、相応の時間ブロックする可能性があります。
    */
    Result IncreaseSession() NN_NOEXCEPT;

    /**
        @brief セッション数を設定します。

        @param[in] sessionCount

        @retresult
        @endretresult

        @pre インスタンスが初期化済み状態である。
        @pre sessionCount > 0
        @pre sessionCount <= nn::sf::HipcSimpleClientSessionManagerSessionCountMax

        @post GetSessionCount() == sessionCount

        @details
         セッション数を sessionCount に設定します。

         GetSessionCount() が sessionCount より少なかった場合には、内部で IncreaseSession() を必要な回数呼びます。
         この際、呼び出しに失敗した場合にはプログラムが停止します。

         GetSessionCount() が sessionCount より多かった場合には、必要な数だけセッションを閉じます。
         sessionCount 個以上のセッションが使用中の場合には、ブロックします。
    */
    Result SetSessionCount(int sessionCount) NN_NOEXCEPT;

    /**
        @brief セッション数を返します。

        @return セッション数を返します。
    */
    int GetSessionCount() const NN_NOEXCEPT;

    //! @}

};

#if defined(NN_BUILD_FOR_DOCUMENT_GENERATION)

/**
    @brief オブジェクトへの呼び出しを、特定の 1 つ以上の HIPC セッションを共通で利用して呼び出すためのクラスです。

    @details
     HipcCustomClientSessionManager クラステンプレートに対し、特にセッションリソース管理を行わない SessionResourceManager を与えたものです。
     詳しくは HipcCustomClientSessionManager を参照してください。

    @see HipcCustomClientSessionManager
*/
class HipcSimpleClientSessionManager
    : public HipcCustomClientSessionManager<unspecified>
{
public:

    /**
        @brief (コンストラクタ) クライアントセッションマネージャを初期化します。

        @post インスタンスが未初期化状態である
    */
    HipcSimpleClientSessionManager() NN_NOEXCEPT;

};

#else

class HipcSimpleClientSessionManager
    : private sf::hipc::client::DefaultSessionResourceManager
    , public HipcCustomClientSessionManager<sf::hipc::client::DefaultSessionResourceManager>
{
public:

    HipcSimpleClientSessionManager() NN_NOEXCEPT
        : HipcCustomClientSessionManager<sf::hipc::client::DefaultSessionResourceManager>(this)
    {
    }

};

#endif

template <typename SessionResourceManager>
template <typename Interface, typename AllocationPolicy>
inline Result HipcCustomClientSessionManager<SessionResourceManager>::InitializeByHandle(SharedPointer<Interface>* pOut, HipcClientSessionHandle handle) NN_NOEXCEPT
{
    return Base::template InitializeByHandle<Interface, AllocationPolicy>(pOut, handle);
}

template <typename SessionResourceManager>
template <typename Interface, typename AllocationPolicy>
inline Result HipcCustomClientSessionManager<SessionResourceManager>::InitializeByHandle(SharedPointer<Interface>* pOut, typename AllocationPolicy::Allocator* pAllocator, HipcClientSessionHandle handle) NN_NOEXCEPT
{
    return Base::template InitializeByHandle<Interface, AllocationPolicy>(pOut, pAllocator, handle);
}

template <typename SessionResourceManager>
template <typename Interface, typename AllocationPolicy>
inline Result HipcCustomClientSessionManager<SessionResourceManager>::InitializeByName(SharedPointer<Interface>* pOut, const char* serviceName) NN_NOEXCEPT
{
    return Base::template InitializeByName<Interface, AllocationPolicy>(pOut, serviceName);
}

template <typename SessionResourceManager>
template <typename Interface, typename AllocationPolicy>
inline Result HipcCustomClientSessionManager<SessionResourceManager>::InitializeByName(SharedPointer<Interface>* pOut, typename AllocationPolicy::Allocator* pAllocator, const char* serviceName) NN_NOEXCEPT
{
    return Base::template InitializeByName<Interface, AllocationPolicy>(pOut, pAllocator, serviceName);
}

template <typename SessionResourceManager>
inline void HipcCustomClientSessionManager<SessionResourceManager>::Finalize() NN_NOEXCEPT
{
    Base::Release();
}

template <typename SessionResourceManager>
inline Result HipcCustomClientSessionManager<SessionResourceManager>::IncreaseSession() NN_NOEXCEPT
{
    return Base::IncreaseSession();
}

template <typename SessionResourceManager>
inline Result HipcCustomClientSessionManager<SessionResourceManager>::SetSessionCount(int sessionCount) NN_NOEXCEPT
{
    return Base::SetSessionCount(sessionCount);
}

template <typename SessionResourceManager>
inline int HipcCustomClientSessionManager<SessionResourceManager>::GetSessionCount() const NN_NOEXCEPT
{
    return Base::GetSessionCount();
}

}}
