﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/**
    @file
    @brief サービスフレームワークで HIPC を使用する際に、サービス名によるプロキシオブジェクト生成を提供します。
    @details
     このヘッダで提供される関数の仕様は、今後変更される(破壊的な変更も含む)可能性があります。

     このファイルは <nn/sf/sf_HipcClient.h> をインクルードすることでインクルードされます。

     このファイルをインクルードすることで、以下のファイルで定義されているものを間接的にインクルードします。

     - <nn/sf/sf_HipcPortCommon.h>
     - <nn/sf/sf_HipcClientSession.h>
*/

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/sf/sf_ISharedObject.h>
#include <nn/sf/hipc/sf_HipcHandleTypes.h>
#include <nn/sf/sf_HipcPortCommon.h>
#include <nn/sf/sf_StandardAllocationPolicy.h>
#include <nn/sf/sf_HipcClientSession.h>

#include <nn/sf/hipc/sf_HipcServiceResolutionApi.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/sf/sf_Helper.h>
#include <nn/sf/sf_MemoryResource.h>

namespace nn { namespace sf {

//! @name HIPC プロキシオブジェクト生成 (サービス名指定)
//! @{

/**
    @brief アロケーションポリシーを指定して、サービス名に関連づけられたサービスオブジェクトへのプロキシを取得します。

    @tparam Interface サービスの提供するインターフェイス型を指定します。
    @tparam AllocationPolicy 状態無しのアロケーションポリシーを指定します。

    @param[in] serviceName サービス名を指定します。

    @retresult{処理に失敗した場合には該当する Result を返し、成功した場合にはプロキシ参照を返します。}
    @endretresult

    @pre std::strlen(serviceName) > 0
    @pre std::strlen(serviceName) <= HipcServiceNameLengthMax

    @details
     serviceName で指定されたサービス名に問い合わせ、
     サービスへのプロキシオブジェクトを生成し *pOut に設定します。
*/
template <typename Interface, typename AllocationPolicy>
ResultAnd<HipcRef<Interface>> CreateHipcProxyByName(const char* serviceName) NN_NOEXCEPT;

/**
    @brief アロケーションポリシーを指定して、サービス名に関連づけられたサービスオブジェクトへのプロキシを取得します。

    @tparam Interface サービスの提供するインターフェイス型を指定します。
    @tparam AllocationPolicy 状態ありのアロケーションポリシーを指定します。

    @param[in] pAllocator アロケータへのポインタを指定します。
    @param[in] serviceName サービス名を指定します。

    @retresult{処理に失敗した場合には該当する Result を返し、成功した場合にはプロキシ参照を返します。}
    @endretresult

    @pre std::strlen(serviceName) > 0
    @pre std::strlen(serviceName) <= HipcServiceNameLengthMax

    @details
     serviceName で指定されたサービス名に問い合わせ、
     サービスへのプロキシオブジェクトを生成し *pOut に設定します。
*/
template <typename Interface, typename AllocationPolicy>
ResultAnd<HipcRef<Interface>> CreateHipcProxyByName(typename AllocationPolicy::Allocator* pAllocator, const char* serviceName) NN_NOEXCEPT;

/**
    @brief MemoryResource をアロケータとして、サービス名に関連づけられたサービスオブジェクトへのプロキシを取得します。

    @tparam Interface サービスの提供するインターフェイス型を指定します。

    @param[in] pMemoryResource MemoryResource へのポインタを指定します。
    @param[in] serviceName サービス名を指定します。

    @retresult{処理に失敗した場合には該当する Result を返し、成功した場合にはプロキシ参照を返します。}
    @endretresult

    @pre std::strlen(serviceName) > 0
    @pre std::strlen(serviceName) <= HipcServiceNameLengthMax

    @details
     serviceName で指定されたサービス名に問い合わせ、
     サービスへのプロキシオブジェクトを生成し *pOut に設定します。

     アロケーションポリシーとしては MemoryResourceAllocationPolicy が使用されます。
*/
template <typename Interface>
ResultAnd<HipcRef<Interface>> CreateHipcProxyByName(MemoryResource* pMemoryResource, const char* serviceName) NN_NOEXCEPT;

/**
    @brief (古い形式) サービス名に関連づけられたサービスオブジェクトへのプロキシを取得します。

    @tparam Interface サービスの提供するインターフェイス型を指定します。
    @tparam AllocationPolicy 状態無しのアロケーションポリシーを指定します。

    @param[out] pOut プロキシへの共有ポインタを格納するバッファのへポインタを指定します。
    @param[in] serviceName サービス名を指定します。

    @retresult
    @endretresult

    @pre std::strlen(serviceName) > 0
    @pre std::strlen(serviceName) <= HipcServiceNameLengthMax

    @details
     serviceName で指定されたサービス名に問い合わせ、
     サービスへのプロキシオブジェクトを生成し *pOut に設定します。
*/
template <typename Interface, typename AllocationPolicy>
Result CreateHipcProxyByName(SharedPointer<Interface>* pOut, const char* serviceName) NN_NOEXCEPT;

/**
    @brief (古い形式) サービス名に関連づけられたサービスオブジェクトへのプロキシを取得します。

    @tparam Interface サービスの提供するインターフェイス型を指定します。
    @tparam AllocationPolicy 状態ありのアロケーションポリシーを指定します。

    @param[out] pOut プロキシへの共有ポインタを格納するバッファのへポインタを指定します。
    @param[in] pAllocator アロケータへのポインタを指定します。
    @param[in] serviceName サービス名を指定します。

    @retresult
    @endretresult

    @pre std::strlen(serviceName) > 0
    @pre std::strlen(serviceName) <= HipcServiceNameLengthMax

    @details
     この関数テンプレートの仕様は今後変更される(破壊的な変更も含む)可能性があります。

     serviceName で指定されたサービス名に問い合わせ、
     サービスへのプロキシオブジェクトを生成し *pOut に設定します。
*/
template <typename Interface, typename AllocationPolicy>
Result CreateHipcProxyByName(SharedPointer<Interface>* pOut, typename AllocationPolicy::Allocator* pAllocator, const char* serviceName) NN_NOEXCEPT;

/**
    @brief (古い形式) MemoryResource をアロケータとして、サービス名に関連づけられたサービスオブジェクトへのプロキシを取得します。

    @tparam Interface サービスの提供するインターフェイス型を指定します。

    @param[out] pOut プロキシへの共有ポインタを格納するバッファのへポインタを指定します。
    @param[in] pMemoryResource MemoryResource へのポインタを指定します。
    @param[in] serviceName サービス名を指定します。

    @retresult
    @endretresult

    @pre std::strlen(serviceName) > 0
    @pre std::strlen(serviceName) <= HipcServiceNameLengthMax

    @details
     この関数テンプレートの仕様は今後変更される(破壊的な変更も含む)可能性があります。

     serviceName で指定されたサービス名に問い合わせ、
     サービスへのプロキシオブジェクトを生成し *pOut に設定します。

     アロケーションポリシーとしては MemoryResourceAllocationPolicy が使用されます。
*/
template <typename Interface>
Result CreateHipcProxyByName(SharedPointer<Interface>* pOut, MemoryResource* pMemoryResource, const char* serviceName) NN_NOEXCEPT;

//! @}

// 以下実装

namespace detail {

inline Result GetHipcClientSessionHandleSafely(HipcClientSessionHandle* pOut, const char* serviceName) NN_NOEXCEPT
{
    hipc::InitializeHipcServiceResolution();
    NN_UTIL_SCOPE_EXIT
    {
        hipc::FinalizeHipcServiceResolution();
    };
    return hipc::ConnectToHipcService(pOut, serviceName);
}

}

template <typename Interface, typename AllocationPolicy>
inline ResultAnd<HipcRef<Interface>> CreateHipcProxyByName(const char* serviceName) NN_NOEXCEPT
{
    HipcClientSessionHandle clientSessionHandle;
    NN_RESULT_DO(detail::GetHipcClientSessionHandleSafely(&clientSessionHandle, serviceName));
    return ResultAnd<HipcRef<Interface>>(detail::HipcRefAccessor::CreateHipcProxyByHandle<Interface, AllocationPolicy>(nullptr, clientSessionHandle));
}

template <typename Interface, typename AllocationPolicy>
inline ResultAnd<HipcRef<Interface>> CreateHipcProxyByName(typename AllocationPolicy::Allocator* pAllocator, const char* serviceName) NN_NOEXCEPT
{
    HipcClientSessionHandle clientSessionHandle;
    NN_RESULT_DO(detail::GetHipcClientSessionHandleSafely(&clientSessionHandle, serviceName));
    return ResultAnd<HipcRef<Interface>>(detail::HipcRefAccessor::CreateHipcProxyByHandle<Interface, AllocationPolicy>(pAllocator, clientSessionHandle));
}

template <typename Interface>
inline ResultAnd<HipcRef<Interface>> CreateHipcProxyByName(MemoryResource* pMemoryResource, const char* serviceName) NN_NOEXCEPT
{
    return CreateHipcProxyByName<Interface, MemoryResourceAllocationPolicy>(pMemoryResource, serviceName);
}

template <typename Interface, typename AllocationPolicy>
inline Result CreateHipcProxyByName(SharedPointer<Interface>* pOut, const char* serviceName) NN_NOEXCEPT
{
    HipcClientSessionHandle clientSessionHandle;
    NN_RESULT_DO(detail::GetHipcClientSessionHandleSafely(&clientSessionHandle, serviceName));
    *pOut = detail::HipcRefAccessor::CreateHipcProxyByHandle<Interface, AllocationPolicy>(nullptr, clientSessionHandle);
    NN_RESULT_SUCCESS;
}

template <typename Interface, typename AllocationPolicy>
inline Result CreateHipcProxyByName(SharedPointer<Interface>* pOut, typename AllocationPolicy::Allocator* pAllocator, const char* serviceName) NN_NOEXCEPT
{
    HipcClientSessionHandle clientSessionHandle;
    NN_RESULT_DO(detail::GetHipcClientSessionHandleSafely(&clientSessionHandle, serviceName));
    *pOut = detail::HipcRefAccessor::CreateHipcProxyByHandle<Interface, AllocationPolicy>(pAllocator, clientSessionHandle);
    NN_RESULT_SUCCESS;
}

template <typename Interface>
inline Result CreateHipcProxyByName(SharedPointer<Interface>* pOut, MemoryResource* pMemoryResource, const char* serviceName) NN_NOEXCEPT
{
    return CreateHipcProxyByName<Interface, MemoryResourceAllocationPolicy>(pOut, pMemoryResource, serviceName);
}

}}
