﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/sf/detail/sf_Config.h>
#include <nn/nn_StaticAssert.h>
#include <nn/nn_Result.h>
#include <nn/nn_Abort.h>
#include <nn/sf/sf_Result.h>
#include <utility>

// SF 実装用のユーティリティ定義

#if NN_SF_DETAIL_USE_CONSTEXPR && !defined(__INTELLISENSE__)
    #define NN_SF_DETAIL_CONSTEXPR constexpr
#else
    #define NN_SF_DETAIL_CONSTEXPR
#endif

#define NN_SF_DETAIL_OUT_RETURN_NAME _pOut

#define NN_SF_DETAIL_STRIP_PAREN_IMPL(...) __VA_ARGS__
#define NN_SF_DETAIL_STRIP_PAREN(x) NN_SF_DETAIL_STRIP_PAREN_IMPL x

#define NN_SF_DETAIL_INTEGRAL_CONSTANT(v) ::std::integral_constant<decltype(v), v>

#define NN_SF_DETAIL_DEFINE_TYPE_LIST(name, types) typedef ::nn::sf::detail::TypeList<NN_SF_DETAIL_STRIP_PAREN(types)> name;

namespace nn { namespace sf { namespace detail {

struct MonoState
{
};

template <typename T, int Size>
class SafeArray
{
private:

    T m_Array[Size];

public:

    T* Get() NN_NOEXCEPT
    {
        return m_Array;
    }

    const T* Get() const NN_NOEXCEPT
    {
        return m_Array;
    }

};

template <typename T>
class SafeArray<T, 0>
{
public:

    T* Get() const NN_NOEXCEPT
    {
        return nullptr;
    }

};

template <typename T>
struct TypeIdentity
{
    typedef T type;
};

template <typename... Types>
struct TypeList
{
};

template <typename Setter, int N, typename TypeList>
struct TypeListSetterImpl;

template <typename Setter, int N>
struct TypeListSetterImpl<Setter, N, TypeList<>>
{
    template <typename ArrayType>
    NN_SF_DETAIL_CONSTEXPR static void Write(ArrayType&) NN_NOEXCEPT
    {
    }
};

template <typename Setter, int N, typename Type, typename... Types>
struct TypeListSetterImpl<Setter, N, TypeList<Type, Types...>>
{
    template <typename ArrayType>
    NN_SF_DETAIL_CONSTEXPR static void Write(ArrayType& x) NN_NOEXCEPT
    {
        Setter::template Set<Type>(x, N);
        TypeListSetterImpl<Setter, N + 1, TypeList<Types...>>::Write(x);
    }
};

template <typename Setter, typename TypeList>
struct TypeListSetter
{
    template <typename ArrayType>
    NN_SF_DETAIL_CONSTEXPR static void Set(ArrayType& x) NN_NOEXCEPT
    {
        TypeListSetterImpl<Setter, 0, TypeList>::Write(x);
    }
};

template <typename T>
NN_SF_DETAIL_CONSTEXPR inline T AlignUp(T n, size_t align) NN_NOEXCEPT
{
    return (n + align - 1) / align * align;
}

template<typename StorageT, typename Tag = void>
struct BitPack
{
    StorageT storage;

    NN_SF_DETAIL_CONSTEXPR void Clear() NN_NOEXCEPT
    {
        storage = StorageT(0);
    }

    NN_SF_DETAIL_CONSTEXPR void SetMaskedBits(int mask, int value) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES((~mask & value) == 0);
        storage &= ~mask;
        storage |= value;
    }

    NN_SF_DETAIL_CONSTEXPR int GetMaskedBits(int mask) const NN_NOEXCEPT
    {
        return static_cast<int>(storage & mask);
    }

    template<typename FieldT>
    NN_SF_DETAIL_CONSTEXPR void Set(typename FieldT::Type value) NN_NOEXCEPT
    {
        typedef std::is_same<
            FieldT,
            Field<FieldT::Pos, FieldT::Width, typename FieldT::Type>
        > IsValidFieldT;
        NN_STATIC_ASSERT(IsValidFieldT::value);
        SetMaskedBits(FieldT::Mask, value <<FieldT::Pos);
    }

    template<typename FieldT>
    NN_SF_DETAIL_CONSTEXPR typename FieldT::Type Get() const NN_NOEXCEPT
    {
        typedef std::is_same<
            FieldT,
            Field<FieldT::Pos, FieldT::Width, typename FieldT::Type>
        > IsValidFieldT;
        NN_STATIC_ASSERT(IsValidFieldT::value);

        typename FieldT::Type value;
        ReadValue(&value, FieldT::Mask, FieldT::Pos);
        return value;
    }

    template<int BitPos, int BitWidth, typename T>
    struct Field
    {
        typedef T Type;
        static const int Pos = BitPos;
        static const int Next = BitPos + BitWidth;
        static const int Mask = static_cast<int>(((uint64_t(1) <<BitWidth) - 1) <<BitPos);

        static const int Width = BitWidth;

        NN_STATIC_ASSERT(std::is_integral<T>::value);
        NN_STATIC_ASSERT(!(std::is_same<T, bool>::value) || BitWidth == 1);
        NN_STATIC_ASSERT(BitWidth <= NN_BITSIZEOF(T));
        NN_STATIC_ASSERT(Next <= NN_BITSIZEOF(StorageT));
    };

private:

    NN_STATIC_ASSERT(std::is_integral<StorageT>::value);
    NN_STATIC_ASSERT(sizeof(StorageT) <= sizeof(int));

    template<typename T>
    NN_SF_DETAIL_CONSTEXPR void ReadValue(T* pValue, int mask, int pos) const NN_NOEXCEPT
    {
        *pValue = static_cast<T>(static_cast<unsigned int>(GetMaskedBits(mask))>> pos);
    }

};

template<typename StorageT, typename Tag>
template<int BitPos, int BitWidth, typename T>
const int BitPack<StorageT, Tag>::Field<BitPos, BitWidth, T>::Pos;

template<typename StorageT, typename Tag>
template<int BitPos, int BitWidth, typename T>
const int BitPack<StorageT, Tag>::Field<BitPos, BitWidth, T>::Next;

template<typename StorageT, typename Tag>
template<int BitPos, int BitWidth, typename T>
const int BitPack<StorageT, Tag>::Field<BitPos, BitWidth, T>::Mask;

template <typename R, typename T>
NN_SF_DETAIL_CONSTEXPR inline R ExtractBits(T value, int position, int width) NN_NOEXCEPT
{
    return static_cast<R>((value >> position) & ((uint64_t(1) << width) - 1));
}

inline Result ToPreConditionResult(Result result) NN_NOEXCEPT
{
    return result;
}

inline Result ToPreConditionResult(bool b) NN_NOEXCEPT
{
    if (!b)
    {
        return nn::sf::ResultPreConditionViolation();
    }
    return nn::ResultSuccess();
}

inline bool ToPreConditionBool(Result result) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    return result.IsSuccess();
}

inline bool ToPreConditionBool(bool b) NN_NOEXCEPT
{
    return b;
}

#define NN_SF_DETAIL_APPLY_LAMBDA(arg, result, target) \
    ([](const ::std::decay<decltype((target))>::type& arg) { return result; })(target)

#if 1 // begin type list

// ConvertContainer

template <template <typename...> class Container, typename List>
struct ConvertContainerImpl;

template <template <typename...> class DestinationContainer, template <typename...> class SourceContainer, typename... Args>
struct ConvertContainerImpl<DestinationContainer, SourceContainer<Args...>>
{
    using type = DestinationContainer<Args...>;
};

template <template <typename...> class Container, typename List>
using ConvertContainer = ConvertContainerImpl<Container, List>;

// TypeSizeOf

template <template <typename...> class Container, typename List>
struct TypeSizeOfImpl;

template <template <typename...> class Container, typename... Args>
struct TypeSizeOfImpl<Container, Container<Args...>>
    : public std::integral_constant<std::size_t, sizeof...(Args)>
{
};

template <typename List>
struct TypeSizeOf;

template <template <typename...> class Container, typename... Args>
struct TypeSizeOf<Container<Args...>>
    : public TypeSizeOfImpl<Container, Container<Args...>>
{
};

// TypeAddHead

template <template <typename...> class Container, typename Head, typename>
struct TypeAddHeadImpl;

template <template <typename...> class Container, typename Head, typename... Tail>
struct TypeAddHeadImpl<Container, Head, Container<Tail...>>
{
    using type = Container<Head, Tail...>;
};

template <typename Head, typename List>
struct TypeAddHead;

template <template <typename...> class Container, typename Head, typename... Args>
struct TypeAddHead<Head, Container<Args...>>
    : public TypeAddHeadImpl<Container, Head, Container<Args...>>
{
};

// TypeRemoveHead

template <template <typename...> class Container, typename>
struct TypeRemoveHeadImpl;

template <template <typename...> class Container, typename Head, typename... Tail>
struct TypeRemoveHeadImpl<Container, Container<Head, Tail...>>
{
    using type = Container<Tail...>;
};

#ifdef NN_BUILD_CONFIG_TOOLCHAIN_VC_VS2013

template <typename T>
struct TypeRemoveHeadImpl<std::tuple, std::tuple<T>>
{
    using type = std::tuple<>;
};

#endif

template <typename List>
struct TypeRemoveHead;

template <template <typename...> class Container, typename... Args>
struct TypeRemoveHead<Container<Args...>>
    : public TypeRemoveHeadImpl<Container, Container<Args...>>
{
};

// TypeElementAt

template <template <typename...> class Container, int N, typename List, typename Enabled = void>
struct TypeElementAtImpl;

template <template <typename...> class Container, typename Head, typename... Tail>
struct TypeElementAtImpl<Container, 0, Container<Head, Tail...>>
{
    using type = Head;
};

template <template <typename...> class Container, int N, typename Head, typename... Tail>
struct TypeElementAtImpl<Container, N, Container<Head, Tail...>, typename std::enable_if<(N > 0)>::type>
    : public TypeElementAtImpl<Container, N - 1, Container<Tail...>>
{
    // typename std::enable_if<(N > 0)>::type をしないと VS のバグを踏む
};

template <int N, typename List>
struct TypeElementAt;

template <template <typename...> class Container, int N, typename... Args>
struct TypeElementAt<N, Container<Args...>>
    : public TypeElementAtImpl<Container, N, Container<Args...>>
{
};

// TypeMap

template <template <typename...> class Container, typename F, typename List>
struct TypeMapImpl;

#ifdef NN_BUILD_CONFIG_TOOLCHAIN_VC_VS2013

template <template <typename...> class Container, typename F>
struct TypeMapImpl<Container, F, Container<>>
{
    using type = Container<>;
};

template <typename F>
struct TypeMapImpl<std::tuple, F, std::tuple<>>
{
    using type = std::tuple<>;
};

template <template <typename...> class Container, typename F, typename Head, typename... Tail>
struct TypeMapImpl<Container, F, Container<Head, Tail...>>
    : public TypeAddHead<typename F::template apply<Head>::type, typename TypeMapImpl<Container, F, Container<Tail...>>::type>
{
};

#else

template <template <typename...> class Container, typename F, typename... Args>
struct TypeMapImpl<Container, F, Container<Args...>>
{
    using type = Container<typename F::template apply<Args>::type...>;
};

#endif

template <typename F, typename List>
struct TypeMap;

template <template <typename...> class Container, typename F, typename... Args>
struct TypeMap<F, Container<Args...>>
    : public TypeMapImpl<Container, F, Container<Args...>>
{
};

// TypeAppend

template <template <typename...> class Container, typename L, typename R>
struct TypeAppendImpl;

template <template <typename...> class Container, typename... Ls, typename... Rs>
struct TypeAppendImpl<Container, Container<Ls...>, Container<Rs...>>
{
    using type = Container<Ls..., Rs...>;
};

template <typename L, typename R>
struct TypeAppend;

template <template <typename...> class Container, typename... Ls, typename... Rs>
struct TypeAppend<Container<Ls...>, Container<Rs...>>
    : public TypeAppendImpl<Container, Container<Ls...>, Container<Rs...>>
{
};

// TypeFlatten

template <template <typename...> class Container, typename Lists>
struct TypeFlattenImpl;

template <template <typename...> class Container>
struct TypeFlattenImpl<Container, Container<>>
{
    using type = Container<>;
};

#ifdef NN_BUILD_CONFIG_TOOLCHAIN_VC_VS2013

template <>
struct TypeFlattenImpl<std::tuple, std::tuple<>>
{
    using type = std::tuple<>;
};

#endif

template <template <typename...> class Container, typename Head, typename... Tail>
struct TypeFlattenImpl<Container, Container<Head, Tail...>>
    : public TypeAppend<Head, typename TypeFlattenImpl<Container, Container<Tail...>>::type>
{
};

template <typename Lists>
struct TypeFlatten;

template <template <typename...> class Container, typename... Lists>
struct TypeFlatten<Container<Lists...>>
    : public TypeFlattenImpl<Container, Container<Lists...>>
{
};

// TypeFilter

template <template <typename...> class Container, typename P, typename List>
struct TypeFilterImpl;

template <template <typename...> class Container, typename P>
struct TypeFilterImpl<Container, P, Container<>>
{
    using type = Container<>;
};

template <template <typename...> class Container, typename P, typename Head, typename... Tail>
struct TypeFilterImpl<Container, P, Container<Head, Tail...>>
    : public std::conditional<P::template predicate<Head>::value
        , TypeAddHead<Head, typename TypeFilterImpl<Container, P, Container<Tail...>>::type>
        , TypeFilterImpl<Container, P, Container<Tail...>>
    >::type
{
};

#ifdef NN_BUILD_CONFIG_TOOLCHAIN_VC_VS2013

template <typename P>
struct TypeFilterImpl<std::tuple, P, std::tuple<>>
{
    using type = std::tuple<>;
};

#endif

template <typename P, typename List>
struct TypeFilter;

template <template <typename...> class Container, typename P, typename... Args>
struct TypeFilter<P, Container<Args...>>
    : public TypeFilterImpl<Container, P, Container<Args...>>
{
};

// TypeCountIf

template <typename P, typename List>
struct TypeCountIf
    : public TypeSizeOf<typename TypeFilter<P, List>::type>
{
};

// TypeFold

template <template <typename...> class Container, class I, typename F, typename List>
struct TypeFoldImpl;

template <template <typename...> class Container, class I, typename F>
struct TypeFoldImpl<Container, I, F, Container<>>
{
    using type = I;
};

template <template <typename...> class Container, class I, typename F,  typename Head, typename... Tail>
struct TypeFoldImpl<Container, I, F, Container<Head, Tail...>>
    : public TypeFoldImpl<Container, typename F::template apply<I, Head>::type, F, Container<Tail...>>
{
};

#ifdef NN_BUILD_CONFIG_TOOLCHAIN_VC_VS2013

template <class I, typename F>
struct TypeFoldImpl<std::tuple, I, F, std::tuple<>>
{
    using type = I;
};

#endif

template <class I, typename F, typename List>
struct TypeFold;

template <template <typename...> class Container, class I, typename F, typename... Args>
struct TypeFold<I, F, Container<Args...>>
    : public TypeFoldImpl<Container, I, F, Container<Args...>>
{
};

// TypeForeachInvoker

template <typename T>
struct TypeInvokeTag
{
    using type = T;
};

template <template <typename...> class Container, typename List>
struct TypeForeachInvokerImpl;

template <template <typename...> class Container>
struct TypeForeachInvokerImpl<Container, Container<>>
{
    template <typename F>
    NN_SF_DETAIL_CONSTEXPR static void Invoke(F&&) NN_NOEXCEPT
    {
    }
};

template <template <typename...> class Container, typename Head, typename... Tail>
struct TypeForeachInvokerImpl<Container, Container<Head, Tail...>>
{
    template <typename F>
    NN_SF_DETAIL_CONSTEXPR static void Invoke(F&& f) NN_NOEXCEPT
    {
        f(TypeInvokeTag<Head>{});
        TypeForeachInvokerImpl<Container, Container<Tail...>>::template Invoke<F>(std::forward<F>(f));
    }
};

#ifdef NN_BUILD_CONFIG_TOOLCHAIN_VC_VS2013

template <>
struct TypeForeachInvokerImpl<std::tuple, std::tuple<>>
{
    template <typename F>
    static void Invoke(F&&) NN_NOEXCEPT
    {
    }
};

#endif

template <typename List>
struct TypeForeachInvoker;

template <template <typename...> class Container, typename... Args>
struct TypeForeachInvoker<Container<Args...>>
{
    template <typename F>
    NN_SF_DETAIL_CONSTEXPR static void Invoke(F&& f = F()) NN_NOEXCEPT
    {
        TypeForeachInvokerImpl<Container, Container<Args...>>::Invoke(std::forward<F>(f));
    }
};

template <typename List, typename F>
NN_SF_DETAIL_CONSTEXPR void TypeForeachInvoke(F&& f = F()) NN_NOEXCEPT
{
    TypeForeachInvoker<List>::template Invoke<F>(std::forward<F>(f));
}

// TypeZip

template <template <typename...> class Container, class Zip, class L, class R>
struct TypeZipImpl;

#ifdef NN_BUILD_CONFIG_TOOLCHAIN_VC_VS2013

// VS2013 では複数のパラメータパックの同時展開ができないため手動で展開する
template <template <typename...> class Container, class Zip>
struct TypeZipImpl<Container, Zip, Container<>, Container<>>
{
    using type = Container<>;
};

template <template <typename...> class Container, class Zip, typename L, typename... Ls, typename R, typename... Rs>
struct TypeZipImpl<Container, Zip, Container<L, Ls...>, Container<R, Rs...>>
    : public TypeAddHead<typename Zip::template apply<L, R>::type, typename TypeZipImpl<Container, Zip, Container<Ls...>, Container<Rs...>>::type>
{
};

template <class Zip>
struct TypeZipImpl<std::tuple, Zip, std::tuple<>, std::tuple<>>
{
    using type = std::tuple<>;
};

#else // NN_BUILD_CONFIG_TOOLCHAIN_VC_VS2013

template <template <typename...> class Container, class Zip, typename... Ls, typename... Rs>
struct TypeZipImpl<Container, Zip, Container<Ls...>, Container<Rs...>>
{
    using type = Container<typename Zip::template apply<Ls, Rs>::type...>;
};

#endif // NN_BUILD_CONFIG_TOOLCHAIN_VC_VS2013

template <class Zip, class L, class R>
struct TypeZip;

template <template <typename...> class Container, class Zip, typename... Ls, typename... Rs>
struct TypeZip<Zip, Container<Ls...>, Container<Rs...>>
    : public TypeZipImpl<Container, Zip, Container<Ls...>, Container<Rs...>>
{
};

#endif // end type list

}}}

#define NN_SF_RESULT_SUCCESS \
    return ::nn::ResultSuccess()

#define NN_SF_RESULT_THROW(r) \
    return (r)

#define NN_SF_RESULT_THROW_UNLESS(condition, r) \
    { \
        if (!(condition)) \
        { \
            NN_SF_RESULT_THROW(r); \
        } \
    } while (NN_STATIC_CONDITION(0))

#define NN_SF_RESULT_DO(r) \
    do \
    { \
        const auto& _nn_result_do_temporary((r)); \
        NN_SF_RESULT_THROW_UNLESS(_nn_result_do_temporary.IsSuccess(), _nn_result_do_temporary); \
    } while (NN_STATIC_CONDITION(0))

#define NN_SF_RESULT_TRY(r) \
    { \
        const auto& _nn_result_try_temporary((r)); \
        if (!_nn_result_try_temporary.IsSuccess()) \
        { \
            if (NN_STATIC_CONDITION(false)) {}

#define NN_SF_RESULT_CATCH(type) \
            else if (type::Includes(_nn_result_try_temporary))

#define NN_SF_RESULT_CATCH_ALL \
            else if (NN_STATIC_CONDITION(true))

#define NN_SF_RESULT_CURRENT_RESULT \
            _nn_result_try_temporary

#define NN_SF_RESULT_RETHROW \
                NN_SF_RESULT_THROW(_nn_result_try_temporary)

#define NN_SF_RESULT_END_TRY \
            else \
            { \
                NN_SF_RESULT_RETHROW; \
            } \
        } \
    }

