﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/**
    @file
    @brief オーバーレイ通知表示用の通知を受信するための機能を提供します。
*/

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <type_traits>
#include <nn/ovln/ovln_CommonTypes.h>
#include <nn/ovln/ovln_Result.h>
#include <nn/os/os_MultipleWaitTypes.h>
#include <nn/os/os_TickTypes.h>

namespace nn { namespace ovln {

//! @name 受信(オーバレイ通知表示用)
//! @{

/**
    @brief オーバレイ通知受信ライブラリを初期化します。

    @retresult
    @endretresult

    @details
     オーバレイ通知受信ライブラリを初期化します。
     この関数は複数回呼ぶことができ、
     この関数と FinalizeReceicerLibraryForOverlay() 関数とが同じ回数だけ呼ばれている状態でこの関数を呼んだときに限り、
     実際の初期化処理が行われます。
*/
Result InitializeReceiverLibraryForOverlay() NN_NOEXCEPT;

/**
    @brief オーバレイ通知受信ライブラリを終了します。

    @details
     オーバレイ通知受信ライブラリを終了します。
     InitializeReceiverLibraryForOverlay() と同じ回数だけ呼んだ場合に実際の終了処理が行われます。
*/
void FinalizeReceicerLibraryForOverlay() NN_NOEXCEPT;

/**
    @brief オーバレイ通知受信に使用するための型です。

    @details
     以下の状態を持ちます。

     - 未初期化状態 (デフォルト)
     - 初期化済み状態

    @see InitializeReceiver(ReceiverForOverlayType*)
*/
struct ReceiverForOverlayType
{
    std::aligned_storage<128>::type _storage;
};

/**
    @brief ReceiverForOverlayType を初期化し、通知を受信できるようにします。

    @param[out] pReceiver 初期化する ReceiverForOverlayType の変数へのポインタを指定します。

    @retresult
    @endretresult

    @pre オーバレイ通知受信ライブラリが初期化されている
    @pre pReceiver が未初期化状態である
    @post pReceiver が初期化済み状態である

    @details
     pReceiver で指定された ReceiverForOverlayType を初期化し、通知を受信できるようにします。
*/
Result InitializeReceiver(ReceiverForOverlayType* pReceiver) NN_NOEXCEPT;

/**
    @brief ReceiverForOverlayType を終了し、通知を受信できないようにします。

    @param[in] pReceiver 終了する ReceiverForOverlayType の変数へのポインタを指定します。

    @pre オーバレイ通知受信ライブラリが初期化されている
    @pre pReceiver が初期化済み状態である
    @post pReceiver が未初期化状態である

    @details
     pReceiver で指定された ReceiverForOverlayType を終了し、使用していたリソースを解放します。
*/
void FinalizeReceiver(ReceiverForOverlayType* pReceiver) NN_NOEXCEPT;

/**
    @brief メッセージの受信を試行します。

    @param[out] pOut 受信したメッセージを書き込む変数へのポインタを指定します。
    @param[in] pReceiver 受信対象の ReceiverForOverlayType を指定します。
    @return 受信できた場合には true を、そうでない場合には false を返します。

    @pre オーバレイ通知受信ライブラリが初期化されている
    @pre pReceiver が初期化済み状態である
    @post true が返ったとき、*pOut に受信したメッセージが書き込まれている

    @details
     メッセージの受信を試行し、受信できる状態であれば *pOut に受信をし、true を返します。
     受信できる状態でなければ、即座に false を返します。
*/
bool TryReceive(Message* pOut, ReceiverForOverlayType* pReceiver) NN_NOEXCEPT;

/**
    @brief メッセージの受信を試行します。

    @param[out] pOut 受信したメッセージを書き込む変数へのポインタを指定します。
    @param[out] pOutTick メッセージの送信時のシステムチックを書き込む変数へのポインタを指定します。
    @param[in] pReceiver 受信対象の ReceiverForOverlayType を指定します。
    @return 受信できた場合には true を、そうでない場合には false を返します。

    @pre オーバレイ通知受信ライブラリが初期化されている
    @pre pReceiver が初期化済み状態である
    @post true が返ったとき、*pOut に受信したメッセージが書き込まれている
    @post true が帰ったとき、*pOutTick に受信メッセージの送信時のシステムチックが書き込まれている

    @details
     メッセージの受信を試行し、受信できる状態であれば *pOut に受信をし、true を返します。
     また、送信時のシステムチックを *pOutTick に書き込みます。
     受信できる状態でなければ、即座に false を返します。

     取得できるチック値は、システムの起動から受信メッセージの送信時までの時間を表します。
     os::Tick::GetSystemTick() で取得できるチック値と差をとることで、
     メッセージの送信時からの経過時間を取得することができます。

     取得できるチック値は単調増加するとは限りません。
     後に受信したメッセージのチック値のほうが小さいことがあります。
     このため、メッセージのフィルタリングなどをする際には、
     そのメッセージの種類とそのメッセージのチック値だけを判断材料とし、
     前後のメッセージのチック値に依存しないようにしてください。
*/
bool TryReceive(Message* pOut, os::Tick* pOutTick, ReceiverForOverlayType* pReceiver) NN_NOEXCEPT;

/**
    @brief メッセージを受信します。

    @param[out] pOut 受信したメッセージを書き込む変数へのポインタを指定します。
    @param[in] pReceiver 受信対象の ReceiverForOverlayType を指定します。

    @pre オーバレイ通知受信ライブラリが初期化されている
    @pre pReceiver が初期化済み状態である
    @post *pOut に受信したメッセージが書き込まれている

    @details
     メッセージを *pOut に受信をします。
     受信できる状態でなければ、受信できる状態になるまでブロックをしたのちに、受信します。
*/
void Receive(Message* pOut, ReceiverForOverlayType* pReceiver) NN_NOEXCEPT;

/**
    @brief メッセージを受信します。

    @param[out] pOut 受信したメッセージを書き込む変数へのポインタを指定します。
    @param[out] pOutTick メッセージの送信時のシステムチックを書き込む変数へのポインタを指定します。
    @param[in] pReceiver 受信対象の ReceiverForOverlayType を指定します。

    @pre オーバレイ通知受信ライブラリが初期化されている
    @pre pReceiver が初期化済み状態である
    @post *pOut に受信したメッセージが書き込まれている
    @post *pOutTick に受信メッセージの送信時のシステムチックが書き込まれている

    @details
     メッセージを *pOut に受信をします。
     また、送信時のシステムチックを *pOutTick に書き込みます。
     受信できる状態でなければ、受信できる状態になるまでブロックをしたのちに、受信します。

     取得できるチック値は単調増加するとは限りません。
     後に受信したメッセージのチック値のほうが小さいことがあります。
     このため、メッセージのフィルタリングなどをする際には、
     そのメッセージの種類とそのメッセージのチック値だけを判断材料とし、
     前後のメッセージのチック値に依存しないようにしてください。
*/
void Receive(Message* pOut, os::Tick* pOutTick, ReceiverForOverlayType* pReceiver) NN_NOEXCEPT;

/**
    @brief 受信できるメッセージがあるかどうかを取得します。

    @return 受信できるメッセージがあれば true を、そうでなければ false を返します。

    @pre *pReceiver が初期化済み状態である
*/
bool HasMessageToReceive(ReceiverForOverlayType* pReceiver) NN_NOEXCEPT;

/**
    @brief メッセージが受信できるようになるまでブロックします。

    @pre オーバレイ通知受信ライブラリが初期化されている
    @pre *pReceiver が初期化済み状態である

    @details
     メッセージが受信できるようになるまでブロックしますが、
     このブロックが解除されても必ずしもメッセージが受信できるとは限りません。
     この関数から返った直後であっても、同じ引数で Receive(Message*, ReceiverForOverlayType*) を呼んだ場合にブロックする可能性があります。
     この関数から返ったのちは TryReceive(Message*, ReceiverForOverlayType*) を呼び、false が返った場合には再度この関数を呼ぶようにしてください。
*/
void WaitForMessageToReceive(ReceiverForOverlayType* pReceiver) NN_NOEXCEPT;

/**
    @brief 多重待ちホルダーを指定した ReceiverForOverlayType で初期化します。

    @pre オーバレイ通知受信ライブラリが初期化されている
    @pre pMultiWaitHolder が未初期化である
    @pre pReceiver が初期化済み状態である

    @details
     WaitForMessageToReceive(ReceiverForOverlayType*) と同等の待機を行う待機オブジェクトを使用して、
     pMultiWaitHodler を初期化します。
     この関数で初期化した pMultiWaitHolder は os::FinalizeMultiWaitHolder() で終了処理を行ってください。
*/
void InitializeMultiWaitHolder(os::MultiWaitHolderType* pMultiWaitHolder, ReceiverForOverlayType* pReceiver) NN_NOEXCEPT;

//! @}

}}
