﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/ovln/ovln_ServiceTypes.h>
#include <nn/ovln/ovln_CommonTypes.h>
#include <nn/os/os_MultipleWaitTypes.h>
#include <nn/os/os_SystemEventTypes.h>
#include <nn/os/os_TickTypes.h>
#include <type_traits>

namespace nn { namespace ovln {

Result InitializeReceiverLibrary() NN_NOEXCEPT;
void FinalizeReceicerLibrary() NN_NOEXCEPT;

class IReceiver;

class Receiver
{
    NN_DISALLOW_COPY(Receiver);
public:

    /**
        @brief デフォルトコンストラクタ

        @post this->IsValid() == false
    */
    Receiver() NN_NOEXCEPT;

    /**
        @brief 初期化

        @retresult
        @retresultend

        @pre this->IsValid() == false
        @post this->IsValid() == true
    */
    Result Initialize() NN_NOEXCEPT;

    /**
        @brief 終了

        @post IsValid() == false
    */
    void Finalize() NN_NOEXCEPT;

    /**
        @brief デストラクタ
    */
    ~Receiver() NN_NOEXCEPT;

    /**
        @brief ムーブコンストラクタ
    */
    Receiver(Receiver&& receiver) NN_NOEXCEPT;

    /**
        @brief ムーブ代入演算子
    */
    Receiver& operator=(Receiver&& rhs) NN_NOEXCEPT;

    /**
        @brief swap
    */
    void swap(Receiver& other) NN_NOEXCEPT;

    /**
        @brief 妥当性判定
    */
    bool IsValid() const NN_NOEXCEPT;

    /**
        @brief 妥当性判定
    */
    NN_EXPLICIT_OPERATOR bool() const NN_NOEXCEPT
    {
        return IsValid();
    }

    /**
        @brief ソース追加

        @pre this->IsValid() == true
        @post sourceName で指定されるソースが受信対象である
    */
    Result AddSource(const SourceName& sourceName) NN_NOEXCEPT;

    /**
        @brief ソース削除

        @pre this->IsValid() == true
        @pre sourceName で指定されるソースが受信対象である
        @post sourceName で指定されるソースが受信対象でない
    */
    void RemoveSource(const SourceName& sourceName) NN_NOEXCEPT;

    /**
        @brief メッセージの受信を試行する

        @return 受信できるメッセージがあれば true を返し、そうでなければ false を返す

        @post true が返ったとき *pMessage に受信したメッセージが書き込まれている
    */
    bool TryReceive(Message* pMessage) NN_NOEXCEPT;

    /**
        @brief メッセージの受信を試行する

        @return 受信できるメッセージがあれば true を返し、そうでなければ false を返す

        @post true が返ったとき *pMessage に受信したメッセージが書き込まれている
        @post *pOutTick に送信時刻の tick が書き込まれている
    */
    bool TryReceive(Message* pMessage, os::Tick* pOutTick) NN_NOEXCEPT;

    /**
        @brief メッセージを受信する(ブロッキング)

        @post *pMessage に受信したメッセージが書き込まれている
    */
    void Receive(Message* pMessage) NN_NOEXCEPT;

    /**
        @brief メッセージを受信する(ブロッキング)

        @post *pMessage に受信したメッセージが書き込まれている
        @post *pOutTick に送信時刻の tick が書き込まれている
    */
    void Receive(Message* pMessage, os::Tick* pOutTick) NN_NOEXCEPT;

    /**
        @brief 受信できるメッセージがあるかどうかを取得する(spurious あり)

        @return 受信できるメッセージがあれば true を返し、そうでなければ false を返す
    */
    bool HasMessageToReceive() NN_NOEXCEPT;

    /**
        @brief 受信できるメッセージを待つ(spurious あり)
    */
    void WaitForMessageToReceive() NN_NOEXCEPT;

    /**
        @brief 受信メッセージを受け取れるかどうかの同期オブジェクトを紐づける
    */
    void AttachWaitHolder(os::MultiWaitHolderType* pWaitHolder) NN_NOEXCEPT;

private:

    IReceiver* m_pReceiver;
    os::SystemEventType m_Event;
    bool m_EventHandleManaged;

};

}}
