﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>

namespace nn { namespace ovln { namespace format {

// 以下のタグ番号は欠番のため割当てないでください。
// - 3000

/**
 * @brief 充電器挿抜を表すタグ番号です。
 */
const uint32_t PowerManagementChargerStateChangeDataTag = 3001;

/**
 * @brief 電池残量低下を表すタグ番号です。
 */
const uint32_t PowerManagementLowBatteryDataTag = 3002;

/**
 * @brief 電池状態変化の通知データです。
 */
struct BatteryStateChangeData
{
    int32_t batteryChargePercentage;
    int32_t chargerType;
};

/**
 * @brief BatteryStateChangeData の通知要因です。
 */
enum BatteryStateChangeCause
{
    BatteryStateChangeCause_ChargerStateChanged, //!< 充電器の挿抜を検出しました。
    BatteryStateChangeCause_LowBattery,          //!< 電池残量が低下しています。
};

/**
 * @brief 充電器の種別です。
 */
enum ChargerType
{
    ChargerType_Unconnected,  //!< 充電器が接続されていません。
    ChargerType_EnoughPower,  //!< 充電可能で、本体の動作状況にかかわらず供給電力が十分な充電器が接続されています。
    ChargerType_LowPower,     //!< 充電可能ですが、本体の動作状況によっては供給電力が十分でない充電器が接続されています。
    ChargerType_NotSupported, //!< 充電不可能な充電器が接続されています。
};

/**
 * @brief      電池残量を取得します。
 * @param[in]  data 電池状態変化の通知データ
 * @return     電池残量 [%]
 * @details    電池状態変化の通知データから電池残量を 1 % から 100 % までの 1 % 刻みで取得します。
 */
inline int GetBatteryChargePercentage(
    const BatteryStateChangeData& data) NN_NOEXCEPT
{
    return static_cast<int>(data.batteryChargePercentage);
}

/**
 * @brief      本体に接続されている充電器の種別を取得します。
 * @param[in]  data 電池状態変化の通知データ
 * @return     本体に接続されている充電器の種別
 * @details    電池状態変化の通知データから本体に接続されている充電器の種別を取得します。
 *             本体に充電器が接続されていない場合には ChargerType_Unconnected が取得できます。
 */
inline ChargerType GetChargerType(
    const BatteryStateChangeData& data) NN_NOEXCEPT
{
    return static_cast<ChargerType>(data.chargerType);
}

}}}
