﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   Transfer memory 機能に関する API の宣言
 */

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/os/os_Types.h>
#include <nn/os/os_NativeHandleTypes.h>
#include <nn/os/os_TransferMemoryTypes.h>


namespace nn { namespace os {

//! @name Transfer memory 関連 API
//! @{

//--------------------------------------------------------------------------
/**
 * @brief   Transfer memory を作成します。
 *
 * @param[out] pOutTransferMemory  TransferMemoryType オブジェクトへのポインタ
 * @param[in]  addr                Transfer memory にする領域の先頭アドレス
 * @param[in]  size                Transfer memory にする領域のサイズ
 * @param[in]  permission          呼び出し元に許可するメモリアクセス権
 *
 * @retresult
 *   @handleresult{nn::os::ResultMaxHandle}
 *   @handleresult{nn::os::ResultOutOfTransferMemory}
 * @endretresult
 *
 * @pre
 *  - pOutTransferMemory が未初期化状態もしくはデタッチ状態である
 *  - addr が nn::os::MemoryPageSize の整数倍である
 *  - size が nn::os::MemoryPageSize の整数倍である
 *  - size > 0
 *  - addr と size で指定されるメモリ領域は有効な領域である
 *
 * @post
 *  - pOutTransferMemory は生成済み状態である
 *
 * @details
 *  与えられた addr, size を元に Transfer memory を生成し、
 *  そのハンドルをシステムから取得します。
 *  また、このハンドルを TransferMemoryType オブジェクトに関連付けます。
 *
 *  この段階で、Transfer memory にした領域へのアクセス権は、
 *  permission で指定した通りに変更されます。
 *  アクセス権は、Transfer memory ハンドルをシステムに返却する際、元に戻されます。
 *
 *  なお、TransferMemoryType オブジェクトは、オブジェクトの状態とは別に、
 *  bool 型のハンドル管理フラグを保持しています。ハンドル管理フラグは、
 *  DestroyTransferMemory() 時にハンドルを自動的にシステムに返却するか否かを
 *  示しており、本 API では必ず true に設定されます。
 *
 *  本 API の動作中は、対象 TransferMemoryType オブジェクトに対する操作を
 *  行なわないで下さい。
 *
 *
 * @platformbegin{Windows}
 *  - Windows 環境では、addr と size が指す領域とは異なる領域が Transfer memory として提供されます。
 *
 * @platformend
 *
 */
nn::Result CreateTransferMemory(TransferMemoryType* pOutTransferMemory, void* addr, size_t size, MemoryPermission permission) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 * @brief   既存の Transfer memory ハンドルを TransferMemoryType オブジェクトに関連付けます
 *
 * @param[out] pOutTransferMemory  TransferMemoryType オブジェクトへのポインタ
 * @param[in]  size            Transfer memory のサイズ
 * @param[in]  handle          Transfer memory のハンドル
 * @param[in]  managed         Transfer memory のハンドル管理フラグ
 *
 * @pre
 *  - pOutTransferMemory が未初期化状態もしくはデタッチ状態である
 *  - handle が有効なハンドル値である
 *  - handle が指す Transfer memory のサイズが size と等しい
 *
 * @post
 *  - pOutTransferMemory は生成済み状態である
 *
 * @details
 *  指定された Transfer memory ハンドルを使って TransferMemoryType オブジェクトを初期化します。
 *  指定される size は、handle が指す Transfer memory 自身のサイズと一致していなければなりません。
 *
 *  AttachTransferMemory() で初期化した TransferMemoryType オブジェクトは、
 *  DestroyTransferMemory() にて破棄することが出来ます。
 *
 *  また、 DestroyTransferMemory() 時に自動的にハンドルをシステムに
 *  返却するかどうかを managed で指定しておきます。@n
 *  managed に true を指定すると DestroyTransferMemory() 時にハンドルを
 *  システムに返却し、false を指定すると DestroyTransferMemory() 時にハンドルを
 *  返却しません。後者の場合、このハンドルは引き続き有効となります。
 *
 *  本 API の動作中は、対象 TransferMemoryType オブジェクトに対する操作を
 *  行なわないで下さい。
 *
 */
void AttachTransferMemory(TransferMemoryType* pOutTransferMemory, size_t size, NativeHandle handle, bool managed) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 * @brief   TransferMemoryType オブジェクトから Transfer memory ハンドルを切り離します
 *
 * @param[in] transferMemory  TransferMemoryType オブジェクトへのポインタ
 *
 * @return  Transfer memory のハンドルを返します。
 *
 * @pre
 *  - transferMemory が生成済み状態である
 *
 * @post
 *  - transferMemory はデタッチ状態である
 *
 * @details
 *  TransferMemoryType オブジェクトから Transfer memory ハンドルを切り離します。
 *  以後、対象の transferMemory は有効な Transfer memory ハンドルを持たなくなります。
 *  切り離された Transfer memory ハンドルは返り値として返されます。
 *  また、本オブジェクトのハンドル管理フラグは false になります。
 *
 *  DetachTransferMemory() によって Transfer memory ハンドルを切り離した場合でも、
 *  TransferMemoryType オブジェクトは DestroyTransferMemory() にて破棄してください。
 *
 *  本 API の動作中は、対象 TransferMemoryType オブジェクトに対する操作を
 *  行なわないで下さい。
 *
 */
NativeHandle DetachTransferMemory(TransferMemoryType* transferMemory) NN_NOEXCEPT;

//--------------------------------------------------------------------------
/**
 * @brief   TransferMemoryType オブジェクトを破棄します。
 *
 * @param[in] transferMemory  TransferMemoryType オブジェクトへのポインタ
 *
 * @pre
 *  - transferMemory は生成済み状態、マップ状態、デタッチ状態のいずれかである
 *
 * @post
 *  - transferMemory は未初期化状態である
 *
 * @details
 *  TransferMemoryType オブジェクトを破棄します。
 *  同時に、ハンドル管理フラグが true ならば、Transfer memory ハンドルを
 *  システムに返却します。false の場合はハンドルは引き続き有効なままです。
 *
 *  TransferMemory がマップ状態だった場合には、事前にアンマップされます。
 *
 *  本 API の動作中は、対象 TransferMemoryType オブジェクトに対する操作を
 *  行なわないで下さい。
 *
 */
void DestroyTransferMemory(TransferMemoryType* transferMemory) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 * @brief   Transfer memory をマップします。
 *
 * @param[out] pOutAddress     Transfer memory をマップしたアドレス
 * @param[in]  transferMemory  TransferMemoryType オブジェクトへのポインタ
 * @param[in]  ownerPermission Transfer memory を生成する際に満たすべきアクセス権
 *
 * @retresult
 *   @handleresult{nn::os::ResultInvalidHandle}
 *   @handleresult{nn::os::ResultInvalidTransferMemoryState}
 *   @handleresult{nn::os::ResultInvalidTransferMemorySize}
 *   @handleresult{nn::os::ResultOutOfAddressSpace}
 * @endretresult
 *
 * @pre
 *  - transferMemory は生成済み状態である
 *
 * @post
 *  - transferMemory はマップ状態である
 *
 * @details
 *  Transfer memory を空きアドレス空間にマップします。
 *  正しくマップできた場合、マップしたアドレスを pOutAddress に代入して返します。
 *  マップできなかった場合、pOutAddress の値は不定です。
 *
 *  Transfer memory を CreateTransferMemory() で生成する際に
 *  指定したアクセス権が ownerPermission に合致しない場合、マップすることはできません。
 *
 *  既にマップ済みの Transfer memory を再マップするには、
 *  一度アンマップする必要があります。
 *
 */
nn::Result MapTransferMemory(void** pOutAddress, TransferMemoryType* transferMemory, MemoryPermission ownerPermission) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 * @brief   Transfer memory をアンマップします。
 *
 * @param[in] transferMemory  TransferMemoryType オブジェクトへのポインタ
 *
 * @pre
 *  - transferMemory は生成済み状態もしくはマップ状態である
 *
 * @post
 *  - transferMemory は生成済み状態である
 *
 * @details
 *  MapTransferMemory() でマップした Transfer memory をアンマップします。
 *  マップ状態でない場合には何も行いません。
 *
 */
void UnmapTransferMemory(TransferMemoryType* transferMemory) NN_NOEXCEPT;


//! @}

}} // namespace nn::os

