﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   コンパイラのスレッドローカル実装用 API の宣言
 */

#pragma once

#include <nn/nn_Common.h>

//! @name   コンパイラのスレッドローカル実装用
//! @{

//--------------------------------------------------------------------------
/**
 * @brief コンパイラのスレッドローカル実装用のメモリアロケータを呼び出します。
 *
 * @param[in] size  確保したいサイズ
 *
 * @return  確保したメモリ領域の先頭アドレスが返ります。
 *
 * @pre
 *  - SetMemoryAllocatorForThreadLocal() でメモリアロケータを登録済みである
 *
 * @details
 *  nn::os::SetMemoryAllocatorForThreadLocal() で登録されたメモリアロケータ
 *  を使って、指定された size のメモリを獲得し、その先頭アドレスを返します。
 *  獲得されたメモリは std::max_align_t でアライメントされていることが
 *  保障されます。
 *
 *  本アロケータで獲得したメモリは、nnosFreeMemoryForThreadLocal() を使って
 *  返却して下さい。
 *
 *  本関数は C リンケージです。@n
 *  また、本関数はコンパイラが生成するコード、もしくは標準 C/C++ ランタイム
 *  ライブラリ内からのみ使用することが出来ます。
 *
 *  nn::os::SetMemoryAllocatorForThreadLocal() でアロケータが登録されて
 *  いない状態で本関数が呼ばれた場合は内部のアサートに失敗します。
 *
 */
extern "C" void* nnosAllocateMemoryForThreadLocal(size_t size);


//--------------------------------------------------------------------------
/**
 * @brief コンパイラのスレッドローカル実装用のメモリデアロケータを呼び出します。
 *
 * @param[in] address   返却したいメモリ領域の先頭アドレス
 * @param[in] size      返却したいサイズ
 *
 * @pre
 *  - SetMemoryAllocatorForThreadLocal() でメモリデアロケータを登録済みである
 *  - address が nnosAllocateMemoryForThreadLocal() で獲得したアドレスである
 *
 * @details
 *  nn::os::SetMemoryAllocatorForThreadLocal() で登録されたメモリデアロケータ
 *  を使って、指定された address と size が指すメモリ領域を返却します。
 *  指定される address は、nnosAllocateMemoryForThreadLocal() で獲得した
 *  メモリでなければなりません。
 *
 *  本関数は C リンケージです。@n
 *  また、本関数はコンパイラが生成するコード、もしくは標準 C/C++ ランタイム
 *  ライブラリ内からのみ使用することが出来ます。
 *
 *  nn::os::SetMemoryAllocatorForThreadLocal() でデアロケータが登録されて
 *  いない状態で本関数が呼ばれた場合は内部のアサートに失敗します。
 *
 */
extern "C" void nnosFreeMemoryForThreadLocal(void* address, size_t size);


//--------------------------------------------------------------------------
/**
 * @brief   プログラムで使用され得るモジュールの最大数を取得します。
 *
 * @return  使用され得るモジュールの最大数が返ります。
 *
 * @details
 *  各プログラムでロードされ得るモジュールの最大数を返します。
 *  モジュールには、静的モジュールと動的モジュールが含まれます。
 *
 *  本関数は C リンケージです。@n
 *  また、本関数はコンパイラが生成するコード、もしくは標準 C/C++ ランタイム
 *  ライブラリ内からのみ使用することが出来ます。
 *
 */
extern "C" int nnosGetModuleCountMax(void);


namespace nn { namespace os {

//--------------------------------------------------------------------------
/**
 * @brief   Test whether SetMemoryAllocatorForThreadLocal() has been called to initialize TLS allocator
 *
 * @return  true     The TLS allocator and deallocator are ready to use
 * @return  false    The TLS allocator and deallocator were not called in nninitStartup() and can not be used
 *
 * @details
 *  MemoryAllocatorForThreadLocalInitialized should only be called by libc.a to test that nninitStartup()
 *  called SetMemoryAllocatorForThreadLocal().  It is necessary to call SetMemoryAllocatorForThreadLocal()
 *  because TLS allocations use the allocator and deallocator set by a call to SetMemoryAllocatorForThreadLocal()
 *  and it must be called in nninitStartup() because TLS must be initialized before static constructors
 *  are called.
 *
 */
bool MemoryAllocatorForThreadLocalInitialized(void);

}}  // namespace nn::os

//! @}

