﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   SDK 専用の省メモリな条件変数に関する公開ヘッダファイル
 */

#pragma once

#include <nn/nn_Macro.h>
#include <nn/nn_TimeSpan.h>
#include <nn/os/detail/os_MacroImpl.h>
#include <nn/os/os_SdkMutex.h>
#include <nn/os/detail/os_InternalConditionVariable.h>

namespace nn { namespace os {

//--------------------------------------------------------------------------
/**
 * @brief   SDK 専用の省メモリな条件変数を扱うための構造体です。
 *
 * @details
 *  オブジェクトを明示的に初期化するには Initialize() を発行して下さい。
 *  使用後はオブジェクトをそのまま破棄できます。
 *
 *  また、以下のようにマクロを使って静的にオブジェクトを初期化できます。
 *
 *  @code
 *      static nn::os::SdkConditionVariableType  cond = NN_OS_SDK_CONDITION_VARIABLE_INITIALIZER();
 *  @endcode
 *
 */
struct SdkConditionVariableType
{
    union
    {
        // 静的初期化用イメージを定義するための配列
        int32_t _conditionImage[sizeof(detail::InternalConditionVariableStorage) / sizeof(int32_t)];
        detail::InternalConditionVariableStorage _condition;
    };

    // 関数
    void Initialize() NN_NOEXCEPT
    {
        Get(this->_condition).Initialize();
    }

    void Wait(SdkMutexType& mutex) NN_NOEXCEPT;
    void Wait(SdkRecursiveMutexType& mutex) NN_NOEXCEPT;
    bool TimedWait(SdkMutexType& mutex, TimeSpan timeout) NN_NOEXCEPT;
    bool TimedWait(SdkRecursiveMutexType& mutex, TimeSpan timeout) NN_NOEXCEPT;

    void Signal() NN_NOEXCEPT
    {
        Get(this->_condition).Signal();
    }

    void Broadcast() NN_NOEXCEPT
    {
        Get(this->_condition).Broadcast();
    }
};

NN_OS_DETAIL_STATIC_ASSERT_TRIVIAL(SdkConditionVariableType);

//--------------------------------------------------------------------------
/**
 * @brief   SDK 専用の省メモリな条件変数を扱うためのクラスです。
 *
 * @details
 *  オブジェクトはコンストラクタで初期化されます。
 *
 *  なお、再帰ロックカウンタが 2 以上の mutex を Wait() や TimedWait() に
 *  渡すと事前条件違反となり、内部でアボートします。
 *
 */
class SdkConditionVariable
{
    SdkConditionVariableType m_SdkCond;

public:
    SdkConditionVariable() NN_NOEXCEPT
    {
        m_SdkCond.Initialize();
    }
    void Wait(SdkMutex& mutex) NN_NOEXCEPT
    {
        m_SdkCond.Wait( mutex.m_SdkMutex );
    }
    void Wait(SdkRecursiveMutex& mutex) NN_NOEXCEPT
    {
        m_SdkCond.Wait( mutex.m_SdkRecursiveMutex );
    }
    bool TimedWait(SdkMutex& mutex, TimeSpan timeout) NN_NOEXCEPT
    {
        return m_SdkCond.TimedWait( mutex.m_SdkMutex, timeout );
    }
    bool TimedWait(SdkRecursiveMutex& mutex, TimeSpan timeout) NN_NOEXCEPT
    {
        return m_SdkCond.TimedWait( mutex.m_SdkRecursiveMutex, timeout );
    }
    void Signal() NN_NOEXCEPT
    {
        m_SdkCond.Signal();
    }
    void Broadcast() NN_NOEXCEPT
    {
        m_SdkCond.Broadcast();
    }
};

//--------------------------------------------------------------------------
/**
 * @brief   nn::os::SdkConditionVariableType オブジェクトを静的に初期化します。
 *
 * @details
 *  nn::os::SdkConditionVariableType オブジェクトを静的に初期化するマクロです。
 *
 *  使用例：
 *      @code
 *      nn::os::SdkConditionVariableType g_StaticCondVar = NN_OS_SDK_CONDITION_VARIABLE_INITIALIZER();
 *      @endcode
 *
 */
#define NN_OS_SDK_CONDITION_VARIABLE_INITIALIZER() { }

//--------------------------------------------------------------------------

}} // namespace nn::os

