﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file nsoc_user.h
 * NSOCモジュール 一般アプリケーション開発者用定義
 */

#ifndef NN_NET_COMPATIBLE_NSOC_NSOC_USER_H_
#define NN_NET_COMPATIBLE_NSOC_NSOC_USER_H_

#include <nn/net/nos/nos.h>	/* for NOSxxxx */

#ifdef  __cplusplus
extern "C" {
#endif

//#undef NSOC_IP6_ENABLE     /**< IPv6を使用 */


/**
 * @brief プロトコルファミリ表す列挙型です。
 */
enum NSOCProtocolFamily {
    NSOC_PF_UNSPEC  = 0,           /**< Unspecified */
    NSOC_PF_INET    = 2,           /**< ARPA Internet protocols */
    NSOC_PF_INET6   = 23           /**< ARPA Internet protocols version 6 */
};

/**
 * @brief アドレスファミリを表す列挙型です。
 */
enum NSOCAddressFamily {
    NSOC_AF_UNSPEC  = NSOC_PF_UNSPEC,   /**< Unspecified */
    NSOC_AF_INET    = NSOC_PF_INET,     /**< ARPA Internet protocols */
    NSOC_AF_INET6   = NSOC_PF_INET6     /**< ARPA Internet protocols version 6 */
};

/**
 * @brief ソケット種別表す列挙型です。
 */
enum NSOCSocketType {
    NSOC_SOCK_STREAM    = 1,           /**< ストリームソケット */
    NSOC_SOCK_DGRAM     = 2            /**< データグラムソケット */
};

/**
 * @brief @ref NSOC_RecvFrom や @ref NSOC_SendTo などに与えるフラグです。
 */
enum NSOCMessageFlag {
    NSOC_MSG_OOB        = 0x01,        /**< 対域外データの送受信を行います。 */
    NSOC_MSG_PEEK       = 0x02,        /**< データ読み出すだけで、取り出さずそのまま残します。 */
    NSOC_MSG_DONTWAIT   = 0x04         /**< 非同期操作を行います。 */
};

/**
 * @brief @ref NSOC_Poll に与えるフラグです。
 */
enum NSOCPollType {
    NSOC_POLLRDNORM     = 0x0001,     /**< 通常のデータを読み出し可能な状態。 SOCListen 関数により受付を開始したソケットが接続要求を受けた状態を含みます。 */
    NSOC_POLLRDBAND     = 0x0002,     /**< 優先帯域のデータを読み出し可能な状態。 TCP の緊急モードのデータを読み出し可能な状態です。 */
    NSOC_POLLPRI        = 0x0004,     /**< 緊急のデータを読み出し可能な状態。この状態になり得るプロトコルをサポートしていません。 */
    NSOC_POLLWRNORM     = 0x0008,     /**< 通常のデータを書き込み可能な状態です。 */
    NSOC_POLLWRBAND     = 0x0010,     /**< 優先帯域のデータを書き込み可能な状態。 TCP の緊急モードのデータを書き込み可能な状態です。 */
    NSOC_POLLERR        = 0x0020,     /**< 何らかの異常を検知した状態。 (revents のみ) */
    NSOC_POLLHUP        = 0x0040,     /**< ハングアップした状態。 (revents のみ) */
    NSOC_POLLNVAL       = 0x0080,     /**< 不正な処理が行われた状態。(revents のみ) */
    NSOC_POLLIN         = (NSOC_POLLRDNORM | NSOC_POLLRDBAND), /**< データを読み出し可能な状態。 NSOC_POLLRDNORM と NSOC_POLLRDBAND の論理和と同じです。 */
    NSOC_POLLOUT        = NSOC_POLLWRNORM /**< データを書き込み可能な状態。NSOC_POLLWRNORM と同じです。 */
};

/**
 * @brief @ref NSOC_GetSockOpt(), @ref NSOC_SetSockOpt() などに与えるオプションレベルです。
 */
enum NSOCSocketLevel {
    NSOC_SOL_SOCKET     = 0xffff,      /**< ソケット */
    NSOC_SOL_CONFIG     = 0xfffe,      /**< インスタンス */
    NSOC_SOL_IP         = 0,           /**< IP */
    NSOC_SOL_ICMP       = 1,           /**< ICMP */
    NSOC_SOL_TCP        = 6,           /**< TCP */
    NSOC_SOL_UDP        = 17,          /**< UDP */
    NSOC_SOL_IP6        = 41           /**< IPv6 */
};

/**
 * @brief @ref NSOC_GetSockOpt(), @ref NSOC_SetSockOpt() に与えるオプション番号です。
 */
enum NSOCSocketOptionType {
    NSOC_IP_TOS             = 0x00000007, /**< IPヘッダのTOSフィールド値(int) */
    NSOC_IP_TTL             = 0x00000008, /**< IPヘッダのTTLフィールド値(int) */
    NSOC_IP_MULTICAST_LOOP  = 0x00000009, /**< マルチキャストパケットをループさせる(unsigned char) */
    NSOC_IP_MULTICAST_TTL   = 0x0000000a, /**< マルチキャストパケットのTTLフィールド値(unsigned char) */
    NSOC_IP_ADD_MEMBERSHIP  = 0x0000000b, /**< マルチキャストグループ追加(NSOCIPMreq) */
    NSOC_IP_DROP_MEMBERSHIP = 0x0000000c, /**< マルチキャストグループ削除(NSOCIPMreq) */
    NSOC_SO_REUSEADDR       = 0x00000004, /**< ソケットの再利用を許可する(BOOL) */
    NSOC_SO_LINGER          = 0x00000080, /**< TCP切断完了を待つ(NSOCLinger) */
    NSOC_SO_OOBINLINE       = 0x00000100, /**< 緊急データをオンラインデータとして扱う(BOOL) */
    NSOC_SO_SNDBUF          = 0x00001001, /**< TCP送信バッファサイズ(int) */
    NSOC_SO_RCVBUF          = 0x00001002, /**< TCP受信バッファサイズ(int) */
    NSOC_SO_SNDLOWAT        = 0x00001003, /**< TCP送信バッファの利用可能下限値(int) */
    NSOC_SO_RCVLOWAT        = 0x00001004, /**< TCP受信バッファの利用可能下限値(int) */
    NSOC_SO_TYPE            = 0x00001008, /**< ソケット種別(int) */
    NSOC_SO_ERROR           = 0x00001009, /**< ソケットにセットされているエラー(int) */
    NSOC_TCP_NODELAY        = 0x00002001, /**< 遅延ACKを使用しない(BOOL) */
    NSOC_TCP_MAXSEG         = 0x00002002, /**< TCPのMSS値(int) */
    NSOC_TCP_STDURG         = 0x00002003, /**< TCPのurgentポインタをRFC仕様にする(BOOL) */
    NSOC_TCP_R2             = 0x00002004, /**< TCPのR2タイマ値(int) */
    NSOC_TCP_R2_SYN         = 0x00002005  /**< TCPのSYN用R2タイマ値(int) */
#ifdef NSOC_IP6_ENABLE
    ,
    NSOC_IP6_MULTICAST_LOOP = 0x00010009, /**< IPv6マルチキャストパケットをループさせる(unsigned char) */
    NSOC_IP6_HOPLIMIT       = 0x0001000a, /**< IPv6ヘッダのホップリミット値（unsigned char） */
    NSOC_IP6_ADD_MEMBERSHIP = 0x0001000b, /**< IPv6マルチキャストグループ追加(NSOCIP6Mreq) */
    NSOC_IP6_DROP_MEMBERSHIP= 0x0001000c  /**< IPv6マルチキャストグループ削除(NSOCIP6Mreq) */
#endif
};

/**
 * @brief @ref NSOC_Shutdown に与えるフラグです。
 */
enum NSOCShutdownType {
    NSOC_SHUT_RD        = 0,           /**< 受信 */
    NSOC_SHUT_WR        = 1,           /**< 送信 */
    NSOC_SHUT_RDWR      = 2            /**< 送受信 */
};

/**
 * @brief @ref NSOC_Fcntl に与えるフラグです。
 */
enum NSOCFctrlFlag {
    NSOC_F_GETFL        = 3,            /**< ソケットのフラグ値取得 */
    NSOC_F_SETFL        = 4             /**< ソケットのフラグ値セット */
};

/**
 * @brief @ref NSOC_Fcntl に与えるモード値です。
 */
enum NSOCFctrlOperation {
    NSOC_O_NONBLOCK     = 0x04          /**< APIをノンブロッキングモードに変更 */
};

/**
 * @brief @ref NSOC_GetAddrInfo に与えるフラグです。
 */
enum NSOCAddrInfoType {
    NSOC_AI_PASSIVE     = 0x01,         /**< nodeNameがNULLのとき、0.0.0.0が設定される */
    NSOC_AI_CANONNAME   = 0x02,         /**< nodeNameに対応する正規名の取得 */
    NSOC_AI_NUMERICHOST = 0x04,         /**< 数値表現のホストアドレス文字列をnodeNameに指定 */
    NSOC_AI_NUMERICSERV = 0x08          /**< 数値表現のポート番号文字列をservNameに指定 */
};

/**
 * @brief @ref NSOC_GetNameInfo に与えるフラグです。
 */
enum NSOCNameInfoType {
    NSOC_NI_NOFQDN      = 0x01,         /**< 完全修飾ドメイン名(FQDN)のうちノード名の部分だけを取得 */
    NSOC_NI_NUMERICHOST = 0x02,         /**< アドレスを数値記法で格納 */
    NSOC_NI_NAMEREQD    = 0x04,         /**< ホスト名の問い合わせに失敗した場合にエラー */
    NSOC_NI_NUMERICSERV = 0x08,         /**< ポート番号を数値記法で格納 */
    NSOC_NI_DGRAM       = 0x10          /**< UDPでの問い合わせ */
};

/**
 * @brief @ref NSOC_GetAddrInfo, @ref NSOC_GetNameInfo が返すエラー値です。
 */
enum NSOCAddrInfoError {
    NSOC_EAI_BADFLAGS   = -301,         /**< フラグが不正 */
    NSOC_EAI_FAIL       = -302,         /**< 修復不可能なエラー */
    NSOC_EAI_FAMILY     = -303,         /**< 不明なアドレスファミリ */
    NSOC_EAI_MEMORY     = -304,         /**< メモリ割り当てに失敗 */
    NSOC_EAI_NONAME     = -305,         /**< 名前を解決できません */
    NSOC_EAI_SOCKTYPE   = -307,         /**< ソケットタイプエラー */
    NSOC_EAI_SYSTEM     = -308          /**< システムエラー */
};

/**
 * @ref NSOC_GetRRSetByName が返すエラー値です。
 */
enum NSOCRRsetError {
    NSOC_ERRSET_NONAME  = -350,
    NSOC_ERRSET_NODATA  = -351,
    NSOC_ERRSET_NOMEMORY= -352,         /**< メモリ不足 */
    NSOC_ERRSET_INVAL   = -353,         /**< 引数が不正 */
    NSOC_ERRSET_FAIL    = -354
};

/**
 * @brief ソケット API 共通で使用するエラー値です。説明のないエラーは内部で使用され、ソケット API が返すことはありません。
 */
enum NSOCSocketError {
    NSOC_E2BIG          = -1,
    NSOC_EACCES         = -2,     /**< 使用する権限がありません */
    NSOC_EADDRINUSE     = -3,     /**< アドレスがすでに使用中です */
    NSOC_EADDRNOTAVAIL  = -4,     /**< アドレスが使用できません */
    NSOC_EAFNOSUPPORT   = -5,     /**< サポートされていないアドレスファミリです */
    NSOC_EAGAIN         = -6,     /**< 要求した操作は封鎖しないと実行できません。(NSOC_EWOULDBLOCKと同様) */
    NSOC_EALREADY       = -7,
    NSOC_EBADF          = -8,     /**< 不正なソケット記述子です */
    NSOC_EBADMSG        = -9,
    NSOC_EBUSY          = -10,
    NSOC_ECANCELED      = -11,
    NSOC_ECHILD         = -12,
    NSOC_ECONNABORTED   = -13,    /**< 接続が中断されました */
    NSOC_ECONNREFUSED   = -14,    /**< 接続が拒否されました */
    NSOC_ECONNRESET     = -15,    /**< 接続がリセットされました */
    NSOC_EDEADLK        = -16,
    NSOC_EDESTADDRREQ   = -17,    /**< 終点アドレスが指定されていません */
    NSOC_EDOM           = -18,
    NSOC_EDQUOT         = -19,
    NSOC_EEXIST         = -20,
    NSOC_EFAULT         = -21,
    NSOC_EFBIG          = -22,
    NSOC_EHOSTUNREACH   = -23,
    NSOC_EIDRM          = -24,
    NSOC_EILSEQ         = -25,
    NSOC_EINPROGRESS    = -26,    /**< 接続がまだ完了していません */
    NSOC_EINTR          = -27,    /**< 中止されました */
    NSOC_EINVAL         = -28,    /**< 無効な処理 */
    NSOC_EIO            = -29,    /**< 入出力エラー */
    NSOC_EISCONN        = -30,    /**< ソケットがすでに接続されています */
    NSOC_EISDIR         = -31,
    NSOC_ELOOP          = -32,
    NSOC_EMFILE         = -33,    /**< ソケット記述子をこれ以上作れません */
    NSOC_EMLINK         = -34,
    NSOC_EMSGSIZE       = -35,    /**< 送信するにはサイズが大きすぎます */
    NSOC_EMULTIHOP      = -36,
    NSOC_ENAMETOOLONG   = -37,
    NSOC_ENETDOWN       = -38,    /**< 該当インスタンスがダウンしています */
    NSOC_ENETRESET      = -39,    /**< ソケットライブラリが初期化されていません */
    NSOC_ENETUNREACH    = -40,    /**< 到達できません */
    NSOC_ENFILE         = -41,
    NSOC_ENOBUFS        = -42,    /**< リソース不足 */
    NSOC_ENODATA        = -43,
    NSOC_ENODEV         = -44,
    NSOC_ENOENT         = -45,
    NSOC_ENOEXEC        = -46,
    NSOC_ENOLCK         = -47,
    NSOC_ENOLINK        = -48,
    NSOC_ENOMEM         = -49,    /**< メモリ不足 */
    NSOC_ENOMSG         = -50,
    NSOC_ENOPROTOOPT    = -51,    /**< サポートされていないオプション */
    NSOC_ENOSPC         = -52,
    NSOC_ENOSR          = -53,
    NSOC_ENOSTR         = -54,
    NSOC_ENOSYS         = -55,
    NSOC_ENOTCONN       = -56,    /**< 接続されていません */
    NSOC_ENOTDIR        = -57,
    NSOC_ENOTEMPTY      = -58,
    NSOC_ENOTSOCK       = -59,
    NSOC_ENOTSUP        = -60,
    NSOC_ENOTTY         = -61,
    NSOC_ENXIO          = -62,
    NSOC_EOPNOTSUPP     = -63,    /**< サポートされていない処理 */
    NSOC_EOVERFLOW      = -64,
    NSOC_EPERM          = -65,
    NSOC_EPIPE          = -66,
    NSOC_EPROTO         = -67,
    NSOC_EPROTONOSUPPORT= -68,    /**< サポートされていないプロトコル */
    NSOC_EPROTOTYPE     = -69,    /**< サポートされていないソケットタイプ */
    NSOC_ERANGE         = -70,
    NSOC_EROFS          = -71,
    NSOC_ESPIPE         = -72,
    NSOC_ESRCH          = -73,
    NSOC_ESTALE         = -74,
    NSOC_ETIME          = -75,
    NSOC_ETIMEDOUT      = -76,    /**< 時間切れ */
    NSOC_ETXTBSY        = -77,
    NSOC_EWOULDBLOCK    = NSOC_EAGAIN,   /**< 要求した操作は封鎖しないと実行できません。(NSOC_EAGAINと同様) */  /* Posix.1g */
    NSOC_EXDEV          = -78
};

/**
 * @ref NSOC_GetInstanceOpt(), @ref NSOC_SetInstanceOpt() に与えるオプション番号です。
 */
enum NSOCInstanceOptionType {
    NSOC_CONFIG_FILTER_INPUT                   =  0x1001,    /**< 受信パケットフィルタ関数 */
    NSOC_CONFIG_FILTER_OUTPUT                  =  0x1002,    /**< 送信パケットフィルタ関数 */
    NSOC_CONFIG_ERROR                          =  0x1003,    /**< インスタンスのエラー */
    NSOC_CONFIG_MAC_ADDRESS                    =  0x1004,    /**< 自MACアドレス */
    NSOC_CONFIG_LINK_STATE                     =  0x1005,    /**< インスタンスの起動状態 */
    NSOC_CONFIG_INTERFACE_STATISTICS           =  0x1006,    /**< インスタンスの統計情報 */
    NSOC_CONFIG_MUTE                           =  0x1007,    /**< パケット送受信停止 */
    NSOC_CONFIG_ARP_NUMBER                     =  0x3001,    /**< ARPキャッシュエントリ数 */
    NSOC_CONFIG_ARP_TABLE                      =  0x3002,    /**< ARPテーブル */
    NSOC_CONFIG_ARP_REFRESH                    =  0x3003,    /**< ARPテーブルのクリア */
    NSOC_CONFIG_ARP_ADD                        =  0x3004,    /**< ARPキャッシュエントリ登録 */
    NSOC_CONFIG_ARP_REMOVE                     =  0x3005,    /**< ARPキャッシュエントリ削除 */
    NSOC_CONFIG_IP_STATISTICS                  =  0x4001,    /**< IPの統計情報 */
    NSOC_CONFIG_IP_ADDR_NUMBER                 =  0x4002,    /**< IPアドレス数 */
    NSOC_CONFIG_IP_ADDR_TABLE                  =  0x4003,    /**< IPアドレステーブル */
    NSOC_CONFIG_IP_MTU                         =  0x4004,    /**< IPのMTU */
    NSOC_CONFIG_IP_ROUTING_NUMBER              =  0x4005,    /**< 経路情報エントリ数 */
    NSOC_CONFIG_IP_ROUTING_TABLE               =  0x4006,    /**< 経路情報テーブル */
    NSOC_CONFIG_IP_ADD_ROUTE                   =  0x4007,    /**< 経路情報エントリ登録 */
    NSOC_CONFIG_IP_REMOVE_ROUTE                =  0x4008,    /**< 経路情報エントリ削除 */
    NSOC_CONFIG_IP_ADDR                        =  0x400a,    /**< IPアドレス */
    NSOC_CONFIG_IP_GATEWAY                     =  0x400b,    /**< デフォルトゲートウエイアドレス */
    NSOC_CONFIG_IP_NETMASK                     =  0x400c,    /**< サブネットマスク */
    NSOC_CONFIG_IP_REFRESH_ROUTE               =  0x400d,    /**< 経路情報を最新状態に更新 */
    NSOC_CONFIG_ICMP_STATISTICS                =  0x5001,    /**< ICMPの統計情報 */
#ifdef NSOC_IP6_ENABLE
    NSOC_CONFIG_IP6_STATISTICS                 =  0x6001,    /**< IPv6の統計情報 */
    NSOC_CONFIG_IP6_MTU                        =  0x6002,    /**< IPv6リンクMTU */
    NSOC_CONFIG_IP6_PREFIX_NUMBER              =  0x6003,    /**< IPv6プレフィックスリストエントリ数 */
    NSOC_CONFIG_IP6_PREFIX_TABLE               =  0x6004,    /**< IPv6プレフィックスリスト */
    NSOC_CONFIG_IP6_ADD_ROUTE                  =  0x6005,    /**< IPv6経路情報エントリ登録 */
    NSOC_CONFIG_IP6_REMOVE_ROUTE               =  0x6006,    /**< IPv6経路情報エントリ削除 */
    NSOC_CONFIG_IP6_ADD_ADDRESS                =  0x6007,    /**< IPv6アドレス追加 */
    NSOC_CONFIG_IP6_REMOVE_ADDRESS             =  0x6008,    /**< IPv6アドレス削除 */
    NSOC_CONFIG_IP6_HOPLIMIT                   =  0x6009,    /**< IPv6アドレスホップリミット */
    NSOC_CONFIG_IP6_RETRANSMIT_TIMER           =  0x600a,    /**< IPv6アドレス重複検査用NS送信間隔 */
    NSOC_CONFIG_IP6_DUP_ADDR_DETECT_TRANSMITS  =  0x600b,    /**< IPv6アドレス重複検査用NS送信回数 */
    NSOC_CONFIG_IP6_INTERFACE_ID               =  0x600c,    /**< IPv6アドレスインタフェースID */
    NSOC_CONFIG_IP6_DESTINATION_CACHE_NUMBER   =  0x600d,    /**< IPv6宛先キャッシュリストエントリ数 */
    NSOC_CONFIG_IP6_DESTINATION_CACHE_TABLE    =  0x600e,    /**< IPv6宛先キャッシュリスト */
    NSOC_CONFIG_IP6_ADDR_NUMBER                =  0x600f,    /**< IPv6アドレスリストエントリ数 */
    NSOC_CONFIG_IP6_ADDR_TABLE                 =  0x6010,    /**< IPv6アドレスリスト */
    NSOC_CONFIG_IP6_ROUTE_NUMBER               =  0x6011,    /**< IPv6経路情報エントリ数 */
    NSOC_CONFIG_IP6_ROUTE_TABLE                =  0x6012,    /**< IPv6経路情報テーブル */
    NSOC_CONFIG_IP6_NEIGHBOR_CACHE_NUMBER      =  0x6013,    /**< IPv6近隣キャッシュリストエントリ数 */
    NSOC_CONFIG_IP6_NEIGHBOR_CACHE_TABLE       =  0x6014,    /**< IPv6近隣キャッシュリスト */
    NSOC_CONFIG_IP6_DEFAULT_ROUTER_NUMBER      =  0x6015,    /**< IPv6デフォルトルータリストエントリ数 */
    NSOC_CONFIG_IP6_DEFAULT_ROUTER_TABLE       =  0x6016,    /**< IPv6デフォルトルータリスト */
    NSOC_CONFIG_IP6_ADD_PREFIX                 =  0x6017,    /**< IPv6プレフィックスエントリ登録 */
    NSOC_CONFIG_IP6_REMOVE_PREFIX              =  0x6018,    /**< IPv6プレフィックスエントリ削除 */
    NSOC_CONFIG_ICMP6_STATISTICS               =  0x7001,    /**< ICMPv6の統計情報 */
#endif /* NSOC_IP6_ENABLE */
    NSOC_CONFIG_UDP_STATISTICS                 =  0x8001,    /**< UDPの統計情報 */
    NSOC_CONFIG_UDP_NUMBER                     =  0x8002,    /**< UDPソケット数 */
    NSOC_CONFIG_UDP_TABLE                      =  0x8003,    /**< UDPソケットテーブル */
    NSOC_CONFIG_TCP_STATISTICS                 =  0x9001,    /**< TCPの統計情報 */
    NSOC_CONFIG_TCP_NUMBER                     =  0x9002,    /**< TCPソケット数 */
    NSOC_CONFIG_TCP_TABLE                      =  0x9003,    /**< TCPソケットテーブル */
    NSOC_CONFIG_TCP_MSL                        =  0x9006,    /**< TCPのMSS */
    NSOC_CONFIG_TCP_RTO_MIN                    =  0x9007,    /**< TCPの最小再送時間 */
    NSOC_CONFIG_TCP_RTO_MAX                    =  0x9008,    /**< TCPの最大再送時間 */
    NSOC_CONFIG_TCP_RTO_DEFAULT                =  0x9009,    /**< TCPのデフォルト再送時間 */
    NSOC_CONFIG_IGMP_MODE                      =  0xA001,    /**< IGMPのバージョン */
    NSOC_CONFIG_DNS_DOMAIN_NAME                =  0xB001,    /**< ドメイン名 */
    NSOC_CONFIG_DNS_SERVER_NUMBER              =  0xB002,    /**< DNSサーバ数 */
    NSOC_CONFIG_DNS_SERVER_TABLE               =  0xB003,    /**< DNSサーバテーブル */
    NSOC_CONFIG_DHCP_REMAINING_LEASE_TIME      =  0xC001     /**< DHCPの残りリース時間 */
};

/**
 * ソケットモジュールのステートを表す列挙型です。
 */
enum NSOCInstanceState {
    NSOC_STATE_INIT = 0,        /**< 初期状態 */
    NSOC_STATE_WAIT_LINKUP,     /**< 起動実行中 */
    NSOC_STATE_ACTIVE,          /**< 起動完了状態 */
    NSOC_STATE_MORIBUND         /**< 終了実行中 */
};

/*---------------------------------------------------------------------------*
 * Types/Declarations
 *---------------------------------------------------------------------------*/

struct NSOCInstance;
/**
 * 外部参照用 @ref NSOCInstance のポインタ型です。
 */
typedef NSOCInstance* NSOCInstancePtr;

/*
 * アドレス長
 */
#define NSOC_ETH_ALEN           6       /**< MACアドレス長 */
#define NSOC_IP_ALEN            4       /**< IPv4アドレス長 */
#define NSOC_IP6_ALEN           16      /**< IPv6アドレス長 */

/**
 * IPv4アドレスの文字列長
 * aaa.bbb.ccc.ddd
 *
 * IPv6アドレスの文字列長
 * RFC3493: 6.3. Address Conversion Functions
 * ffff:ffff:ffff:ffff:ffff:ffff:255.255.255.255
 */
#define NSOC_INET_ADDRSTRLEN    16
#define NSOC_INET6_ADDRSTRLEN   46

/**
 * @brief IPv4アドレス構造体です。
 */
#define NSOC_INADDR_ANY           ((u32) 0x00000000)  /**< 0.0.0.0 */

typedef struct NSOCInAddr
{
    u32         addr;   /**< ネットワークバイトオーダでのIPv4アドレス */
} NSOCInAddr;

/**
 * @brief IPv4ソケット構造体です。
 */
typedef struct NSOCSockAddrIn
{
    u8          len;            /**< ソケット構造体のサイズ */
    u8          family;         /**< アドレスファミリ */
    u16         port;           /**< ポート番号 */
    NSOCInAddr  addr;           /**< IPv4アドレス構造体 */
} NSOCSockAddrIn;

/**
 * IPv6アドレス構造体です。
 */
typedef struct NSOCIn6Addr
{
    u8         addr[NSOC_IP6_ALEN];    /**< IPv6アドレス */
} NSOCIn6Addr;

/**
 * IPv6ソケット構造体です。
 */
typedef struct NSOCSockAddrIn6
{
    u8          len;            /**< ソケット構造体のサイズ */
    u8          family;         /**< アドレスファミリ */
    u16         port;           /**< ポート番号 */
    NSOCIn6Addr addr;           /**< IPv6アドレス構造体 */
    u32         flowInfo;       /**< IPv6 flow information */
    u32         scopeId;        /**< set of interfaces for a scope */
} NSOCSockAddrIn6;

/**
 * @brief ソケット構造体です。
 */
typedef struct NSOCSockAddr
{
    u8          len;            /**< ソケット構造体のサイズ */
    u8          family;         /**< アドレスファミリ */
    u16         pad1;           /**< padding */
    u32         pad2[6];        /**< padding2 */
} NSOCSockAddr;

/**
 * @ref NSOCSockAddrIn と @ref NSOCSockAddrIn6を格納できるだけの大きさの構造体です。
 */
typedef struct NSOCSockAddrStorage
{
    u8          len;            /**< ソケット構造体のサイズ */
    u8          family;         /**< アドレスファミリ */
    u16         pad1;           /**< padding */
    u32         pad2[6];        /**< padding2 */
} NSOCSockAddrStorage;

/**
 * @brief @ref NSOC_GetSockOpt(), @ref NSOC_SetSockOpt()で NSOC_SO_LINGER を使用する時の構造体です。
 */
typedef struct NSOCLinger
{
    int         onoff;          /**< 1：linger有効、0：linger無効 */
    int         linger;         /**< linger時間（秒単位） */
} NSOCLinger;

/**
 * @brief @ref NSOC_GetSockOpt(), @ref NSOC_SetSockOpt()で NSOC_IP_ADD_MEMBERSHIP, NSOC_IP_DROP_MEMBERSHIP を使用する時の構造体です。
 */
typedef struct NSOCIPMreq
{
   NSOCInAddr     multiaddr;      /**< グループのIPアドレス */
   NSOCInAddr     interface;      /**< インターフェイスのIPアドレス */
} NSOCIPMreq;

/**
 * @ref NSOC_GetSockOpt(), @ref NSOC_SetSockOpt()で NSOC_IP6_ADD_MEMBERSHIP, NSOC_IP6_DROP_MEMBERSHIP を使用する時の構造体です。
 */
typedef struct NSOCIP6Mreq
{
   u32            scopeId;                  /**< set of interfaces for a scope */
   u8             maddr[NSOC_IP6_ALEN];     /**< グループのIPアドレス */
   u8             filterType;               /**< フィルタタイプ */
   u32            srcNum;                   /**< 送信元アドレス数 */
   u8             srcList[NSOC_IP6_ALEN];   /**< 送信元アドレス。2つ以上指定するときは続けてアドレスを入れる領域を用意すること */
} NSOCIP6Mreq;

enum NSOCIP6MreqType {
    NSOC_IP6MREQ_INCLUDE = 0,   /**< フィルタタイプ：INCLUDE */
    NSOC_IP6MREQ_EXCLUDE = 1    /**< フィルタタイプ：EXCLUDE */
};

/**
 * @brief @ref NSOC_Poll() を使用する時の構造体です。
 */
typedef struct NSOCPollFD
{
    int         fd;             /**< ソケット番号 */
    int         events;         /**< 取得したいイベント */
    int         revents;        /**< 実際に取得したイベントが返ります */
} NSOCPollFD;


typedef struct NSOCAddrInfo   NSOCAddrInfo;

/**
 * @brief @ref NSOC_GetAddrInfo() を使用する時の構造体です。
 */
struct NSOCAddrInfo
{
    int             flags;         /**< フラグ */
    int             family;        /**< ソケットのアドレスファミリ */
    int             sockType;      /**< ソケットタイプ */
    int             protocol;      /**< ソケットのプロトコル */
    unsigned        addrLen;       /**< ソケット構造体のサイズ */
    char*           canonName;     /**< 正規名 */
    void*           addr;          /**< ソケット構造体へのポインタ */
    NSOCAddrInfo*   next;          /**< 次のNSOCAddrInfoへのポインタ。リストの最後ではNULL */
};

typedef struct NSOCRDataInfo {
    u32 length;
    u8 *data;
} NSOCRDataInfo;

/**
 * @ref NSOC_GetRRSetByName() を使用する時の構造体です。
 */
typedef struct NSOCRRSetInfo {
    u16 rdClass;
    u16 rdType;
    u32 ttl;
    u32 nrDatas;
    char *name;
    NSOCRDataInfo *rDatas;
} NSOCRRSetInfo;

/**
 * @brief @ref NSOC_GetHostByAddr(), @ref NSOC_GetHostByName() を使用する時の構造体です。
 */
typedef struct NSOCHostEnt
{
    char*       name;           /**< ホストのオフィシャル名 */
    char**      aliases;        /**< エイリアスリストへのポインタ */
    s16         addrType;       /**< アドレスタイプ。NSOC_PF_INET */
    s16         length;         /**< ソケット構造体のサイズ */
    u8**        addrList;       /**< ソケット構造体リストへのポインタ */
} NSOCHostEnt;


/**
 * @brief DNSサーバアドレスの構造体です。
 */
typedef struct NSOCDNSAddr {
    int af;                 /**< アドレスファミリ。NSOC_AF_INETまたはNSOC_AF_INET6、使用しない場合は0 */
    u8 addr[NSOC_IP6_ALEN];  /**< IPv4またはIPv6アドレス */
} NSOCDNSAddr;

/**
 * NSOC_SetResolverIns()、NSOC_GetResolverIns() 使用時の構造体です。
 */
#define NSOC_DNSSERVER_NUM_MAX          4      /**< 保持できる最大のDNSサーバ数 */
#define NSOC_DOMAIN_NAME_LEN_MAX        255
typedef struct NSOCDNSServerInfo {
    NSOCDNSAddr entry[NSOC_DNSSERVER_NUM_MAX];         /**< DNSサーバアドレス構造体 */

    char domain[NSOC_DOMAIN_NAME_LEN_MAX + 1];        /**< ドメイン名 */

} NSOCDNSServerInfo;

#define NSOC_DNS_NAME_MAX               255 /**< DNSにおける名前の最大文字数*/
#define NSOC_MAXDNAME                   (NSOC_DNS_NAME_MAX + 1) /**< DNSにおける名前文字列の最大長(DNS名の最大文字数に'\0'の一文字分を加えた値)*/
#define NSOC_DNS_RESOLV_ADDRESS_MAX     24  /**< DNSの名前解決において、一つの名前に対して取得できるアドレスの最大数*/
#define NSOC_DNS_RESOLV_ALIASNAME_MAX   24  /**< DNSの名前解決において、一度の解決で取得できるエイリアス名(CNAMEレコードで指定される名称)の最大数*/

/*
 * well-known address 初期化用定義
 */
#define NSOC_IN6ADDR_UNSPECIFIED_INIT               {0}
#define NSOC_IN6ADDR_LOOPBACK_INIT                  {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1}
#define NSOC_IN6ADDR_LINKLOCAL_ALL_NODES_INIT       {0xff,0x02,0,0,0,0,0,0,0,0,0,0,0,0,0,1}
#define NSOC_IN6ADDR_LINKLOCAL_ALL_ROUTERS_INIT     {0xff,0x02,0,0,0,0,0,0,0,0,0,0,0,0,0,2}
#define NSOC_IN6ADDR_MULTICAST_SOLICITED_NODE_INIT  {0xff,0x02,0,0,0,0,0,0,0,0,0,0x01,0xff,0,0,0}
#define NSOC_IN6ADDR_LINKLOCAL_UNICAST_INIT         {0xfe,0x80,0,0,0,0,0,0,0,0,0,0,0,0,0,0}
#define NSOC_IN6ADDR_MULTICAST_ALL_ROUTERS_INIT     {0xff,0x02,0,0,0,0,0,0,0,0,0,0,0,0,0,16}

#ifdef SWITCH_SO_ENDIAN_BIG
// Big Endian environment
#define NSOC_NtoHl(netlong)        ((u32) (netlong))
#define NSOC_NtoHs(netshort)       ((u16) (netshort))
#define NSOC_HtoNl(hostlong)       ((u32) (hostlong))
#define NSOC_HtoNs(hostshort)      ((u16) (hostshort))

#else
// Little Endian environment
/**
 * 4byte長変数をネットワークバイトオーダからホストバイトオーダに変換
 */
#define NSOC_NtoHl(netlong)        ( (u32)( (((u32)(netlong)>>24)&0x000000FF) | (((u32)(netlong)>> 8)&0x0000FF00) | (((u32)(netlong)<< 8)&0x00FF0000) | (((u32)(netlong)<<24)&0xFF000000) ) )

/**
 * 2byte長変数をネットワークバイトオーダからホストバイトオーダに変換
 */
#define NSOC_NtoHs(netshort)       ( (u16)( (((u16)(netshort)>>8)&0x00FF) | (((u16)(netshort)<<8)&0xFF00) ) )

/**
 * 4byte長変数をホストオーダからネットワークバイトオーダに変換
 */
#define NSOC_HtoNl(hostlong)       ( (u32)( (((u32)(hostlong)>>24)&0x000000FF) | (((u32)(hostlong)>> 8)&0x0000FF00) | (((u32)(hostlong)<< 8)&0x00FF0000) | (((u32)(hostlong)<<24)&0xFF000000) ) )

/**
 * 2byte長変数をホストオーダからネットワークバイトオーダに変換
 */
#define NSOC_HtoNs(hostshort)      ( (u16)( (((u16)(hostshort)>>8)&0x00FF) | (((u16)(hostshort)<<8)&0xFF00) ) )

#endif

/**
 * NSOCモジュールのAPI
 */
/* nsoc_socket.c */
extern int NSOC_Socket(int af, int type, int protocol);
extern int NSOC_SocketEx(int af, int type, int protocol, void *userInfo);
extern int NSOC_Listen(int s, int backlog);
extern int NSOC_Accept(int s, void* sockAddr);
extern int NSOC_Bind(int s, const void* sockAddr);
extern int NSOC_Connect(int s, const void* sockAddr);
extern int NSOC_Read(int s, void* buf, int len);
extern int NSOC_Recv(int s, void* buf, int len, int flags);
extern int NSOC_RecvFrom(int s, void* buf, int len, int flags, void* sockFrom);
extern int NSOC_Write(int s, const void* buf, int len);
extern int NSOC_Send(int s, const void* buf, int len, int flags);
extern int NSOC_SendTo(int s, const void* buf, int len, int flags, const void* sockTo);
extern int NSOC_Close(int s);
extern int NSOC_Shutdown(int s, int how);
extern int NSOC_GetSockOpt(int s, int level, int optname, void* optval, int* optlen);
extern int NSOC_SetSockOpt(int s, int level, int optname,  const void* optval, int optlen);
extern int NSOC_Fcntl(int s, int cmd, ...);
extern int NSOC_Poll(NSOCPollFD fds[], unsigned nfds, s32 timeout);
extern int NSOC_SockAtMark(int s);
extern u32 NSOC_GetHostID(NSOCInstancePtr insP);
extern NSOCHostEnt* NSOC_GetHostByAddr(const void* addr, int len, int type);
extern NSOCHostEnt* NSOC_GetHostByName(const char* name);
extern int NSOC_GetSockName(int s, void* sockAddr);
extern int NSOC_GetPeerName(int s, void* sockAddr);
extern int NSOC_SetDNSInstance(NSOCInstancePtr insP);
extern int NSOC_SetResolverIns(NSOCInstancePtr insP, NSOCDNSServerInfo *dnsServer);
extern int NSOC_GetResolverIns(NSOCInstancePtr insP, NSOCDNSServerInfo *dnsServer);
extern int NSOC_SetResolver(const NSOCInAddr* dns1, const NSOCInAddr* dns2); /* deprecated */
extern int NSOC_GetResolver(NSOCInAddr* dns1, NSOCInAddr* dns2); /* deprecated */
extern int NSOC_SetSuffix(const char* domainName);
extern int NSOC_GetSuffix(char* domainName);
extern int NSOC_SetInstanceOpt(NSOCInstancePtr insP, int level, int optname, const void* optval, int optlen);
extern int NSOC_GetInstanceOpt(NSOCInstancePtr insP, int level, int optname, void* optval, int* optlen);
extern void NSOC_SetLastError(int error);
extern int NSOC_GetLastError(void);

/* nsoc_dns.c */
extern int NSOC_GetAddrInfo(const char* nodeName, const char* servName, const NSOCAddrInfo* hints, NSOCAddrInfo** res);
extern int NSOC_GetNameInfo(const void* sa, char* node, unsigned nodeLen, char* service, unsigned serviceLen, int flags);
extern void NSOC_FreeAddrInfo(NSOCAddrInfo* head);
extern void NSOC_FreeRRSet(NSOCRRSetInfo *rrset);
extern int NSOC_GetRRSetByName(const char *hostName, u16 rdClass, u16 rdType, u32 flags, NSOCRRSetInfo **res);
extern int NSOC_GetServiceLocation(const char* domainName, const char* servName, const NSOCAddrInfo* hints, NSOCAddrInfo** res);

/* nsoc_inet.c */
extern char* NSOC_IPAtoN(const char* dotted, u8* addr);
extern char* NSOC_IPNtoA(const u8* addr);
#ifdef NSOC_IP6_ENABLE
extern char *NSOC_IP6AtoN(const char *dotted, u8 *addr);
extern char *NSOC_IP6NtoA(const u8 *addr, char *ascii, u32 buflen);
extern char *NSOC_IP6NtoAs(const u8 *addr);
#endif
extern int NSOC_InetAtoN(const char* cp, NSOCInAddr* inp);
extern char* NSOC_InetNtoA(NSOCInAddr in);
extern int NSOC_InetPtoN(int af, const char* src, void* dst);
extern const char* NSOC_InetNtoP(int af, const void* src, char* dst, unsigned len);

/* nsoc_main.c */
extern uintptr_t NSOC_IP6GetScopeIdByInstance(NSOCInstancePtr insP);

/* nsoc_state.c */
extern s32 NSOC_GetStateIns(NSOCInstancePtr insP);
extern void NSOC_ChangeStateIns(NSOCInstancePtr insP, s32 state);


extern int NSOC_TST_GetNodeParam(int socket, NOSMutexId *mtxid, u8 *proto, u8 *flag, s32 *id);
extern int NSOC_TST_UDPGetObj(int socket, void*  getobj);
extern int NSOC_TST_IPGetInfo(int socket, void* ipinfo);
extern int NSOC_TST_TCPGetObj(int socket, void*  getobj);

#ifdef  __cplusplus
}
#endif

/* NN_NET_COMPATIBLE_NSOC_NSOC_USER_H_ */
#endif
