﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/util/util_Uuid.h>
#include <nn/os/os_Mutex.h>
#include <nn/fs.h>
#include <nn/ncm/ncm_PathString.h>
#include <nn/ncm/ncm_MakePlaceHolderPathFunction.h>


namespace nn { namespace ncm {

class PlaceHolderAccessor
{
    NN_DISALLOW_COPY(PlaceHolderAccessor);
    NN_DISALLOW_MOVE(PlaceHolderAccessor);

public:
    static const int CacheCount = 2;

    PlaceHolderAccessor() NN_NOEXCEPT : m_CacheMutex(false), m_DelayFlush(false)
    {
        m_CurrentCounter = 0;
        for (auto& cache : m_Caches)
        {
            cache.id.uuid = util::InvalidUuid;
        }
    }

    ~PlaceHolderAccessor() NN_NOEXCEPT { InvalidateAll(); }

    static void MakeBaseDirectoryPath(PathString* outPath, const char* rootPath) NN_NOEXCEPT;
    static Result GetPlaceHolderIdFromFileName(PlaceHolderId* outValue, const char* name) NN_NOEXCEPT;

    void Initialize(PathString* path, MakePlaceHolderPathFunction func, bool delayFlush) NN_NOEXCEPT
    {
        m_RootPath = path;
        m_Func = func;
        m_DelayFlush = delayFlush;
    }

    Result CreatePlaceHolderFile(PlaceHolderId id, int64_t size) NN_NOEXCEPT;
    Result SetPlaceHolderFileSize(PlaceHolderId id, int64_t size) NN_NOEXCEPT;
    Result TryGetPlaceHolderFileSize(bool* outIsFileSizeSet, int64_t* outFileSize, PlaceHolderId id) NN_NOEXCEPT;
    Result DeletePlaceHolderFile(PlaceHolderId id) NN_NOEXCEPT;
    Result WritePlaceHolderFile(PlaceHolderId id, int64_t offset, const void* buffer, size_t size) NN_NOEXCEPT;
    void   GetPath(PathString* outPath, PlaceHolderId id) NN_NOEXCEPT;
    int GetHierarchicalDirectoryDepth() const NN_NOEXCEPT { return GetHierarchicalPlaceHolderDirectoryDepth(m_Func); }
    // キャッシュ操作は行わずに、パスを取得する
    void MakePath(PathString* outPath, PlaceHolderId id) const NN_NOEXCEPT;
    void   InvalidateAll() NN_NOEXCEPT;
    Result EnsurePlaceHolderDirectory(PlaceHolderId id) NN_NOEXCEPT;
private:
    struct Cache
    {
        PlaceHolderId  id;
        fs::FileHandle handle;
        uint64_t       counter;
    };

    Result Open(fs::FileHandle* outHandle, PlaceHolderId id) NN_NOEXCEPT;
    // キャッシュから探し、見つかった場合キャッシュ上からは削除する (FileHandle の所有権は呼び出し元にいく)
    bool   LoadCache(fs::FileHandle* outHandle, PlaceHolderId id) NN_NOEXCEPT;
    // キャッシュに FileHandle を登録する (FileHandle の所有権が Cache にいく)
    void   StoreCache(PlaceHolderId id, fs::FileHandle handle) NN_NOEXCEPT;
    void   Invalidate(Cache* cache) NN_NOEXCEPT;
    Cache* FindCache(PlaceHolderId id) NN_NOEXCEPT;
    Cache* GetCacheSpace() NN_NOEXCEPT;

    Cache       m_Caches[CacheCount];
    PathString* m_RootPath;
    uint64_t    m_CurrentCounter;
    os::Mutex   m_CacheMutex;
    MakePlaceHolderPathFunction m_Func;
    bool        m_DelayFlush;
};

}}
