﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <cstring>
#include <nn/ncm/ncm_ContentInfo.h>
#include <nn/ncm/ncm_ContentMetaId.h>
#include <nn/ncm/ncm_ContentMeta.h>

namespace nn { namespace ncm {

    enum class UpdateType : Bit8
    {
        ApplyAsDelta = 0,
        Overwrite = 1,
        Create = 2,
    };

    struct FragmentIndicator
    {
        uint16_t  contentInfoIndex; // Delta 内の何番目のコンテンツか
        uint16_t  fragmentIndex;
    };
    struct FragmentSet
    {
        ContentId   sourceContentId;
        ContentId   destinationContentId;
        uint32_t    sourceSizeLow;
        uint16_t    sourceSizeHigh;
        uint16_t    destinationSizeHigh; // Struct サイズを抑えるために、わざと順番を変えている
        uint32_t    destinationSizeLow;
        uint16_t    fragmentCount;
        ContentType targetContentType;
        UpdateType  updateType;
        Bit8        reserved[4];

        int64_t GetSourceSize() const NN_NOEXCEPT
        {
            return (static_cast<int64_t>(sourceSizeHigh) << 32) + sourceSizeLow;
        }
        int64_t GetDestinationSize() const NN_NOEXCEPT
        {
            return (static_cast<int64_t>(destinationSizeHigh) << 32) + destinationSizeLow;
        }
        void SetSourceSize(int64_t size) NN_NOEXCEPT
        {
            sourceSizeLow = size & 0xFFFFFFFFLL;
            sourceSizeHigh = static_cast<uint16_t>(size >> 32);
        }
        void SetDestinationSize(int64_t size) NN_NOEXCEPT
        {
            destinationSizeLow = size & 0xFFFFFFFFLL;
            destinationSizeHigh = static_cast<uint16_t>(size >> 32);
        }
    };

    struct DeltaMetaExtendedDataHeader
    {
        PatchId  sourceId;
        PatchId  destinationId;
        uint32_t sourceVersion;
        uint32_t destinationVersion;
        uint16_t fragmentSetCount;
        Bit8     reserved[6];
    };

// データ構造
// [PatchMetaExtendedDataHeader]
// [PatchHistoryHeader x M]
// [PatchDeltaHistory x N]
// [PatchDeltaHeader x O]
// [FragmentSet x P]
// [PatchHistory's ContentInfo .. ]
// [PatchDelta's PackagedContentInfo .. ] // この情報から直接 InstallContentMeta を作っているので、Packaged
// [PatchFragmentSet's Fragments ..]
    struct PatchMetaExtendedDataHeader
    {
        uint32_t historyCount;
        uint32_t deltaHistoryCount;
        uint32_t deltaCount;
        uint32_t fragmentSetCount;
        uint32_t historyContentTotalCount;
        uint32_t deltaContentTotalCount;
        Bit8     reserved[2];
    };
    struct PatchHistoryHeader
    {
        ContentMetaKey  key;
        Hash            digest;
        uint16_t        contentCount;
        Bit8            reserved[2];
    };
    struct PatchDeltaHistory
    {
        PatchId  sourceId;
        PatchId  destinationId;
        uint32_t sourceVersion;
        uint32_t destinationVersion;
        uint64_t downloadSize;
        Bit8     reserved[4];
    };
    struct PatchDeltaHeader
    {
        DeltaMetaExtendedDataHeader delta;
        uint16_t contentCount;
        Bit8     reserved[4];
    };

    // 愚直な線形探索となる構造
    // 可変長が多いのでやむなしな面もあるのだが。。
    // できるだけ新しいデータを手前に置くようにすると、パフォーマンスは良いはず
    template<typename TDataPointer, typename TBitPointer>
    class PatchMetaExtendedDataReaderWriterBase
    {
    public:
        PatchMetaExtendedDataReaderWriterBase(TDataPointer data, size_t size) NN_NOEXCEPT : m_Data(data), m_Size(size)
        {
            NN_UNUSED(m_Size);
        }
        const PatchMetaExtendedDataHeader* GetHeader() const NN_NOEXCEPT
        {
            return reinterpret_cast<const PatchMetaExtendedDataHeader*>(GetHeaderAddress());
        }
        const PatchHistoryHeader* GetPatchHistoryHeader(int index) const NN_NOEXCEPT
        {
            return reinterpret_cast<const PatchHistoryHeader*>(GetPatchHistoryHeaderAddress(index));
        }
        const PatchDeltaHistory* GetPatchDeltaHistory(int index) const NN_NOEXCEPT
        {
            return reinterpret_cast<const PatchDeltaHistory*>(GetPatchDeltaHistoryAddress(index));
        }
        const ContentInfo* GetPatchHistoryContentInfo(int historyIndex, int contentIndex) const NN_NOEXCEPT
        {
            return reinterpret_cast<const ContentInfo*>(GetPatchHistoryContentInfoAddress(historyIndex, contentIndex));
        }
        const PatchDeltaHeader* GetPatchDeltaHeader(int index) const NN_NOEXCEPT
        {
            return reinterpret_cast<const PatchDeltaHeader*>(GetPatchDeltaHeaderAddress(index));
        }
        const PackagedContentInfo* GetPatchDeltaPackagedContentInfo(int deltaIndex, int contentIndex) const NN_NOEXCEPT
        {
            return reinterpret_cast<const PackagedContentInfo*>(GetPatchDeltaPackagedContentInfoAddress(deltaIndex, contentIndex));
        }
        const FragmentSet* GetFragmentSet(int deltaIndex, int fragmentSetIndex) const NN_NOEXCEPT
        {
            return reinterpret_cast<const FragmentSet*>(GetFragmentSetAddress(deltaIndex, fragmentSetIndex));
        }
        // 特定の fragmentIndex のものを探す場合は、FindFragmentIndicator を使う
        const FragmentIndicator* GetFragmentIndicator(int deltaIndex, int fragmentSetIndex, int index) const NN_NOEXCEPT
        {
            return reinterpret_cast<const FragmentIndicator*>(GetFragmentIndicatorAddress(deltaIndex, fragmentSetIndex, index));
        }
        const FragmentIndicator* FindFragmentIndicator(int deltaIndex, int fragmentSetIndex, int fragmentIndex) const NN_NOEXCEPT
        {
            auto fragmentSet = GetFragmentSet(deltaIndex, fragmentSetIndex);
            auto fragment = GetFragmentIndicator(deltaIndex, fragmentSetIndex, 0);
            for(int i = 0; i < fragmentSet->fragmentCount; ++i)
            {
                if(fragment[i].fragmentIndex == fragmentIndex)
                {
                    return &(fragment[i]);
                }
            }
            return nullptr;
        }

    protected:
        // 処理高速化用
        int CountFragmentSet(int deltaIndex) const NN_NOEXCEPT
        {
            // delta を順番に見て、fragmentSet の総数を出す
            int count = 0;
            auto deltaHeader = GetPatchDeltaHeader(0);
            for(int i = 0; i < deltaIndex; ++i)
            {
                count += deltaHeader[i].delta.fragmentSetCount;
            }
            return count;
        }
        int CountHisotryContent(int historyIndex) const NN_NOEXCEPT
        {
            int count = 0;
            auto historyHeader = GetPatchHistoryHeader(0);
            for (int i = 0; i < historyIndex; ++i)
            {
                count += historyHeader[i].contentCount;
            }
            return count;
        }
        int CountDeltaContent(int deltaIndex) const NN_NOEXCEPT
        {
            int count = 0;
            auto deltaHeader = GetPatchDeltaHeader(0);
            for (int i = 0; i < deltaIndex; ++i)
            {
                count += deltaHeader[i].contentCount;
            }
            return count;
        }
        // delta 内の fragmentIndex ではなく、fragmentSet 全体でのインデックス
        int CountFragment(int fragmentSetAbsoluteIndex) const NN_NOEXCEPT
        {
            int count = 0;
            auto fragmentSet = GetFragmentSet(0, 0); // FragmentSet の先頭を取得
            for (int i = 0; i < fragmentSetAbsoluteIndex; ++i)
            {
                count += fragmentSet[i].fragmentCount;
            }
            return count;
        }

        TBitPointer GetHeaderAddress() const NN_NOEXCEPT
        {
            return reinterpret_cast<TBitPointer>(m_Data);
        }
        TBitPointer GetPatchHistoryHeaderAddress(int index) const NN_NOEXCEPT
        {
            auto header = GetHeader();
            NN_ABORT_UNLESS(static_cast<uint16_t>(index) < header->historyCount);

            return GetHeaderAddress()
                + sizeof(PatchMetaExtendedDataHeader)
                + sizeof(PatchHistoryHeader) * index;
        }
        TBitPointer GetPatchDeltaHistoryAddress(int index) const NN_NOEXCEPT
        {
            auto header = GetHeader();
            NN_ABORT_UNLESS(static_cast<uint16_t>(index) < header->deltaHistoryCount);

            return GetHeaderAddress()
                + sizeof(PatchMetaExtendedDataHeader)
                + sizeof(PatchHistoryHeader) * header->historyCount
                + sizeof(PatchDeltaHistory) * index;
        }
        TBitPointer GetPatchDeltaHeaderAddress(int index) const NN_NOEXCEPT
        {
            auto header = GetHeader();
            NN_ABORT_UNLESS(static_cast<uint16_t>(index) < header->deltaCount);

            return GetHeaderAddress()
                + sizeof(PatchMetaExtendedDataHeader)
                + sizeof(PatchHistoryHeader) * header->historyCount
                + sizeof(PatchDeltaHistory) * header->deltaHistoryCount
                + sizeof(PatchDeltaHeader) * index;
        }
        TBitPointer GetFragmentSetAddress(int deltaIndex, int fragmentSetIndex) const NN_NOEXCEPT
        {
            auto header = GetHeader();
            NN_ABORT_UNLESS(static_cast<uint16_t>(deltaIndex) < header->deltaCount);

            auto deltaHeader = GetPatchDeltaHeader(deltaIndex);
            NN_ABORT_UNLESS(static_cast<uint16_t>(fragmentSetIndex) < deltaHeader->delta.fragmentSetCount);

            auto totalFragmentSetCount = CountFragmentSet(deltaIndex);

            return GetHeaderAddress()
                + sizeof(PatchMetaExtendedDataHeader)
                + sizeof(PatchHistoryHeader) * header->historyCount
                + sizeof(PatchDeltaHistory) * header->deltaHistoryCount
                + sizeof(PatchDeltaHeader) * header->deltaCount
                + sizeof(FragmentSet) * (totalFragmentSetCount + fragmentSetIndex);
        }
        TBitPointer GetPatchHistoryContentInfoAddress(int historyIndex, int contentIndex) const NN_NOEXCEPT
        {
            auto header = GetHeader();
            auto historyHeader = GetPatchHistoryHeader(historyIndex);
            NN_ABORT_UNLESS(static_cast<uint16_t>(contentIndex) < historyHeader->contentCount);

            auto contentCount = CountHisotryContent(historyIndex);

            return GetHeaderAddress() + sizeof(PatchMetaExtendedDataHeader)
                + sizeof(PatchHistoryHeader) * header->historyCount
                + sizeof(PatchDeltaHistory) * header->deltaHistoryCount
                + sizeof(PatchDeltaHeader) * header->deltaCount
                + sizeof(FragmentSet) * header->fragmentSetCount
                + sizeof(ContentInfo) * (contentCount + contentIndex);
        }
        TBitPointer GetPatchDeltaPackagedContentInfoAddress(int deltaIndex, int contentIndex) const NN_NOEXCEPT
        {
            auto header = GetHeader();
            auto deltaHeader = GetPatchDeltaHeader(deltaIndex);
            NN_ABORT_UNLESS(static_cast<uint16_t>(contentIndex) < deltaHeader->contentCount);

            auto contentCount = CountDeltaContent(deltaIndex);

            return GetHeaderAddress() + sizeof(PatchMetaExtendedDataHeader)
                + sizeof(PatchHistoryHeader) * header->historyCount
                + sizeof(PatchDeltaHistory) * header->deltaHistoryCount
                + sizeof(PatchDeltaHeader) * header->deltaCount
                + sizeof(FragmentSet) * header->fragmentSetCount
                + sizeof(ContentInfo) * header->historyContentTotalCount
                + sizeof(PackagedContentInfo) * (contentCount + contentIndex);
        }
        TBitPointer GetFragmentIndicatorAddress(int deltaIndex, int fragmentSetIndex, int index) const NN_NOEXCEPT
        {
            auto header = GetHeader();
            auto fragmentSet = GetFragmentSet(deltaIndex, fragmentSetIndex);
            NN_ABORT_UNLESS(static_cast<uint16_t>(index) < fragmentSet->fragmentCount);

            int fragmentSetCount = CountFragmentSet(deltaIndex);
            int fragmentCount = CountFragment(fragmentSetCount + fragmentSetIndex);

            return GetHeaderAddress() + sizeof(PatchMetaExtendedDataHeader)
                + sizeof(PatchHistoryHeader) * header->historyCount
                + sizeof(PatchDeltaHistory) * header->deltaHistoryCount
                + sizeof(PatchDeltaHeader) * header->deltaCount
                + sizeof(FragmentSet) * header->fragmentSetCount
                + sizeof(ContentInfo) * header->historyContentTotalCount
                + sizeof(PackagedContentInfo) * header->deltaContentTotalCount
                + sizeof(FragmentIndicator) * (fragmentCount + index);

        }

        TDataPointer m_Data;
        const size_t m_Size;
    };
    class PatchMetaExtendedDataReader : public PatchMetaExtendedDataReaderWriterBase<const void*, const Bit8*>
    {
    public:
        PatchMetaExtendedDataReader(const void* data, size_t size) NN_NOEXCEPT : PatchMetaExtendedDataReaderWriterBase(data, size) {}
    };
    class PatchMetaExtendedDataWriter : public PatchMetaExtendedDataReaderWriterBase<void*, Bit8*>
    {
    public:
        PatchMetaExtendedDataWriter(void* data, size_t size) NN_NOEXCEPT : PatchMetaExtendedDataReaderWriterBase(data, size) {}
        void WritePatchMetaExtendedDataHeader(const PatchMetaExtendedDataHeader& header) NN_NOEXCEPT
        {
            std::memcpy(GetHeaderAddress(), &header, sizeof(header));
        }
        void WritePatchHistoryHeader(int index, const PatchHistoryHeader& header) NN_NOEXCEPT
        {
            std::memcpy(GetPatchHistoryHeaderAddress(index), &header, sizeof(header));
        }
        void WritePatchHistoryContentInfo(int historyIndex, int contentIndex, const ContentInfo& info) NN_NOEXCEPT
        {
            std::memcpy(GetPatchHistoryContentInfoAddress(historyIndex, contentIndex), &info, sizeof(info));
        }
        void WritePatchDeltaHistory(int historyIndex, const PatchDeltaHistory& info) NN_NOEXCEPT
        {
            std::memcpy(GetPatchDeltaHistoryAddress(historyIndex), &info, sizeof(info));
        }
        void WritePatchDeltaHeader(int index, const PatchDeltaHeader& header) NN_NOEXCEPT
        {
            std::memcpy(GetPatchDeltaHeaderAddress(index), &header, sizeof(header));
        }
        void WritePatchDeltaPackagedContentInfo(int deltaIndex, int contentIndex, const PackagedContentInfo& info) NN_NOEXCEPT
        {
            std::memcpy(GetPatchDeltaPackagedContentInfoAddress(deltaIndex, contentIndex), &info, sizeof(info));
        }
        void WriteFragmentSet(int deltaIndex, int fragmentIndex, const FragmentSet& fragmentSet) NN_NOEXCEPT
        {
            std::memcpy(GetFragmentSetAddress(deltaIndex, fragmentIndex), &fragmentSet, sizeof(fragmentSet));
        }
        void WriteFragmentIndicator(int deltaIndex, int fragmentIndex, int index, const FragmentIndicator& indicator) NN_NOEXCEPT
        {
            std::memcpy(GetFragmentIndicatorAddress(deltaIndex, fragmentIndex, index), &indicator, sizeof(indicator));
        }
        static size_t CalculateSize(int historyCount, int historyContentCount, int deltaHistoryCount, int deltaCount, int deltaContentCount, int fragmentSetCount, int fragmentCount) NN_NOEXCEPT
        {
            return sizeof(PatchMetaExtendedDataHeader)
                + sizeof(PatchHistoryHeader) * historyCount
                + sizeof(ContentInfo) * historyContentCount
                + sizeof(PatchDeltaHistory) * deltaHistoryCount
                + sizeof(PatchDeltaHeader) * deltaCount
                + sizeof(PackagedContentInfo) * deltaContentCount
                + sizeof(FragmentSet) * fragmentSetCount
                + sizeof(FragmentIndicator) * fragmentCount;
        }
    };

    template<typename TDataPointer, typename TBitPointer>
    class DeltaMetaExtendedDataReaderWriterBase
    {
    public:
        DeltaMetaExtendedDataReaderWriterBase(TDataPointer data, size_t size) NN_NOEXCEPT : m_Data(data), m_Size(size)
        {
            NN_UNUSED(m_Size);
        }

        const DeltaMetaExtendedDataHeader* GetHeader() const NN_NOEXCEPT
        {
            return reinterpret_cast<const DeltaMetaExtendedDataHeader*>(GetHeaderAddress());
        }
        const FragmentSet* GetFragmentSet(int index) const NN_NOEXCEPT
        {
            return reinterpret_cast<const FragmentSet*>(GetFragmentSetAddress(index));
        }
        // 特定の fragmentIndex のものを探す場合は、FindFragmentIndicator を使う
        const FragmentIndicator* GetFragmentIndicator(int fragmentSetIndex, int index) const NN_NOEXCEPT
        {
            return reinterpret_cast<const FragmentIndicator*>(GetFragmentIndicatorAddress(fragmentSetIndex, index));
        }
        const FragmentIndicator* FindFragmentIndicator(int fragmentSetIndex, int fragmentIndex) const NN_NOEXCEPT
        {
            auto fragmentSet = GetFragmentSet(fragmentSetIndex);
            auto fragment = GetFragmentIndicator(fragmentSetIndex, 0);

            for (int i = 0; i < fragmentSet->fragmentCount; ++i)
            {
                if (fragment[i].fragmentIndex == fragmentIndex)
                {
                    return &(fragment[i]);
                }
            }
            return nullptr;
        }
    protected:
        int CountFragment(int fragmentSetIndex) const NN_NOEXCEPT
        {
            int count = 0;
            auto fragmentSet = GetFragmentSet(0);
            for (int i = 0; i < fragmentSetIndex; ++i)
            {
                count += fragmentSet[i].fragmentCount;
            }
            return count;
        }

        TBitPointer GetHeaderAddress() const NN_NOEXCEPT
        {
            return reinterpret_cast<TBitPointer>(m_Data);
        }
        TBitPointer GetFragmentSetAddress(int index) const NN_NOEXCEPT
        {
            auto header = GetHeader();
            NN_ABORT_UNLESS(static_cast<uint16_t>(index) < header->fragmentSetCount); // TODO: コンテントメタが不正な時に abort でよいのか

            return GetHeaderAddress()
                + sizeof(DeltaMetaExtendedDataHeader)
                + sizeof(FragmentSet) * index;
        }
        TBitPointer GetFragmentIndicatorAddress(int fragmentSetIndex, int index) const NN_NOEXCEPT
        {
            auto header = GetHeader();
            auto fragmentSet = GetFragmentSet(fragmentSetIndex); // fragmentIndex の正当性はここでみている
            NN_ABORT_UNLESS(static_cast<uint16_t>(index) < fragmentSet->fragmentCount); // TODO: コンテントメタが不正な時に abort でよいのか

            int fragmentCount = CountFragment(fragmentSetIndex);

            return GetHeaderAddress()
                + sizeof(DeltaMetaExtendedDataHeader)
                + sizeof(FragmentSet) * header->fragmentSetCount
                + sizeof(FragmentIndicator) * (fragmentCount + index);
        }

    private:
        TDataPointer m_Data;
        const size_t m_Size;
    };

    class DeltaMetaExtendedDataReader : public DeltaMetaExtendedDataReaderWriterBase<const void*, const Bit8*>
    {
    public:
        DeltaMetaExtendedDataReader(const void* data, size_t size) NN_NOEXCEPT : DeltaMetaExtendedDataReaderWriterBase(data, size) {}
    };

    class DeltaMetaExtendedDataWriter : public DeltaMetaExtendedDataReaderWriterBase<void*, Bit8*>
    {
    public:
        DeltaMetaExtendedDataWriter(void* data, size_t size) NN_NOEXCEPT : DeltaMetaExtendedDataReaderWriterBase(data, size) {}

        DeltaMetaExtendedDataHeader* GetHeader() const NN_NOEXCEPT
        {
            return reinterpret_cast<DeltaMetaExtendedDataHeader*>(GetHeaderAddress());
        }
        void WriteHeader(const DeltaMetaExtendedDataHeader& header) NN_NOEXCEPT
        {
            std::memcpy(GetHeader(), &header, sizeof(DeltaMetaExtendedDataHeader));
        }
        void WriteFragmentSet(const FragmentSet& set, int index) NN_NOEXCEPT
        {
            std::memcpy(reinterpret_cast<void*>(GetFragmentSetAddress(index)), &set, sizeof(set));
        }
        void WriteFragmentIndicator(const FragmentIndicator& indicator, int fragmentSetIndex, int fragmentIndex) NN_NOEXCEPT
        {
            std::memcpy(reinterpret_cast<void*>(GetFragmentIndicatorAddress(fragmentSetIndex, fragmentIndex)), &indicator, sizeof(indicator));
        }

        static size_t CalculateSize(int fragmentSetCount, int fragmentIndicatorCount) NN_NOEXCEPT
        {
            return sizeof(DeltaMetaExtendedDataHeader)
                + sizeof(FragmentSet) * fragmentSetCount
                + sizeof(FragmentIndicator) * fragmentIndicatorCount;
        }

    private:
    };

    class IDeltaReaderWrapper
    {
    public:
        virtual ~IDeltaReaderWrapper() {}

        virtual const DeltaMetaExtendedDataHeader* GetHeader() const NN_NOEXCEPT = 0;
        virtual const FragmentSet* GetFragmentSet(int index) const NN_NOEXCEPT = 0;
        virtual const FragmentIndicator* GetFragmentIndicator(int fragmentSetIndex, int index) const NN_NOEXCEPT = 0;
        virtual const FragmentIndicator* FindFragmentIndicator(int fragmentSetIndex, int fragmentIndex) const NN_NOEXCEPT = 0;
        virtual const PackagedContentInfo* GetContentInfo(int index) const NN_NOEXCEPT = 0;
    };

    class PatchDeltaReaderWrapper : public IDeltaReaderWrapper
    {
    public:
        PatchDeltaReaderWrapper(const void* ptr, size_t size, int index) NN_NOEXCEPT : m_Reader(ptr, size), m_BindingIndex(index) {}
        virtual ~PatchDeltaReaderWrapper() NN_NOEXCEPT {}

        void SetIndex(int index) NN_NOEXCEPT
        {
            auto header = m_Reader.GetHeader();
            NN_ABORT_UNLESS(static_cast<uint16_t>(index) < header->deltaCount);
            m_BindingIndex = index;
        }

        virtual const DeltaMetaExtendedDataHeader* GetHeader() const NN_NOEXCEPT override
        {
            return &(m_Reader.GetPatchDeltaHeader(m_BindingIndex)->delta);
        }
        virtual const FragmentSet* GetFragmentSet(int index) const NN_NOEXCEPT override
        {
            return m_Reader.GetFragmentSet(m_BindingIndex, index);
        }
        virtual const FragmentIndicator* GetFragmentIndicator(int fragmentSetIndex, int index) const NN_NOEXCEPT override
        {
            return m_Reader.GetFragmentIndicator(m_BindingIndex, fragmentSetIndex, index);
        }
        virtual const FragmentIndicator* FindFragmentIndicator(int fragmentSetIndex, int fragmentIndex) const NN_NOEXCEPT override
        {
            return m_Reader.FindFragmentIndicator(m_BindingIndex, fragmentSetIndex, fragmentIndex);
        }
        virtual const PackagedContentInfo* GetContentInfo(int index) const NN_NOEXCEPT override
        {
            return m_Reader.GetPatchDeltaPackagedContentInfo(m_BindingIndex, index);
        }
    private:
        PatchMetaExtendedDataReader m_Reader;
        int m_BindingIndex;
    };

    class DeltaReaderWrapper : public IDeltaReaderWrapper
    {
    public:
        DeltaReaderWrapper(const void* ptr1, size_t size1, const void* ptr2, size_t size2) NN_NOEXCEPT : m_MetaReader(ptr1, size1), m_Reader(ptr2, size2) {}
        virtual ~DeltaReaderWrapper() NN_NOEXCEPT {}

        virtual const DeltaMetaExtendedDataHeader* GetHeader() const NN_NOEXCEPT override
        {
            return m_Reader.GetHeader();
        }
        virtual const FragmentSet* GetFragmentSet(int index) const NN_NOEXCEPT override
        {
            return m_Reader.GetFragmentSet(index);
        }
        virtual const FragmentIndicator* GetFragmentIndicator(int fragmentSetIndex, int index) const NN_NOEXCEPT override
        {
            return m_Reader.GetFragmentIndicator(fragmentSetIndex, index);
        }
        virtual const FragmentIndicator* FindFragmentIndicator(int fragmentSetIndex, int fragmentIndex) const NN_NOEXCEPT override
        {
            return m_Reader.FindFragmentIndicator(fragmentSetIndex, fragmentIndex);
        }
        virtual const PackagedContentInfo* GetContentInfo(int index) const NN_NOEXCEPT override
        {
            return m_MetaReader.GetContentInfo(index);
        }
    private:
        PackagedContentMetaReader   m_MetaReader;
        DeltaMetaExtendedDataReader m_Reader;
    };
}}
