﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <memory>
#include <nn/nn_Common.h>
#include <nn/nn_StaticAssert.h>
#include <nn/crypto/crypto_Sha256Generator.h>
#include <nn/os/os_Mutex.h>
#include <nn/util/util_Optional.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/ncm/ncm_Result.h>
#include <nn/ncm/ncm_AutoBuffer.h>
#include <nn/ncm/ncm_ContentStorage.h>
#include <nn/ncm/ncm_ContentMetaDatabase.h>
#include <nn/ncm/ncm_ContentMeta.h>
#include <nn/ncm/ncm_ContentMetaExtendedData.h>
#include <nn/ncm/ncm_ContentMetaKey.h>
#include <nn/ncm/ncm_ContentInfo.h>
#include <nn/ncm/ncm_ContentInfoData.h>
#include <nn/ncm/ncm_ApplyDeltaProgress.h>
#include <nn/ncm/ncm_DeltaApplier.h>

namespace nn { namespace fs {

    struct FileHandle;

}}

namespace nn { namespace ncm { namespace detail {

    struct MountName;

}}}

namespace nn { namespace ncm {
    const int MaxPlaceHolderSize = 7;          // 1 個は meta 用に移譲
    const size_t MinimumBufferSize = 32 << 10; // 特に根拠がある数字ではないが、とりあえず

    class ApplyDeltaTaskBase
    {
        NN_DISALLOW_COPY(ApplyDeltaTaskBase);
        NN_DISALLOW_MOVE(ApplyDeltaTaskBase);

    public:
        struct TaskState
        {
            // 0
            StorageContentMetaKey sourceKey;  // 更新対象の StorageContentMetaKey
            StorageContentMetaKey deltaKey;   // 更新に利用する StorageContentMetaKey
            // 48
            int     fragmentSetIndex;         // 何番目の fragmentSet を処理中か
            int     fragmentIndex;            // 何番目の fragment を処理中か
            int64_t fragmentSetProcessedSize; // fragmentSet 全体での処理済みサイズ
            int64_t fragmentProcessedSize;    // fragment 内での処理済みサイズ
            int64_t deltaApplierOffset;       // DeltaApplier が利用するオフセット
            char    deltaApplierStatus[DeltaApplier::StatusSize];  // DeltaApplier 用中断状態

            // 168 (48 + 24 + 96)
            ContentInfo   metaContentInfo;    // 更新後の ContentMeta の ContentInfo
            PlaceHolderId placeHolders[MaxPlaceHolderSize];
            PlaceHolderId metaPlaceHolder;
            int           fragmentSetCount;
            Bit8          reserved[4];

            // 328 (168 + 160)
            ApplyDeltaProgress progress;      // プログレス

            // 352
        };

    public:
        ApplyDeltaTaskBase() NN_NOEXCEPT : m_ProgressMutex(false), m_CancelMutex(false), m_ApplyBuffer(nullptr) {}
        virtual ~ApplyDeltaTaskBase() NN_NOEXCEPT {}

        /**
         *   @brief      パッチ間差分を適用するための準備をします。
         *   @details    placeholder を作成し、更新対象のコンテンツをそこに move します。
         */
        virtual Result Prepare() NN_NOEXCEPT = 0;

        /**
         *   @brief      Cleanup を行います。
         *   @details    更新途中の placeholder、更新に利用している delta を削除します。
         */
        Result Cleanup() NN_NOEXCEPT;

        /**
         *   @brief      パッチ間差分の適用を始めます。
         */
        Result Execute() NN_NOEXCEPT;

        /**
         *   @brief      適用が完了したパッチをコミットし、利用できるようにします。
         */
        Result Commit(bool doCleanup = true) NN_NOEXCEPT;

        /**
         *   @brief      パッチ間差分の適用状況を取得します。
         */
        ApplyDeltaProgress GetProgress() NN_NOEXCEPT;

        /**
         *   @brief      パッチ間差分の適用を中断します。
         */
        Result Cancel() NN_NOEXCEPT;

        /**
         *   @brief      パッチ間差分適用時のバッファを与えます。
         */
        Result SetBuffer(void* buffer, size_t size) NN_NOEXCEPT
        {
            NN_RESULT_THROW_UNLESS(size >= MinimumBufferSize, ResultBufferNotEnough());

            m_ApplyBuffer = buffer;
            m_ApplyBufferSize = size;

            NN_RESULT_SUCCESS;
        }

        /**
         *   @brief      このタスクが消費しているストレージのサイズを計算します。
         *   @details    計上するのは placeholder のサイズです。
         */
        Result CalculateOccupiedPlaceholderSize(int64_t* outValue, StorageId storageId) NN_NOEXCEPT;

        /**
        *   @brief      このタスクが要求する空きストレージのサイズを計算します。
        */
        Result CalculateRequiredSizeMaxForPrepare(int64_t* outValue) NN_NOEXCEPT;
        Result CalculateRequiredSizeMaxForResume(int64_t* outValue) NN_NOEXCEPT;

    protected:
        Result InitializeCommon() NN_NOEXCEPT;
        Result MoveNext(bool* outComplete) NN_NOEXCEPT;

        Result CreatePlaceHolder(int fragmentSetIndex, const PlaceHolderId& placeHolder) NN_NOEXCEPT;

        void SetTotalDeltaSize(int64_t totalSize) NN_NOEXCEPT;
        void IncrementAppliedDeltaProgress(int64_t appliedSize) NN_NOEXCEPT;
        void SetProgressState(ApplyDeltaProgressState state) NN_NOEXCEPT;
        void CleanupProgress() NN_NOEXCEPT;
        Result CopyContentMeta(ContentStorage* destinationStorage, const PlaceHolderId& placeHolder) NN_NOEXCEPT;

        Result CalculateRequiredSizeMaxForPrepareInternal(int64_t* outValue) NN_NOEXCEPT;

    private:
        Result Register(int fragmentSetIndex, const PlaceHolderId& placeHolder) NN_NOEXCEPT;
        Result CreateContentMeta(ContentMetaKey* key, AutoBuffer* buffer, const PlaceHolderId& placeHolder, const ContentInfo& metaContentInfo) NN_NOEXCEPT;
        Result CreateContentMeta(ContentMetaKey* key, AutoBuffer* buffer, const ContentId& contentId, const ContentInfo& metaContentInfo) NN_NOEXCEPT;
        Result OpenFragment(detail::MountName* pOutMountName, fs::FileHandle* pOutFile, const ContentInfo& fragment) NN_NOEXCEPT;
        void CloseFragment(const detail::MountName& mountName, fs::FileHandle file) NN_NOEXCEPT;
        Result ApplyDataFragment(bool* outCompleteFragment, int64_t* outProcessedOffset, int64_t targetDestinationSize, int64_t fragmentSetOffset, int64_t fragmentOffset, const PlaceHolderId& placeHolder, const ContentInfo& fragment) NN_NOEXCEPT;
        Result ApplyDeltaFragment(bool* outCompleteFragment, int64_t* outProcessedOffset, int64_t fragmentSetOffset, int64_t fragmentOffset, const PlaceHolderId& placeHolder, const ContentInfo& fragment) NN_NOEXCEPT;
        bool IsCancelRequested() NN_NOEXCEPT { return m_CancelToken; }
        void ClearCancelRequest() NN_NOEXCEPT { m_CancelToken = false; }

    protected:
        TaskState* m_TaskState;

        AutoBuffer                           m_MetaBuffer; // パッチだと 1000 バージョンまで考えると 200KiB まで使う
        std::unique_ptr<IDeltaReaderWrapper> m_DeltaReader;

        os::Mutex    m_ProgressMutex;
        os::Mutex    m_CancelMutex;
        bool         m_CancelToken;

        void*        m_ApplyBuffer;
        size_t       m_ApplyBufferSize;
    };
NN_STATIC_ASSERT(sizeof(ApplyDeltaTaskBase::TaskState) == 352);
}}
