﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/os/os_Mutex.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_Optional.h>
#include <nn/ncm/ncm_Result.h>
#include <nn/ncm/ncm_ApplyDeltaTaskBase.h>

namespace nn { namespace ncm {
    class ApplyPatchDeltaTask : public ApplyDeltaTaskBase
    {
        NN_DISALLOW_COPY(ApplyPatchDeltaTask);
        NN_DISALLOW_MOVE(ApplyPatchDeltaTask);

    public:
        ApplyPatchDeltaTask() NN_NOEXCEPT : ApplyDeltaTaskBase() {}
        virtual ~ApplyPatchDeltaTask() NN_NOEXCEPT {}

        /**
         *   @brief      指定した source / destination 間をパッチ間差分を用いて更新します
         *   @details    destination に指定する Key の InstallType は Fragment-Only で、インストールしておく必要があります。
         *
         *   @param[in]  source               更新対象のパッチ
         *   @param[in]  destination          更新後のパッチ
         *   @param[in]  taskState            ApplyDeltaTask の適用状態を保存する領域です。これ以降、タスクが完了するまで外部で書き換えることは禁止します。
         */
        Result Initialize(const StorageContentMetaKey& source, const StorageContentMetaKey& destination, TaskState* taskState) NN_NOEXCEPT;

        /**
         *   @brief      TaskState を指定して、パッチ間差分の適用を再開します。
         *   @details    taskState には、前回の Initialize 時に指定していたものを書き込んでおく必要があります。
         *
         *   @param[in]  taskState     保存されている TaskState です。これ以降、タスクが完了するまで外部でこの領域を書き換えることは禁止します。
         */
        Result InitializeForResume(TaskState* taskState) NN_NOEXCEPT;

        /**
         *   @brief      パッチ間差分を適用するための準備をします。
         *   @details    placeholder を作成し、更新対象のコンテンツをそこに move します。
         */
        virtual Result Prepare() NN_NOEXCEPT NN_OVERRIDE;


    private:
        Result PrepareMeta() NN_NOEXCEPT;
        Result PrepareDeltaReader(const PackagedContentMetaReader& reader, const PatchMetaExtendedDataReader& patchReader) NN_NOEXCEPT;

    };

    class ApplyDeltaTask : public ApplyDeltaTaskBase
    {
        NN_DISALLOW_COPY(ApplyDeltaTask);
        NN_DISALLOW_MOVE(ApplyDeltaTask);

    public:
        ApplyDeltaTask() NN_NOEXCEPT : ApplyDeltaTaskBase(){}
        virtual ~ApplyDeltaTask() NN_NOEXCEPT {}

        /**
         *   @brief      delta を用いたパッチの更新を初期化します。
         *   @details    更新対象のパッチを source、更新に利用するパッチ間差分を delta として初期化します。
         *               あらかじめ、source, delta がインストールされている必要があります。
         */
        Result Initialize(const StorageContentMetaKey& source, const StorageContentMetaKey& delta, TaskState* taskState) NN_NOEXCEPT;

        /**
         *   @brief      パッチ間差分を適用するための準備をします。
         *   @details    placeholder を作成し、更新対象のコンテンツをそこに move します。
         */
        virtual Result Prepare() NN_NOEXCEPT NN_OVERRIDE;
    };
}}
