﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <cstring>
#include <memory>
#include <type_traits>
#include <nn/nn_StaticAssert.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_File.h>
#include <nn/fs/fs_Directory.h>
#include <nn/fs/fs_FileSystem.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/kvdb/kvdb_Result.h>
#include <nn/kvdb/kvdb_BoundedString.h>
#include <nn/kvdb/kvdb_FileKeyValueStore.h>

namespace nn { namespace kvdb {

namespace detail {

    struct LeastRecentlyUsedHeader
    {
        int entryCount;
    };

    template<class Key, int MaxEntryCount>
    class LeastRecentlyUsedList
    {
    public:
        LeastRecentlyUsedList() NN_NOEXCEPT : m_KeyList() {}

        static const size_t RequiredBufferSize = sizeof(Key) * MaxEntryCount;

        static const size_t ListFileSize = sizeof(LeastRecentlyUsedHeader) + RequiredBufferSize;

        static const size_t KeySize = sizeof(Key);

        static Result Create(const char* filePath) NN_NOEXCEPT
        {
            NN_RESULT_DO(fs::CreateFile(filePath, ListFileSize));

            fs::FileHandle file;
            NN_RESULT_DO(fs::OpenFile(&file, filePath, fs::OpenMode_Write));
            NN_UTIL_SCOPE_EXIT{ fs::CloseFile(file); };

            LeastRecentlyUsedHeader header = { 0 };
            NN_RESULT_DO(fs::WriteFile(file, 0, &header, sizeof(header), fs::WriteOption::MakeValue(fs::WriteOptionFlag_Flush)));

            NN_RESULT_SUCCESS;
        }

        Result Initialize(const char* filePath, void* buffer, size_t bufferSize) NN_NOEXCEPT
        {
            NN_UNUSED(bufferSize);
            NN_SDK_REQUIRES(buffer);
            NN_SDK_REQUIRES(bufferSize >= RequiredBufferSize);
            NN_SDK_REQUIRES(!m_KeyList);

            m_KeyList = static_cast<Key*>(buffer);
            m_FilePath.Assign(filePath);

            fs::FileHandle file;
            NN_RESULT_DO(fs::OpenFile(&file, m_FilePath, fs::OpenMode_Read));
            NN_UTIL_SCOPE_EXIT{ fs::CloseFile(file); };
            NN_RESULT_DO(fs::ReadFile(file, 0, &m_Header, sizeof(m_Header)));
            NN_RESULT_DO(fs::ReadFile(file, sizeof(m_Header), m_KeyList, RequiredBufferSize));

            NN_RESULT_SUCCESS;
        }

        bool IsEmpty() const NN_NOEXCEPT
        {
            return Count() == 0;
        }

        bool IsFull() const NN_NOEXCEPT
        {
            return Count() >= MaxEntryCount;
        }

        int Count() const NN_NOEXCEPT
        {
            return m_Header.entryCount;
        }

        Key Peak() const NN_NOEXCEPT
        {
            return Get(0);
        }

        Key Get(int index) const NN_NOEXCEPT
        {
            NN_SDK_REQUIRES(Count() > index);

            return m_KeyList[index];
        }

        void Pop() NN_NOEXCEPT
        {
            DeleteByIndex(0);
        }

        void Put(const Key& key) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES(Count() < MaxEntryCount);

            m_KeyList[Count()] = key;
            m_Header.entryCount++;
        }

        bool Delete(const Key& key) NN_NOEXCEPT
        {
            for (int i = Count() - 1; i >= 0; i--)
            {
                if (key == m_KeyList[i])
                {
                    DeleteByIndex(i);
                    return true;
                }
            }

            return false;
        }

        bool Touch(const Key& key) NN_NOEXCEPT
        {
            if (Delete(key))
            {
                Put(key);
                return true;
            }

            return false;
        }

        Result Flush() NN_NOEXCEPT
        {
            fs::WriteOption option = {};
            fs::FileHandle file;
            NN_RESULT_DO(fs::OpenFile(&file, m_FilePath, fs::OpenMode_Write));
            NN_UTIL_SCOPE_EXIT{ fs::CloseFile(file); };
            NN_RESULT_DO(fs::WriteFile(file, 0, &m_Header, sizeof(m_Header), option));
            NN_RESULT_DO(fs::WriteFile(file, sizeof(m_Header), m_KeyList, RequiredBufferSize, option));
            NN_RESULT_DO(fs::FlushFile(file));

            NN_RESULT_SUCCESS;
        }

    private:
        static size_t CalculateBodySize(uint32_t keySize, int maxEntryCount) NN_NOEXCEPT
        {
            return keySize * maxEntryCount;
        }

        void DeleteByIndex(int index) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES(Count() > index);

            std::memmove(&m_KeyList[index], &m_KeyList[index + 1], KeySize * (Count() - (index + 1)));
            m_Header.entryCount--;
        }

        LeastRecentlyUsedHeader m_Header;
        Key* m_KeyList;
        kvdb::BoundedString<768> m_FilePath;
    };
}

    template<class Key, int N, size_t MaxPathLength = 768>
    class FileKeyValueCache
    {
    public:
        typedef Key KeyType;
        static const int MaxEntryCount = N;
        static const size_t RequiredBufferSize = detail::LeastRecentlyUsedList<Key, MaxEntryCount>::RequiredBufferSize;
        typedef detail::LeastRecentlyUsedList<Key, MaxEntryCount> List;

        static Result Verify(const char* directoryPath) NN_NOEXCEPT
        {
            bool hasLruList;
            NN_RESULT_DO(HasFile(&hasLruList, MakeLeastRecentlyUsedListPath(directoryPath)));

            bool hasKvsDirectory;
            NN_RESULT_DO(HasDirectory(&hasKvsDirectory, MakeKeyValueStoreDirectoryPath(directoryPath)));

            if (!hasLruList && !hasKvsDirectory)
            {
                NN_RESULT_THROW(ResultDatabaseNotCreated());
            }

            NN_RESULT_THROW_UNLESS(hasLruList && hasKvsDirectory, ResultInvalidFileComposition());

            NN_RESULT_SUCCESS;
        }

        static Result Create(const char* directoryPath) NN_NOEXCEPT
        {
            NN_RESULT_DO(List::Create(MakeLeastRecentlyUsedListPath(directoryPath)));
            NN_RESULT_DO(fs::CreateDirectory(MakeKeyValueStoreDirectoryPath(directoryPath)));

            NN_RESULT_SUCCESS;
        }

        Result Initialize(const char* directoryPath, void* buffer, size_t bufferSize) NN_NOEXCEPT
        {
            NN_RESULT_DO(m_List.Initialize(MakeLeastRecentlyUsedListPath(directoryPath), buffer, bufferSize));
            NN_RESULT_DO(m_Kvs.Initialize(directoryPath));

            NN_RESULT_SUCCESS;
        }

        Result Flush() NN_NOEXCEPT
        {
            return m_List.Flush();
        }

        Result Put(const Key& key, const void* value, size_t valueSize) NN_NOEXCEPT
        {
            if (m_List.Touch(key))
            {
                m_Kvs.Delete(&key, sizeof(key));
            }
            else if (m_List.IsFull())
            {
                const auto& oldKey = m_List.Peak();
                m_Kvs.Delete(&oldKey, sizeof(oldKey));
                m_List.Pop();
                m_List.Put(key);
            }
            else
            {
                m_List.Put(key);
            }

            while (NN_STATIC_CONDITION(true))
            {
                NN_RESULT_TRY(m_Kvs.Put(&key, sizeof(key), value, valueSize))
                    NN_RESULT_CATCH(fs::ResultUsableSpaceNotEnough)
                    {
                        if (m_List.Count() == 1) // Put しようとしているキー以外を消しても追加できなかった
                        {
                            m_List.Pop();
                            NN_RESULT_DO(m_List.Flush());
                            NN_RESULT_RETHROW;
                        }

                        const auto& oldKey = m_List.Peak();
                        m_Kvs.Delete(&oldKey, sizeof(oldKey));
                        m_List.Pop();

                        continue;
                    }
                NN_RESULT_END_TRY

                break;
            }

            NN_RESULT_DO(m_List.Flush());
            NN_RESULT_SUCCESS;
        }

        bool Has(const Key& key) NN_NOEXCEPT
        {
            for (int i = m_List.Count() - 1; i >= 0; i--)
            {
                if (m_List.Get(i) == key)
                {
                    return true;
                }
            }

            return false;
        }

        // Get は LeastRecentlyUsedList を更新するが、Flush はしない
        Result Get(size_t* outValue, void* buffer, size_t bufferSize, const Key& key) NN_NOEXCEPT
        {
            m_List.Touch(key);
            return m_Kvs.Get(outValue, buffer, bufferSize, &key, sizeof(Key));
        }

        Result GetSize(size_t* outValue, const Key& key) NN_NOEXCEPT
        {
            return m_Kvs.GetSize(outValue, &key, sizeof(key));
        }

        Result Delete(const Key& key) NN_NOEXCEPT
        {
            m_List.Delete(key);
            NN_RESULT_DO(m_Kvs.Delete(&key, sizeof(key)));
            NN_RESULT_DO(m_List.Flush());

            NN_RESULT_SUCCESS;
        }

        Result DeleteAll() NN_NOEXCEPT
        {
            while (!m_List.IsEmpty())
            {
                const auto& key = m_List.Peak();
                m_Kvs.Delete(&key, sizeof(key));
                m_List.Pop();
            }
            NN_RESULT_DO(m_List.Flush());

            NN_RESULT_SUCCESS;
        }

        int Count() const NN_NOEXCEPT
        {
            return m_List.Count();
        }

        Key GetKey(int index) const NN_NOEXCEPT
        {
            return m_List.Get(index);
        }

    private:
#if !defined( NN_BUILD_CONFIG_COMPILER_GCC )
        NN_STATIC_ASSERT(std::is_trivially_copyable<Key>::value);
#endif

        typedef BoundedString<MaxPathLength> Path;

        FileKeyValueStore m_Kvs;
        List m_List;

        static Path MakeLeastRecentlyUsedListPath(const char* directoryPath) NN_NOEXCEPT
        {
            return Path::MakeFormat("%s/%s", directoryPath, "lru_list.dat");
        }

        static Path MakeKeyValueStoreDirectoryPath(const char* directoryPath) NN_NOEXCEPT
        {
            return Path::MakeFormat("%s/%s", directoryPath, "kvs");
        }

        static Result HasEntry(bool* outValue, const char* path, fs::DirectoryEntryType type) NN_NOEXCEPT
        {
            fs::DirectoryEntryType entryType;
            NN_RESULT_TRY(fs::GetEntryType(&entryType, path))
                NN_RESULT_CATCH(fs::ResultPathNotFound)
            {
                *outValue = false;
                NN_RESULT_SUCCESS;
            }
            NN_RESULT_END_TRY
                NN_RESULT_THROW_UNLESS(entryType == type, ResultInvalidFileComposition());

            *outValue = true;

            NN_RESULT_SUCCESS;
        }

        static Result HasFile(bool* outValue, const char* path) NN_NOEXCEPT
        {
            return HasEntry(outValue, path, fs::DirectoryEntryType_File);
        }

        static Result HasDirectory(bool* outValue, const char* path) NN_NOEXCEPT
        {
            return HasEntry(outValue, path, fs::DirectoryEntryType_Directory);
        }
    };

}}
