﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <cstdlib>

#include <nn/os.h>
#include <nn/nn_SdkLog.h>
#include <nn/gc/detail/gc_Types.h>
#include <nn/gc/detail/gc_Util.h>
#include <nn/gc/detail/gc_GcCrypto.h>

namespace nn { namespace gc { namespace detail {


//#define ASN1_DEBUG  // デバッグ用、コメントを外すと SerialNumber の16バイト固定解除と、ASN1 解析ログが出力されます。

//---------------------------------------------------------------------------
//! @brief        ASN1フォーマットでDER形式の証明書データからSerialNumber、Modulus、Exponent を取得するクラスです。
//---------------------------------------------------------------------------
class Asn1Parser
{
public:
    //---------------------------------------------------------------------------
    //! @brief        コンストラクタ
    //---------------------------------------------------------------------------
    explicit Asn1Parser() :
        m_IsParseFinished(false),
        m_TbsCertificateIndex(0),
        m_TbsCertificateLength(0),
        m_SignatureAlgorithmIndex(0),
        m_SignatureAlgorithmLength(0),
        m_SignatureIndex(0),
        m_SignatureLength(0)
    {
        memset(m_SerialNumber, 0x00, sizeof(m_SerialNumber));
        memset(m_Modulus, 0x00, sizeof(m_Modulus));
        memset(m_Exponent, 0x00, sizeof(m_Exponent));
    }
    //---------------------------------------------------------------------------
    //! @brief        デストラクタ
    //---------------------------------------------------------------------------
    ~Asn1Parser()
    {
    }

    //---------------------------------------------------------------------------
    //! @brief            GC ASIC の証明書をパースします。
    //!
    //! @param[in]        pInData  データポインタ
    //! @param[in]        length   データサイズ
    //! @return           成功の場合は true、そうでない場合は false が返ります。
    //---------------------------------------------------------------------------
    bool ParseGcAsicCertificate( const char* pInData, const size_t length ) NN_NOEXCEPT;


    //---------------------------------------------------------------------------
    //! @brief        serialNumber を取得します。ParseForGcAsicCert を呼ぶ前に呼ぶと何もしません。
    //!
    //! @param[out]       pOutBuffer       serialNumber を入れる配列のポインタ
    //! @param[in]        outBufferLength  pOutBuffer の確保しているサイズ
    //---------------------------------------------------------------------------
    void GetSerialNumber(char* pOutBuffer, const size_t outBufferLength) NN_NOEXCEPT
    {
        NN_DETAIL_GC_ABORT_UNLESS_SDK_REQUIRES(outBufferLength >= sizeof(m_SerialNumber));
        if(m_IsParseFinished)
        {
            memcpy(pOutBuffer, m_SerialNumber, sizeof(m_SerialNumber));
        }
        else
        {
            memset(pOutBuffer, 0, sizeof(m_SerialNumber));
        }
    }

    //---------------------------------------------------------------------------
    //! @brief        Modulus を取得します。ParseForGcAsicCert を呼ぶ前に呼ぶと何もしません。
    //!
    //! @param[out]       pOutBuffer       Modulus を入れる配列のポインタ
    //! @param[in]        outBufferLength  pOutBuffer の確保しているサイズ
    //---------------------------------------------------------------------------
    void GetModulus(char* pOutBuffer, const size_t outBufferLength) NN_NOEXCEPT
    {
        NN_DETAIL_GC_ABORT_UNLESS_SDK_REQUIRES(outBufferLength >= sizeof(m_Modulus));
        if(m_IsParseFinished)
        {
            memcpy(pOutBuffer, m_Modulus, sizeof(m_Modulus));
        }
        else
        {
            memset(pOutBuffer, 0, sizeof(m_Modulus));
        }
    }

    //---------------------------------------------------------------------------
    //! @brief        Exponent を取得します。ParseForGcAsicCert を呼ぶ前に呼ぶと何もしません。
    //!
    //! @param[out]       pOutBuffer       Exponent を入れる配列のポインタ
    //! @param[in]        outBufferLength  pOutBuffer の確保しているサイズ
    //---------------------------------------------------------------------------
    void GetExponent(char* pOutBuffer, const size_t outBufferLength) NN_NOEXCEPT
    {
        NN_DETAIL_GC_ABORT_UNLESS_SDK_REQUIRES(outBufferLength >= sizeof(m_Exponent));
        if(m_IsParseFinished)
        {
            memcpy(pOutBuffer, m_Exponent, sizeof(m_Exponent));
        }
        else
        {
            memset(pOutBuffer, 0, sizeof(m_Exponent));
        }
    }

    //---------------------------------------------------------------------------
    //! @brief        被署名領域が開始するインデックスを取得します。
    //!
    //! @return       被署名領域が開始するインデックス
    //---------------------------------------------------------------------------
    size_t GetTbsCertificateIndex() NN_NOEXCEPT
    {
        if(m_IsParseFinished)
        {
            return m_TbsCertificateIndex;
        }
        else
        {
            return 0;
        }
    }
    //---------------------------------------------------------------------------
    //! @brief        被署名領域のサイズを取得します。
    //!
    //! @return       被署名領域のサイズ
    //---------------------------------------------------------------------------
    size_t GetTbsCertificateLength() NN_NOEXCEPT
    {
        if(m_IsParseFinished)
        {
            return m_TbsCertificateLength;
        }
        else
        {
            return 0;
        }
    }

    //---------------------------------------------------------------------------
    //! @brief        署名領域が開始するインデックスを取得します。
    //!
    //! @return       署名領域が開始するインデックス
    //---------------------------------------------------------------------------
    size_t GetSignatureIndex() NN_NOEXCEPT
    {
        if(m_IsParseFinished)
        {
            return m_SignatureIndex;
        }
        else
        {
            return 0;
        }
    }

    //---------------------------------------------------------------------------
    //! @brief        署名領域のサイズを取得します。
    //!
    //! @return       署名領域のサイズ
    //---------------------------------------------------------------------------
    size_t GetSignatureLength() NN_NOEXCEPT
    {
        if(m_IsParseFinished)
        {
            return m_SignatureLength;
        }
        else
        {
            return 0;
        }
    }


private:
    bool m_IsParseFinished;
    char m_SerialNumber[GcCrypto::GcAsicSeriralNumberLength];
    char m_Modulus[GcCrypto::GcRsaKeyLength];
    char m_Exponent[GcCrypto::GcExponentLength];
    size_t m_TbsCertificateIndex;
    size_t m_TbsCertificateLength;
    size_t m_SignatureAlgorithmIndex;
    size_t m_SignatureAlgorithmLength;
    size_t m_SignatureIndex;
    size_t m_SignatureLength;
    static const size_t Asn1TagInteger             = 2;
    static const size_t Asn1TagBitString           = 3;
    static const size_t Asn1TagSequence            = 16;
    static const size_t Asn1TagConstructed         = 0x20;
    static const size_t Asn1LongLengthBitFlag      = 0x80;
    static const size_t TbsCertBlockNum            = 1;
    static const size_t SerialNumberBlockNum       = 4;
    static const size_t ModulusBlockNum            = 51;
    static const size_t ExponentBlockNum           = 52;
    static const size_t SignatureAlgorithmBlockNum = 54;
    static const size_t SignatureBlockNum          = 56;
    static const size_t LoopNum                    = 57;
    static void  AnalyzeBlock(size_t* pOutNextIndex, size_t* pOutValueIndex, size_t* pOutValueLength, const char* pInData, const size_t index, const bool isSkipDisabled) NN_NOEXCEPT;
};

} } }
