﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/util/util_BitPack.h>
#include <nn/gc/detail/gc_Types.h>
#include <nn/nn_Result.h>

namespace nn { namespace gc {
namespace detail {

typedef nn::util::BitPack32 BitPack32;

struct AsicRegisterErrorInfo
{
    uint16_t gameCardCrcErrorNum;
    uint16_t asicCrcErrorNum;
    uint16_t refreshNum;
    uint16_t gameCardTimeoutErrorNum;
};

class AsicRegister
{
    NN_DISALLOW_COPY(AsicRegister);

public:
    static const int GcAsicRegisterSize = 0x30;
    char* const registerArray;

NN_DETAIL_GC_PRIVATE:
    union
    {
        char m_RegisterArray[ GcAsicRegisterSize ];
        struct
        {
            BitPack32 asicStatus;     // これを m_FirstRegisterAddress としてセットしている
            BitPack32 accessControl1;
            BitPack32 readWait1Time;
            BitPack32 readWait2Time;
            BitPack32 writeWait1Time;
            BitPack32 writeWait2Time;
            BitPack32 pageRemainder;
            BitPack32 latencyTime;
            BitPack32 limitedArea;
            BitPack32 cupVersion;
            BitPack32 asicVersion;
            BitPack32 standby2Control;
        } m_RegisterMap;
    };

    BitPack32* const m_FirstRegisterAddress;

private:
    AsicRegister() NN_NOEXCEPT;

    AsicRegisterErrorInfo m_AsicRegisterErrorInfo;

public:
    // Singleton パターン
    static AsicRegister& GetInstance() NN_NOEXCEPT;

    void ResetRegister() NN_NOEXCEPT;

    nn::Result ClearRegisterErrorStatus() NN_NOEXCEPT;
    // 返り値は ASIC へのレジスタアクセスの成否、引数での出力はレジスタ値の内容による結果
    // （！）結果を見たいオペレーション前に ClearRegisterErrorStatus を読んでおく必要がある
    nn::Result ReadAndCheckRegisterErrorStatus(nn::Result* pOutRegisterResult) NN_NOEXCEPT;

    // ** レジスタ操作
    nn::Result SetLatencyCheckValue(const u16 value) NN_NOEXCEPT;

    void GetRegisterErrorInfo(AsicRegisterErrorInfo* pOutAsicRegisterErrorInfo) NN_NOEXCEPT;

    void ClearErrorInfo() NN_NOEXCEPT;

NN_DETAIL_GC_PRIVATE:
    // レジスタの書き込み
    nn::Result WriteRegister(const BitPack32 *registerValueAddress) NN_NOEXCEPT;
    // レジスタ変数へのアドレスをブリッジASIC上のレジスタアドレスへ変換
    u32 ConvertRegisterValueAddressToRegisterAddress(const BitPack32 *registerValueAddress) NN_NOEXCEPT;

    nn::Result ReadAndCheckRegisterErrorStatusImpl() NN_NOEXCEPT;
    nn::Result CheckAsicErrorStatus();
    nn::Result CheckTimoutError();
    nn::Result CheckCardStatus();
    nn::Result CheckCrcResult();
};

} } }
