﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <vector>
#include <nn/fs/fs_ResultPrivate.h>
#include <nn/fssystem/fs_NcaHeader.h>
#include <nn/fssystem/fs_IndirectStorage.h>
#include <nn/fssystem/utilTool/fs_BinaryMatch.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_IntUtil.h>

namespace nn { namespace fssystem { namespace utilTool {

class RelocationTable;
class RelocationProgress;

/**
 * @brief   IndirectStorage で扱うデータを構築するクラスです。
 */
class RelocatedIndirectStorageBuilder
{
    NN_DISALLOW_COPY(RelocatedIndirectStorageBuilder);

public:
    typedef MemoryResource IAllocator;

    /**
     * @brief   パッチの各種情報です。
     */
    class PatchInfo
    {
    public:
        PatchInfo() NN_NOEXCEPT
            : m_EntryCount(0)
            , m_StorageOffset(0)
            , m_NodeStorage()
            , m_EntryStorage()
            , m_DataStorage()
        {
        }

        PatchInfo(
            int entryCount,
            int64_t storageOffset,
            fs::SubStorage nodeStorage,
            fs::SubStorage entryStorage,
            fs::SubStorage dataStorage
        ) NN_NOEXCEPT
            : m_EntryCount(entryCount)
            , m_StorageOffset(storageOffset)
            , m_NodeStorage(nodeStorage)
            , m_EntryStorage(entryStorage)
            , m_DataStorage(dataStorage)

        {
        }

        template< typename T >
        Result Initialize(const NcaPatchInfo& patchInfo, int64_t storageOffset, T pStorage) NN_NOEXCEPT;

        int GetEntryCount() const NN_NOEXCEPT
        {
            return m_EntryCount;
        }

        int64_t GetStorageOffset() const NN_NOEXCEPT
        {
            return m_StorageOffset;
        }

        fs::SubStorage& GetNodeStorage() NN_NOEXCEPT
        {
            return m_NodeStorage;
        }

        const fs::SubStorage& GetNodeStorage() const NN_NOEXCEPT
        {
            return m_NodeStorage;
        }

        fs::SubStorage& GetEntryStorage() NN_NOEXCEPT
        {
            return m_EntryStorage;
        }

        const fs::SubStorage& GetEntryStorage() const NN_NOEXCEPT
        {
            return m_EntryStorage;
        }

        fs::SubStorage& GetDataStorage() NN_NOEXCEPT
        {
            return m_DataStorage;
        }

        const fs::SubStorage& GetDataStorage() const NN_NOEXCEPT
        {
            return m_DataStorage;
        }

    private:
        int m_EntryCount;
        int64_t m_StorageOffset;
        fs::SubStorage m_NodeStorage;
        fs::SubStorage m_EntryStorage;
        fs::SubStorage m_DataStorage;
    };

public:
    /**
     * @brief   コンストラクタです。
     */
    RelocatedIndirectStorageBuilder() NN_NOEXCEPT;

    /**
     * @brief   デストラクタです。
     */
    ~RelocatedIndirectStorageBuilder() NN_NOEXCEPT;

    /**
     * @brief   初期化をします。
     *
     * @param[in]   oldPatchInfo    古いデータを指すパッチの各種情報
     * @param[in]   newPatchInfo    新しいデータを指すパッチの各種情報
     *
     * @return  実行結果を返します。
     *
     * @pre
     *      - IsInitialize() == false
     *      - IsBuilt() == false
     */
    Result Initialize(const PatchInfo& oldPatchInfo, const PatchInfo& newPatchInfo, int64_t outputDataStorageOffset) NN_NOEXCEPT;

    /**
     * @brief   終了処理をします。
     */
    void Finalize() NN_NOEXCEPT;

    /**
     * @brief   初期化済みかどうかを取得します。
     *
     * @return  初期化済みかどうかを返します。
     */
    bool IsInitialized() const NN_NOEXCEPT
    {
        return m_IsInitialized;
    }

    /**
     * @brief   リージョンハッシュの情報を設定します。
     *
     * @param[in]   region  リージョンハッシュの情報
     */
    void SetBinaryRegion(const BinaryRegionArray& region) NN_NOEXCEPT
    {
        m_BinaryRegion.first = region;
    }

    /**
     * @brief   リージョンハッシュの情報を取得します。
     *
     * @return  リージョンハッシュの情報
     */
    const BinaryRegionArray& GetBinaryRegion() const NN_NOEXCEPT
    {
        return m_BinaryRegion.first;
    }

    /**
     * @brief   テーブル作成済みかどうかを取得します。
     *
     * @return  テーブル作成済みかどうかを返します。
     */
    bool IsBuilt() const NN_NOEXCEPT
    {
        return m_IsBuilt;
    }

    /**
     * @brief   ストレージを解析してテーブルを作成します。
     *
     * @param[in]   pAllocator  アロケータのポインタ
     * @param[in]   blockSize   ハッシュ化するデータのサイズ
     * @param[in]   regionSize  比較するデータの最小単位
     *
     * @return  実行結果を返します。
     *
     * @pre
     *      - pAllocator != nullptr
     *      - 8 <= blockSize
     *      - ispow2(blockSize) != false
     *      - blockSize < regionSize
     *      - ispow2(regionSize) != false
     *      - IsInitialized() != false
     *      - IsBuilt() == false
     */
    Result Build(IAllocator* pAllocator, size_t blockSize, size_t regionSize) NN_NOEXCEPT
    {
        return Build(pAllocator, blockSize, blockSize, regionSize, 0);
    }

    /**
     * @brief   ストレージを解析してテーブルを作成します。
     *
     * @param[in]   pAllocator      アロケータのポインタ
     * @param[in]   shiftSize       ハッシュ化するデータの移動量
     * @param[in]   blockSize       ハッシュ化するデータのサイズ
     * @param[in]   regionSize      比較するデータの最小単位
     * @param[in]   matchSize       マッチするデータの最小単位
     *
     * @return  実行結果を返します。
     *
     * @pre
     *      - pAllocator != nullptr
     *      - 0 < shiftSize
     *      - shiftSize <= blockSize
     *      - ispow2(shiftSize) != false
     *      - 8 <= blockSize
     *      - ispow2(blockSize) != false
     *      - blockSize < regionSize
     *      - ispow2(regionSize) != false
     *      - 0 <= matchSize
     *      - IsInitialized() != false
     *      - IsBuilt() == false
     */
    Result Build(IAllocator* pAllocator, size_t shiftSize, size_t blockSize, size_t regionSize, int64_t matchSize) NN_NOEXCEPT;

    /**
     * @brief   テーブルを書き出します。
     *
     * @param[in]   pAllocator      アロケータのポインタ
     * @param[in]   headerStorage   テーブルヘッダのストレージ
     * @param[in]   nodeStorage     テーブルノードのストレージ
     * @param[in]   entryStorage    テーブルエントリのストレージ
     *
     * @return  実行結果を返します。
     *
     * @pre
     *      - pAllocator != nullptr
     *      - IsBuilt() != false
     */
    Result WriteTable(
               IAllocator* pAllocator,
               fs::SubStorage headerStorage,
               fs::SubStorage nodeStorage,
               fs::SubStorage entryStorage
           ) NN_NOEXCEPT;

    /**
     * @brief   テーブルを元に必要なデータを読み込みます。
     *
     * @param[in]   offset  読み込むデータのオフセット
     * @param[in]   buffer  データを読み込むバッファのアドレス
     * @param[in]   size    読み込むデータのサイズ
     *
     * @return  実行結果を返します。
     *
     * @pre
     *      - IsBuilt() != false
     *      - size == 0 || buffer != nullptr
     *      - nn::fs::IStorage::CheckOffsetAndSize(offset, size) != false
     *
     * offset,size で指定された領域が [0, QueryDataStorageSize()) を超えている場合、
     * その領域は 0 埋めします。
     */
    Result ReadData(int64_t offset, void* buffer, size_t size) NN_NOEXCEPT;

    /**
     * @brief   作成された 逆 RelocationTable を取得します。
     *
     * @return  RelocationTable を返します。
     *
     * @pre
     *      - IsBuilt() != false
     */
    const RelocationTable& GetInvertedRelocationTable() const NN_NOEXCEPT;

    /**
     * @brief   Build() の進捗を取得します。
     */
    std::shared_ptr<BinaryMatchProgress> GetProgress() const NN_NOEXCEPT
    {
        return m_pProgress;
    }

    /**
     * @brief   リージョンハッシュの開始オフセットを取得します。
     *
     * @param[out]  pOutValue   オフセットの格納先
     * @param[in]   blockSize   ハッシュ化するデータのサイズ
     *
     * @return  実行結果を返します。
     */
    Result QueryBinaryRegionOffset(int64_t* pOutValue, size_t blockSize) const NN_NOEXCEPT;

    /**
     * @brief   テーブルでヘッダ部分のストレージのサイズを取得します。
     *
     * @return  ヘッダ部分のストレージサイズを返します。
     */
    int64_t QueryTableHeaderStorageSize() const NN_NOEXCEPT
    {
        return IndirectStorage::QueryHeaderStorageSize();
    }

    /**
     * @brief   テーブルでノード部分のストレージのサイズを取得します。
     *
     * @return  ノード部分のストレージサイズを返します。
     *
     * @pre
     *      - IsBuilt() != false
     */
    int64_t QueryTableNodeStorageSize() const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsBuilt());

        return IndirectStorage::QueryNodeStorageSize(GetEntryCount());
    }

    /**
     * @brief   テーブルでエントリ部分のストレージサイズを取得します。
     *
     * @return  エントリ部分のストレージサイズを返します。
     *
     * @pre
     *      - IsBuilt() != false
     */
    int64_t QueryTableEntryStorageSize() const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsBuilt());

        return IndirectStorage::QueryEntryStorageSize(GetEntryCount());
    }

    /**
     * @brief   テーブルのストレージサイズを取得します。
     *
     * @return  テーブルのストレージサイズを返します。
     *
     * @pre
     *      - IsBuilt() != false
     */
    int64_t QueryTableStorageSize() const NN_NOEXCEPT
    {
        return QueryTableHeaderStorageSize() +
               QueryTableNodeStorageSize() +
               QueryTableEntryStorageSize();
    }

    /**
     * @brief   データのストレージサイズを取得します。
     *
     * @return  データのストレージサイズを返します。
     *
     * @pre
     *      - IsBuilt() != false
     */
    int64_t QueryDataStorageSize() const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(IsBuilt());

        return m_RelocatedDataSize;
    }

    /**
     * @brief   RelocatedIndirectStorage のビルド情報を出力します。
     *
     * @param[in]   pAllocator      アロケータのポインタ
     * @param[in]   pTableStorage   テーブル情報を書き出すストレージのポインタ
     * @param[in]   pNodeStorage    ノード情報を書き出すストレージのポインタ
     * @param[in]   pEntryStorage   エントリ情報を書き出すストレージのポインタ
     * @param[in]   pDataStorage    データ情報を書き出すストレージのポインタ
     * @param[in]   startOffset     IndirectStorage の開始オフセット
     *
     * @pre
     *      - pAllocator != nullptr
     *      - pTableStorage != nullptr
     *      - pNodeStorage != nullptr
     *      - pEntryStorage != nullptr
     *      - pDataStorage != nullptr
     *      - 0 <= startOffset
     */
    Result OutputBuildLog(
               IAllocator* pAllocator,
               fs::IStorage* pTableStorage,
               fs::IStorage* pNodeStorage,
               fs::IStorage* pEntryStorage,
               fs::IStorage* pDataStorage,
               int64_t startOffset
           ) const NN_NOEXCEPT;

private:
    typedef BinaryMatch::RegionInfo RegionInfo;

    // IndirectStorage のエントリ数を取得します。
    int GetEntryCount() const NN_NOEXCEPT
    {
        NN_SDK_ASSERT(util::IsIntValueRepresentable<int>(m_RelocatedEntries.size()));
        return static_cast<int>(m_RelocatedEntries.size());
    }

    // RelocationTable をストレージに適用します。
    Result BuildImpl(
               IAllocator* pAllocator,
               const RelocationTable& relocationTable
           ) NN_NOEXCEPT;

private:
    bool m_IsInitialized;
    bool m_IsBuilt;
    PatchInfo m_OldPatchInfo;
    PatchInfo m_NewPatchInfo;
    RegionInfo m_BinaryRegion;
    int64_t m_OutputDataStorageOffset;
    int64_t m_OldDataStorageSize;
    int64_t m_NewDataStorageSize;
    int64_t m_RelocatedDataSize;
    int64_t m_IndirectTableSize;
    std::unique_ptr<RelocationTable> m_pInvertedRelocationTable;
    std::vector<IndirectStorage::Entry> m_RelocatedEntries;
    std::shared_ptr<BinaryMatchProgress> m_pProgress;

    friend class IndirectStorageBuilderTest;
};

template< typename T >
Result RelocatedIndirectStorageBuilder::PatchInfo::Initialize(const NcaPatchInfo& patchInfo, int64_t storageOffset, T pStorage) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_GREATER_EQUAL(storageOffset, 0);
    NN_SDK_REQUIRES_NOT_NULL(pStorage);

    NN_RESULT_THROW_UNLESS(patchInfo.HasIndirectTable(), fs::ResultInvalidArgument());

    BucketTree::Header header;
    std::memcpy(&header, patchInfo.indirectHeader, sizeof(header));
    NN_RESULT_DO(header.Verify());

    const auto nodeSize = IndirectStorage::QueryNodeStorageSize(header.entryCount);
    const auto entrySize = IndirectStorage::QueryEntryStorageSize(header.entryCount);
    NN_RESULT_THROW_UNLESS(nodeSize + entrySize <= patchInfo.indirectSize, fs::ResultInvalidIndirectStorageSize());

    const auto dataSize = patchInfo.indirectOffset;
    const auto nodeOffset = dataSize;
    const auto entryOffset = nodeOffset + nodeSize;

    m_EntryCount = header.entryCount;
    m_StorageOffset = storageOffset;
    m_NodeStorage = fs::SubStorage(pStorage, nodeOffset, nodeSize);
    m_EntryStorage = fs::SubStorage(pStorage, entryOffset, entrySize);
    m_DataStorage = fs::SubStorage(pStorage, 0, dataSize);

    NN_RESULT_SUCCESS;
}

}}}
