﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <array>
#include <vector>
#include <nn/fssystem/fs_NcaHeader.h>
#include <nn/fssystem/save/fs_BlockCacheBufferedStorage.h>

namespace nn { namespace fssystem {

namespace save {

struct HierarchicalIntegrityVerificationInformation;

}

namespace utilTool {

/**
 * @brief   指定した領域が含まれるハッシュを収集するクラスです。
 */
class BlockSignatureContainer
{
public:
    typedef save::HierarchicalIntegrityVerificationInformation SaveDataLayerInfo;
    typedef NcaFsHeader::HashData::IntegrityMetaInfo::InfoLevelHash NcaRomFsLayerInfo;

    // 各レイヤーの領域
    struct Block
    {
        int64_t size;        // 領域のサイズ
        int64_t position;    // 全レイヤーからのオフセット
        int64_t layerOffset; // 所属するレイヤーでのオフセット
        int layerIndex;      // 所属するレイヤーのインデックス
    };
    NN_STATIC_ASSERT(std::is_pod<BlockSignatureContainer::Block>::value);

    // コンストラクタです。
    BlockSignatureContainer() NN_NOEXCEPT;

    // 初期化します。
    void Initialize(const SaveDataLayerInfo& info) NN_NOEXCEPT;

    // 初期化します。
    void Initialize(const NcaRomFsLayerInfo& info) NN_NOEXCEPT;

    // 指定領域のハッシュを収集します。
    void Collect(int64_t dataOffset, int64_t dataSize) NN_NOEXCEPT;

    // 収集したデータをまとめます。
    void Optimize() NN_NOEXCEPT;

    // 収集したデータを取得します。
    const std::vector<Block>& GetBlock() const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(m_IsOptimized);
        return m_Blocks;
    }

private:
    // レイヤー情報
    struct Layer
    {
        int64_t start;
        int64_t width;
        int hashBlockSize;
    };
    NN_STATIC_ASSERT(std::is_pod<BlockSignatureContainer::Layer>::value);

    static const int LayerCountMin = 2;
    static const int LayerCountMax = 6;

private:
    // 初期化します。
    template< typename T >
    void InitializeImpl(const T& info) NN_NOEXCEPT;

    // データ部分のブロックを追加します。
    bool AddDataBlock(const Block& block) NN_NOEXCEPT;

    // ハッシュ部分のブロックを追加します。
    void AddHashBlock(const Block& dataBlock) NN_NOEXCEPT;

private:
    std::array<Layer, LayerCountMax> m_Layers;
    int m_LayerCount;
    bool m_IsOptimized;
    std::vector<Block> m_Blocks;
};

}}}
