﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <memory>

#include <nn/nn_Common.h>
#include <nn/os.h>
#include <nn/fs/fs_Result.h>

#include <nn/fs/fs_IStorage.h>
#include <nn/fs/fs_SubStorage.h>

namespace nn { namespace fssystem { namespace save {

//! 書き込んだ内容をログ領域と呼ばれる一時的なストレージに記録し、まとめて反映するストレージです
class UnionStorage : public fs::IStorage
{
public:
    /**
    * @brief        コンストラクタです。
    */
    UnionStorage() NN_NOEXCEPT
    : m_StorageOriginal(),
      m_StorageLog(),
      m_SizeBlock(0),
      m_Buffer(),
      m_CountLogs(0),
      m_Mutex(false)
    {
    }

    /**
    * @brief        デストラクタです。
    */
    virtual ~UnionStorage() NN_NOEXCEPT NN_OVERRIDE
    {
    }

    /**
    * @brief        ログ領域をフォーマットします。
    *
    * @param[in]    storageLog  書き込みをロギングするストレージ
    * @param[in]    sizeBlock   ブロックのサイズ
    *
    * @return       関数の処理結果を返します。
    *
    * @pre
    *               - storageLog のサイズが 0 より大きい
    *               - sizeBlock が 2 のべき乗
    */
    static Result Format(
                      fs::SubStorage storageLog,
                      int64_t sizeBlock
                  ) NN_NOEXCEPT;

    /**
    * @brief        ユニオンストレージとしてマウントします。
    *
    * @param[in]    storageOriginal  基礎となるストレージ
    * @param[in]    storageLog       書き込みをバッファリングするストレージ
    * @param[in]    sizeBlock        ブロックのサイズ
    *
    * @return       関数の処理結果を返します。
    *
    * @pre
    *               - ストレージが未初期化である
    */
    Result Initialize(
               fs::SubStorage storageOriginal,
               fs::SubStorage storageLog,
               int64_t sizeBlock
           ) NN_NOEXCEPT;

    /**
    * @brief        ログストレージに終端を示す値を書き込み、再マウントできる状態にします。
    *
    * @return       関数の処理結果を返します。
    *
    * @pre
    *               - ストレージが初期化済みである
    */
    Result Freeze() NN_NOEXCEPT;

    /**
    * @brief        ログストレージにバッファリングした書き込みをオリジナルストレージに反映します。
    *
    * @return       関数の処理結果を返します。
    *
    * @pre
    *               - ストレージが初期化済みである
    *
    * @details      オリジナルストレージへの反映は常にブロック単位で行われます。
    *               書き込みを行いたい範囲の最初または最後のオフセットが
    *               ブロックサイズでアライメントされていない場合、
    *               その範囲の前後の書き込みを意図していない領域も含めて
    *               ロギングされることになります。
    *               もし、書き込みを行ってからコミットを行うまでの間に、
    *               本ストレージを介さずにその範囲外の領域に書き込みを行っていたとすると、
    *               コミット操作により元の値に上書きされてしまう点に注意してください。
    */
    Result Commit() NN_NOEXCEPT;

    /**
    * @brief        ストレージの内容をバッファに読み込みます。
    *
    * @param[in]    offset  読み込み開始位置
    * @param[out]   buffer  読み込んだ内容をコピーするバッファ
    * @param[in]    size    読み込むデータサイズ
    *
    * @return       関数の処理結果を返します。
    *
    * @pre
    *               - ストレージが初期化済みである
    */
    virtual Result Read(int64_t offset, void* buffer, size_t size) NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief        バッファの内容をストレージに書き込みます。
    *
    * @param[in]    offset  書き込み開始位置
    * @param[in]    buffer  書き込むデータ
    * @param[in]    size    書き込むデータサイズ
    *
    * @return       関数の処理結果を返します。
    *
    * @pre
    *               - ストレージが初期化済みである
    */
    virtual Result Write(int64_t offset, const void* buffer, size_t size) NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief        フラッシュします。
    *
    * @return       関数の処理結果を返します。
    *
    * @pre
    *               - ストレージが初期化済みである
    *
    * @details      基礎となるストレージに加えて、ログ記録用ストレージもフラッシュします。
    */
    virtual Result Flush() NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief        ストレージのサイズを取得します。
    *
    * @param[out]   outValue    取得したサイズ格納先
    *
    * @return       関数の処理結果を返します。
    *
    * @pre
    *               - ストレージが初期化済みである
    *
    * @details      基礎となるストレージのサイズを取得します。
    *               ログ記録用ストレージのサイズを取得することはできません。
    */
    virtual Result GetSize(int64_t* outValue) NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief       範囲指定処理を行います。
    *
    * @param[out]  outBuffer        範囲指定処理の結果を格納するバッファ
    * @param[in]   outBufferSize    範囲指定処理の結果を格納するバッファのサイズ
    * @param[in]   operationId      範囲指定処理の種類
    * @param[in]   offset           範囲指定処理開始位置
    * @param[in]   size             範囲指定処理を行うデータサイズ
    * @param[in]   inBuffer         範囲指定処理に渡すバッファ
    * @param[in]   inBufferSize     範囲指定処理に渡すバッファのサイズ
    *
    * @return      関数の処理結果を返します。
    */
    virtual Result OperateRange(
                       void* outBuffer,
                       size_t outBufferSize,
                       fs::OperationId operationId,
                       int64_t offset,
                       int64_t size,
                       const void* inBuffer,
                       size_t inBufferSize
                   ) NN_NOEXCEPT NN_OVERRIDE;

private:
    /**
    * @brief        指定のオフセットのデータが書かれたログを探索します。
    *
    * @param[out]   outFound       ログが見つかったかどうかの格納先
    * @param[out]   outOffsetLog   ログのオフセットの格納先
    * @param[in]    offsetOriginal 探索するオフセット
    *
    * @return       関数の処理結果を返します。
    *
    * @pre
    *               - ストレージが初期化済みである
    *
    * @details      ログが見つかった場合 ResultSuccess が返り、
    *               outFound に true が, outOffsetLog にはそのオフセットが格納されます。
    *               ログが見つからなかった場合 ResultSuccess が返り、
    *               outFound に false が, outOffsetLog には
    *               次にログデータを書き込むべきオフセットが格納されます。
    *               いずれの場合も、格納されるのはログデータの先頭のオフセットであり、
    *               実データはそこから sizeof(int64_t) だけ進んだ先に位置する点に注意してください。
    *               途中で読み込みに失敗した場合、エラーリザルトを返し、格納される値は不定です。
    */
    Result FindLog(bool* outFound, int64_t* outOffsetLog, int64_t offsetOriginal) NN_NOEXCEPT;

private:
    fs::SubStorage                       m_StorageOriginal;    //!< 書き込みをバッファリングするもとのストレージです
    fs::SubStorage                       m_StorageLog;         //!< バッファリングした書き込みログを出力するストレージです
    int64_t                              m_SizeBlock;          //!< バッファリングするブロックのサイズです
    std::unique_ptr<char[], nn::fs::detail::Deleter> m_Buffer; //!< 書き込み時に一時的にデータを書き込むためのバッファです
    int                                  m_CountLogs;          //!< 書き込まれているログの数です
    nn::os::Mutex                        m_Mutex;              //!< 書き込み処理を排他するためのミューテックスです
};

}}}
