﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/fs/fs_SubStorage.h>
#if defined(USE_RANGE_LOCK)
    #include <nn/fssystem/fs_RangeLock.h>
#endif // defined(USE_RANGE_LOCK)
#include <nn/fssystem/save/fs_MappingTable.h>

namespace nn { namespace fssystem { namespace save {

/**
* @brief マッピングテーブルレイヤー
*/
class JournalStorage : public fs::IStorage
{
public:
    //! マッピングテーブルの管理領域
    struct ControlArea
    {
        uint32_t signature;                     //! 識別子
        uint32_t version;                       //! バージョン
        int64_t areaSize;                       //! 物理アドレス領域のサイズ
        int64_t reservedAreaSize;               //! 予約領域のサイズ
        int64_t sizeBlock;                      //! ブロックサイズ
        MappingTable::ControlArea mappingInfo;  //! マッピングテーブルの管理領域
        uint8_t _reserved[464];
    };
    NN_STATIC_ASSERT(sizeof(ControlArea) == 512);
    NN_STATIC_ASSERT(std::is_pod<ControlArea>::value);

private:
    /*!
    * @brief        値を指定したアライメントまで切り下げます。
    *
    * @param[in]    value       値
    * @param[in]    align       アライメント
    *
    * @return       アライメント後の値を返します。
    */
    inline int64_t AlignDown64(int64_t value, int64_t align) NN_NOEXCEPT
    {
        return (value & ~(align - 1));
    }

    /*!
    * @brief        値を指定したアライメントまで切り上げます。
    *
    * @param[in]    value       値
    * @param[in]    align       アライメント
    *
    * @return       アライメント後の値を返します。
    */
    inline int64_t AlignUp64(int64_t value, int64_t align) NN_NOEXCEPT
    {
        return ((value + align - 1) & ~(align - 1LL));
    }

public:
    //! コンストラクタ
    JournalStorage() NN_NOEXCEPT;

    //! デストラクタ
    virtual ~JournalStorage() NN_NOEXCEPT;

public:
    /**
    * @brief        マッピングテーブルのメタデータサイズを取得します。
    *
    * @param[out]   outTableSize                    マッピングテーブルに必要なサイズ
    * @param[out]   outBitmapSizeUpdatedPhysical    更新済みビットマップ(物理インデックス)に必要なサイズ
    * @param[out]   outBitmapSizeUpdatedVirtual     更新済みビットマップ(論理インデックス)に必要なサイズ
    * @param[out]   outBitmapSizeUnassigned         未割当ビットマップ(物理インデックス)に必要なサイズ
    * @param[in]    sizeArea                        データ領域の総サイズ
    * @param[in]    sizeReservedArea                予約領域のサイズ
    * @param[in]    sizeBlock                       ブロックサイズ
    */
    static void QueryMappingMetaSize(
                    int64_t* outTableSize,
                    int64_t* outBitmapSizeUpdatedPhysical,
                    int64_t* outBitmapSizeUpdatedVirtual,
                    int64_t* outBitmapSizeUnassigned,
                    int64_t sizeArea,
                    int64_t sizeReservedArea,
                    int64_t sizeBlock
                ) NN_NOEXCEPT;

    /**
    * @brief        マッピングテーブルメタデータ領域をフォーマットします。
    *
    * @param[in]    storageControlArea              管理領域を配置するレイヤー
    * @param[in]    storageTable                    マッピングテーブルのメタデータを配置するレイヤー
    * @param[in]    storageBitmapUpdatedPhysical    更新済みビットマップ(物理インデックス)を配置するレイヤー
    * @param[in]    storageBitmapUpdatedVirtual     更新済みビットマップ(論理インデックス)を配置するレイヤー
    * @param[in]    storageBitmapUnassigned         未割当ビットマップ(物理インデックス)を配置するレイヤー
    * @param[in]    sizeArea                        データ領域の総サイズ
    * @param[in]    sizeReservedArea                予約領域のサイズ
    * @param[in]    sizeBlock                       1ブロックのサイズ
    *
    * @return       関数の処理結果を返します。
    */
    static Result Format(
                      fs::SubStorage storageControlArea,
                      fs::SubStorage storageTable,
                      fs::SubStorage storageBitmapUpdatedPhysical,
                      fs::SubStorage storageBitmapUpdatedVirtual,
                      fs::SubStorage storageBitmapUnassigned,
                      int64_t sizeArea,
                      int64_t sizeReservedArea,
                      int64_t sizeBlock
                  ) NN_NOEXCEPT;

    /**
    * @brief        マッピングテーブルメタデータ領域を拡張します。
    *
    * @param[in]    storageControlArea              管理領域を配置するレイヤー
    * @param[in]    storageTable                    マッピングテーブルを配置するレイヤー
    * @param[in]    storageBitmapUpdatedPhysical    更新済みビットマップ(物理インデックス)を配置するレイヤー
    * @param[in]    storageBitmapUpdatedVirtual     更新済みビットマップ(論理インデックス)を配置するレイヤー
    * @param[in]    storageBitmapUnassigned         未割当ビットマップ(物理インデックス)を配置するレイヤー
    * @param[in]    sizeAreaNew                     拡張後のデータ領域の総サイズ
    * @param[in]    sizeReservedAreaNew             拡張後の予約領域のサイズ
    *
    * @return       関数の処理結果を返します。
    *
    * @details      ストレージが管理する領域を拡張します。
    *               マウントしていない状態で実行してください。
    */
    static Result Expand(
                      fs::SubStorage storageControlArea,
                      fs::SubStorage storageTable,
                      fs::SubStorage storageBitmapUpdatedPhysical,
                      fs::SubStorage storageBitmapUpdatedVirtual,
                      fs::SubStorage storageBitmapUnassigned,
                      int64_t sizeAreaNew,
                      int64_t sizeReservedAreaNew
                  ) NN_NOEXCEPT;

    /**
    * @brief        マッピングテーブルレイヤーとしてマウントします。
    *
    * @param[in]    storageControlArea              管理領域を配置するレイヤー
    * @param[in]    storageTable                    マッピングテーブルを配置するレイヤー
    * @param[in]    storageBitmapUpdatedPhysical    更新済みビットマップ(物理インデックス)を配置するレイヤー
    * @param[in]    storageBitmapUpdatedVirtual     更新済みビットマップ(論理インデックス)を配置するレイヤー
    * @param[in]    storageBitmapUnassigned         未割当ビットマップ(物理インデックス)を配置するレイヤー
    * @param[in]    storageData                     実データを配置するレイヤー
    *
    * @return       関数の処理結果を返します。
    */
    Result Initialize(
               fs::SubStorage storageControlArea,
               fs::SubStorage storageTable,
               fs::SubStorage storageBitmapUpdatedPhysical,
               fs::SubStorage storageBitmapUpdatedVirtual,
               fs::SubStorage storageBitmapUnassigned,
               fs::SubStorage storageData
           ) NN_NOEXCEPT;

    /**
    * @brief        コミット操作を実行します。
    *
    * @return       関数の処理結果を返します。
    */
    Result Commit() NN_NOEXCEPT;

    /**
    * @brief        変更を巻き戻します。
    *
    * @return       関数の処理結果を返します。
    */
    Result Rollback() NN_NOEXCEPT;

    /**
    * @brief        マッピングテーブル経由でデータを書き込みます。
    *
    * @param[in]    offset  書き込み開始位置
    * @param[in]    buffer  書き込むデータ
    * @param[in]    size    書き込むデータサイズ
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result Write(int64_t offset, const void* buffer, size_t size) NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief        マッピングテーブル経由でデータを読み込みます。
    *
    * @param[in]    offset  読み込み開始位置
    * @param[out]   buffer  読み込んだ内容をコピーするバッファ
    * @param[in]    size    読み込むデータサイズ
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result Read(int64_t offset, void* buffer, size_t size) NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief        書き込みのフラッシュを行ないます。
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result Flush() NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief        ファイルサイズを変更します。
    *
    * @param[in]    size    変更後のファイルサイズ
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result SetSize(int64_t size) NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief        ファイルサイズを取得します。
    *
    * @param[out]   outValue    ファイルサイズ
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result GetSize(int64_t* outValue) NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief       範囲指定処理を行います。
    *
    * @param[out]  outBuffer        範囲指定処理の結果を格納するバッファ
    * @param[in]   outBufferSize    範囲指定処理の結果を格納するバッファのサイズ
    * @param[in]   operationId      範囲指定処理の種類
    * @param[in]   offset           範囲指定処理開始位置
    * @param[in]   size             範囲指定処理を行うデータサイズ
    * @param[in]   inBuffer         範囲指定処理に渡すバッファ
    * @param[in]   inBufferSize     範囲指定処理に渡すバッファのサイズ
    *
    * @return      関数の処理結果を返します。
    */
    virtual Result OperateRange(
        void* outBuffer,
        size_t outBufferSize,
        fs::OperationId operationId,
        int64_t offset,
        int64_t size,
        const void* inBuffer,
        size_t inBufferSize) NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief        データ領域にアクセスするためのストレージを取得します。
    *
    * @return       データ領域にアクセスするためのストレージを返します。
    */
    const fs::SubStorage CreateDataStorage() NN_NOEXCEPT
    {
        return fs::SubStorage(this, 0, m_ControlArea.areaSize - m_ControlArea.reservedAreaSize);
    }

    /**
    * @brief        予約領域を除くデータ領域のサイズを取得します。
    *
    * @return       予約領域を除くデータ領域のサイズを返します。
    */
    int64_t GetDataAreaSize() const NN_NOEXCEPT
    {
        return m_ControlArea.areaSize - m_ControlArea.reservedAreaSize;
    }

    /**
    * @brief        予約領域のサイズを取得します。
    *
    * @return       予約領域のサイズを返します。
    */
    int64_t GetReservedAreaSize() const NN_NOEXCEPT
    {
        return m_ControlArea.reservedAreaSize;
    }

    /**
    * @brief        ブロックサイズを取得します。
    *
    * @return       ブロックサイズを返します。
    */
    int64_t GetBlockSize() const NN_NOEXCEPT
    {
        return m_ControlArea.sizeBlock;
    }

    /**
    * @brief        マッピングテーブルの変更を凍結するかどうかを設定します。
    *
    * @param[in]    isMappingTableFrozon  マッピングの変更を凍結するかどうか
    */
    void SetMappingTableFrozon(bool isMappingTableFrozon) NN_NOEXCEPT
    {
        m_IsMappingTableFrozon = isMappingTableFrozon;
    }

private:
    /**
    * @brief        アドレスをブロックサイズに切り下げます。
    *
    * @param[in]    address アドレス
    *
    * @return       アライン済みアドレスを返します。
    */
    inline int64_t AlignAddress(int64_t address) NN_NOEXCEPT
    {
        return AlignDown64(address, GetBlockSize());
    }

    /**
    * @brief        アドレスをブロックサイズに切り上げます。
    *
    * @param[in]    address アドレス
    *
    * @return       アライン済みアドレスを返します。
    */
    inline int64_t AlignUpAddress(int64_t address) NN_NOEXCEPT
    {
        return AlignUp64(address, GetBlockSize());
    }

    /**
    * @brief        マッピングテーブル上の論理位置を物理アドレスに変換します。
    *
    * @param[out]   outValue    物理アドレス
    * @param[in]    index       マッピングテーブル上の論理位置
    *
    * @return       関数の処理結果を返します。
    */
    Result GetPhysicalAddress(
               int64_t* outValue,
               MappingTable::Index index
           ) NN_NOEXCEPT;

    /**
    * @brief        データ書き込み前にマッピングテーブルを更新します。
    *
    * @param[in]    offset  書き込み開始位置
    * @param[in]    size    書き込むデータサイズ
    *
    * @return       関数の処理結果を返します。
    */
    Result MarkUpdate(int64_t offset, int64_t size) NN_NOEXCEPT;

    /**
    * @brief        マッピングテーブルエントリーのイテレーションを行います。
    *
    * @param[in]    offset  イテレーション開始位置
    * @param[in]    size    データサイズ
    * @param[in]    func    エントリーごとに行う処理
    *
    * @return       関数の処理結果を返します。
    */
    template<typename TFunc>
    Result IterateMappingTable(int64_t offset, int64_t size, TFunc func) NN_NOEXCEPT;

private:
    ControlArea m_ControlArea;          //! 管理領域
    MappingTable m_MappingTable;        //! マッピングテーブル
    fs::SubStorage m_StorageData;       //! 実データ領域
    bool m_IsMappingTableFrozon;        //! マッピングテーブルの変更が凍結されているかどうか
#if defined(USE_RANGE_LOCK)
    RangeReaderWriterLock m_Lock;       //! リーダーライターロック
#endif // defined(USE_RANGE_LOCK)
};

}}} // namespace nn::fssystem::save

