﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>

#include <nn/fs/fs_SubStorage.h>
#include <nn/fssystem/save/fs_ProxyFileSystemWithRetryingBufferAllocation.h>
#include <nn/fssystem/save/fs_SaveStorageFile.h>
#include <nn/fssystem/save/fs_IExclusiveFileSystem.h>
#include <nn/fssystem/save/fs_IInternalStorageFileSystemVisitor.h>
#include <nn/fssystem/save/fs_SaveDataFileSystemCore.h>
#include <nn/fssystem/save/fs_JournalIntegritySaveDataFileSystem.h>
#include <nn/fssystem/fs_IMacGenerator.h>

namespace nn { namespace fssystem { namespace save {

// セーブデータ管理領域のためのパラメータ
struct JournalIntegritySaveDataParameters
{
    save::HierarchicalDuplexStorageControlArea::InputParam                paramDuplex;
    save::HierarchicalIntegrityVerificationStorageControlArea::InputParam paramIntegrity;
    uint32_t         countDataBlock;
    uint32_t         countJournalBlock;
    int32_t          blockSize;
    int              countExpandMax;
    uint32_t         version;
};
NN_STATIC_ASSERT(std::is_pod<JournalIntegritySaveDataParameters>::value);

// セーブデータクラス
class JournalIntegritySaveDataFileSystemDriver : public ProxyFileSystemWithRetryingBufferAllocation, public IInternalStorageFileSystem
{
    NN_DISALLOW_COPY(JournalIntegritySaveDataFileSystemDriver);

public:
    JournalIntegritySaveDataFileSystemDriver() NN_NOEXCEPT;

    virtual ~JournalIntegritySaveDataFileSystemDriver() NN_NOEXCEPT;

    //! 空き領域のサイズをバイト数単位で取得します。
    virtual Result GetFreeBytes(int64_t* outValue) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_RESULT_DO(buffers::DoContinouslyUntilBufferIsAllocated(
            [=]() NN_NOEXCEPT -> Result
            {
                buffers::ScopedBufferManagerContextRegistration scopedRegistration;
                NN_RESULT_DO(m_SaveDataFileSystem.GetFreeBytes(outValue));
                NN_RESULT_SUCCESS;
            },
            NN_CURRENT_FUNCTION_NAME
        ));
        NN_RESULT_SUCCESS;
    }

    //! 実データ領域のサイズをバイト数単位で取得します。
    virtual Result GetDataAreaBytes(int64_t* outValue) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_RESULT_DO(buffers::DoContinouslyUntilBufferIsAllocated(
            [=]() NN_NOEXCEPT -> Result
            {
                buffers::ScopedBufferManagerContextRegistration scopedRegistration;
                NN_RESULT_DO(m_SaveDataFileSystem.GetDataAreaBytes(outValue));
                NN_RESULT_SUCCESS;
            },
            NN_CURRENT_FUNCTION_NAME
        ));
        NN_RESULT_SUCCESS;
    }

    /**
    * @brief        セーブデータ領域のサイズから実データブロック数を求めます。
    *
    * @param[out]   outCountDataBlock   実データブロックサイズを格納する領域
    * @param[in]    sizeTotal           セーブデータ領域のサイズ
    * @param[in]    sizeReservedArea    予約領域のサイズ
    * @param[in]    sizeBlock           ブロックサイズ
    * @param[in]    countExpandMax      拡張できる回数の最大
    * @param[in]    paramDuplex         二重化パラメータ
    * @param[in]    paramIntegrity      完全性検証パラメータ
    *
    * @return       関数の処理結果を返します。
    *
    * @pre
    *               - outCountDataBlock != nullptr
    */
    static Result QueryDataBlockCount(
                      uint32_t* outCountDataBlock,
                      int64_t sizeTotal,
                      int64_t sizeReservedArea,
                      size_t sizeBlock,
                      int countExpandMax,
                      const HierarchicalDuplexStorageControlArea::InputParam& paramDuplex,
                      const HierarchicalIntegrityVerificationStorageControlArea::InputParam& paramIntegrity
                  ) NN_NOEXCEPT;

    /**
    * @brief        セーブデータ ファイルシステムをフォーマットするのに必要なデータサイズを計算します。
    *
    * @param[out]   outSizeTotal        ファイルシステムを使用するために最適なサイズを格納する領域
    * @param[in]    sizeBlock           ブロックサイズ
    * @param[in]    countExpandMax      拡張できる回数の最大
    * @param[in]    paramDuplex         二重化パラメータ
    * @param[in]    paramIntegrity      完全性検証パラメータ
    * @param[in]    countDataBlock      データブロック数
    * @param[in]    countReservedBlock  予約領域のブロック数
    *
    * @return       関数の処理結果を返します。
    *
    * @pre
    *               - outSizeTotal != nullptr
    */
    static Result QueryTotalSize(
                      int64_t* outSizeTotal,
                      size_t sizeBlock,
                      int countExpandMax,
                      const HierarchicalDuplexStorageControlArea::InputParam& paramDuplex,
                      const HierarchicalIntegrityVerificationStorageControlArea::InputParam& paramIntegrity,
                      uint32_t countDataBlock,
                      uint32_t countReservedBlock
                  ) NN_NOEXCEPT;

    /**
    * @brief        セーブデータ ファイルシステムをフォーマットするのに必要なデータサイズを計算します。
    *
    * @param[out]   outSizeTotal        ファイルシステムを使用するために最適なサイズを格納する領域
    * @param[in]    sizeBlock           ブロックサイズ
    * @param[in]    countExpandMax      拡張できる回数の最大
    * @param[in]    paramDuplex         二重化パラメータ
    * @param[in]    paramIntegrity      完全性検証パラメータ
    * @param[in]    countDataBlock      データブロック数
    * @param[in]    countReservedBlock  予約領域のブロック数
    * @param[in]    version             セーブデータバージョン
    *
    * @return       関数の処理結果を返します。
    *
    * @pre
    *               - outSizeTotal != nullptr
    */
    static Result QueryTotalSize(
                      int64_t* outSizeTotal,
                      size_t sizeBlock,
                      int countExpandMax,
                      const HierarchicalDuplexStorageControlArea::InputParam& paramDuplex,
                      const HierarchicalIntegrityVerificationStorageControlArea::InputParam& paramIntegrity,
                      uint32_t countDataBlock,
                      uint32_t countReservedBlock,
                      uint32_t version
                  ) NN_NOEXCEPT;

    /**
    * @brief        セーブデータ領域をフォーマットします。
    *
    * @param[in]    baseStorage         セーブデータ領域として使用するストレージ
    * @param[in]    sizeBlock           ブロックサイズ
    * @param[in]    countExpandMax      拡張できる回数の最大
    * @param[in]    paramDuplex         二重化パラメータ
    * @param[in]    paramIntegrity      完全性検証パラメータ
    * @param[in]    countDataBlock      データブロック数
    * @param[in]    countReservedBlock  予約領域のブロック数
    * @param[in]    pBufferManager      バッファマネージャ
    * @param[in]    pMacGenerator       MAC 生成クラス
    *
    * @return       関数の処理結果を返します。
    *
    * @pre
    *               - pBufferManager != nullptr
    */
    static Result Format(
                      fs::SubStorage baseStorage,
                      size_t sizeBlock,
                      int countExpandMax,
                      const HierarchicalDuplexStorageControlArea::InputParam& paramDuplex,
                      const HierarchicalIntegrityVerificationStorageControlArea::InputParam& paramIntegrity,
                      uint32_t countDataBlock,
                      uint32_t countReservedBlock,
                      IBufferManager* pBufferManager,
                      IMacGenerator* pMacGenerator,
                      const nn::fs::SaveDataHashSalt& hashSalt
                  ) NN_NOEXCEPT;

    /**
    * @brief        セーブデータ ファイルシステムを拡張するのに必要なログ領域のサイズを計算します。
    *
    * @param[in]    sizeBlock           ブロックサイズ
    * @param[in]    countDataBlock      拡張後のデータブロック数
    * @param[in]    countReservedBlock  拡張後の予約領域のブロック数
    *
    * @return       ファイルシステムを拡張するのに必要なログ領域のサイズ
    */
    static int64_t QueryExpandLogSize(
                       size_t sizeBlock,
                       uint32_t countDataBlock,
                       uint32_t countReservedBlock
                   ) NN_NOEXCEPT;

    /**
    * @brief        セーブデータ領域を拡張します。
    *
    * @param[in]    baseStorage         セーブデータ領域として使用するストレージ
    * @param[in]    logStorage          ログを記録するストレージ
    * @param[in]    sizeBlock           ブロックサイズ
    * @param[in]    countDataBlock      拡張後のデータブロック数
    * @param[in]    countReservedBlock  拡張後の予約領域のブロック数
    * @param[in]    pBufferManager      バッファマネージャ
    * @param[in]    pMacGenerator       MAC 生成クラス
    * @param[in]    minimumVersion      最低バージョン
    *
    * @return       関数の処理結果を返します。
    *
    * @pre
    *               - pBufferManager != nullptr
    *
    * @details      本関数は変更をログ領域に書き込み、セーブデータを変更しません。
    *               変更を反映するためには CommitExpand() を呼び出します。
    */
    static Result OperateExpand(
                      fs::SubStorage baseStorage,
                      fs::SubStorage logStorage,
                      size_t sizeBlock,
                      uint32_t countDataBlock,
                      uint32_t countReservedBlock,
                      IBufferManager* pBufferManager,
                      IMacGenerator* pMacGenerator,
                      uint32_t minimumVersion
                  ) NN_NOEXCEPT;

    /**
    * @brief        セーブデータ領域の拡張を反映します。
    *
    * @param[in]    baseStorage     セーブデータ領域として使用するストレージ
    * @param[in]    logStorage      ログを記録したストレージ
    * @param[in]    sizeBlock       ブロックサイズ
    * @param[in]    pBufferManager  バッファマネージャ。将来必要になる場合のために予約されています
    *
    * @return       関数の処理結果を返します。
    */
    static Result CommitExpand(
                      fs::SubStorage baseStorage,
                      fs::SubStorage logStorage,
                      size_t sizeBlock,
                      IBufferManager* pBufferManager
                  ) NN_NOEXCEPT;

    /**
    * @brief        拡張データを読み込みます
    *
    * @param[out]   outData        拡張データ
    * @param[in]    baseStorage    セーブデータ領域として使用するストレージ
    * @param[in]    pBufferManager バッファマネージャ
    * @param[in]    pMacGenerator  MAC 生成クラス
    * @param[in]    minimumVersion 最低バージョン
    *
    * @return       関数の処理結果を返します。
    *
    * @details      最後にコミットした際のデータを読み込みます。
    *               マウントしていなくても読み込むことが可能です。
    */
    static Result ReadExtraData(
                      JournalIntegritySaveDataFileSystem::ExtraData* outData,
                      fs::SubStorage baseStorage,
                      IBufferManager* pBufferManager,
                      IMacGenerator* pMacGenerator,
                      uint32_t minimumVersion
                  ) NN_NOEXCEPT;


    /**
    * @brief        セーブデータの署名 (MAC) を更新します。
    *
    * @param[in]    baseStorage    セーブデータ領域として使用するストレージ
    * @param[in]    pMacGenerator  MAC 生成クラス
    * @param[in]    minimumVersion 最低バージョン
    *
    * @return       関数の処理結果を返します。
    *
    * @details      マスター管理領域の MAC のみを付け直します。
    */
    static Result UpdateMac(
                      fs::SubStorage baseStorage,
                      IMacGenerator* pMacGenerator,
                      uint32_t minimumVersion
                  ) NN_NOEXCEPT;

    //! セーブデータをマウントします。
    Result Initialize(
               fs::SubStorage baseStorage,
               IBufferManager* pBufferManager,
               IMacGenerator* pMacGenerator,
               uint32_t minimumVersion
           ) NN_NOEXCEPT;

    //! セーブデータをアンマウントします。
    void Finalize() NN_NOEXCEPT;

    //! 拡張データを更新します
    Result WriteExtraData(const JournalIntegritySaveDataFileSystem::ExtraData& extraData) NN_NOEXCEPT;

    //! 拡張データを取得します
    //! WriteExtraData で書き込んだデータを読み込みます
    void ReadExtraData(JournalIntegritySaveDataFileSystem::ExtraData* outData) NN_NOEXCEPT;

    //! セーブデータ ファイルシステムに対してコミットを行います。
    virtual Result Commit() NN_NOEXCEPT NN_OVERRIDE;

    //! セーブデータ ファイルシステムに対して仮コミットを行います。
    virtual Result CommitProvisionally(int64_t counter) NN_NOEXCEPT NN_OVERRIDE;

    //! セーブデータ ファイルシステムに対してロールバックを行います。
    virtual Result Rollback() NN_NOEXCEPT NN_OVERRIDE;

    //! 複数セーブデータの一括コミット用カウンタ値を取得します。
    int64_t GetCounterForBundledCommit() const NN_NOEXCEPT;

    //! セーブデータ管理領域のパラメータを抽出します。
    void ExtractParameters(JournalIntegritySaveDataParameters* outParameters) NN_NOEXCEPT;

    //! セーブデータ管理領域のパラメータを自動設定します。
    static JournalIntegritySaveDataParameters SetUpSaveDataParameters(int32_t blockSize, int64_t size, int64_t journalSize) NN_NOEXCEPT;

    //! セーブデータ内部ストレージアクセス用の Vistor を受け入れます。
    Result AcceptVisitor(IInternalStorageFileSystemVisitor* pVisitor) NN_NOEXCEPT NN_OVERRIDE;

private:
    fs::SubStorage m_BaseStorage;
    os::Mutex m_Locker;                                      //!< ファイルシステムレベルの排他制御オブジェクト
    IBufferManager*             m_pBufferManager;            //!< バッファマネージャ
    FilesystemBufferManagerSet  m_BufferManagerSet;          //!< バッファマネージャセット
    JournalIntegritySaveDataFileSystem m_SaveDataFileSystem; //!< セーブデータファイルシステム
};

}}}
