﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>

#include <nn/os/os_Mutex.h>
#include <nn/fs/fs_StorageType.h>
#include <nn/fssystem/save/fs_ISaveFileSystemDriver.h>
#include <nn/fssystem/buffers/fs_FileSystemBufferManager.h>

namespace nn { namespace fssystem { namespace save {

    //! 完全性検証の最大階層数 (セーブデータ系)
    static const size_t IntegrityLayerCountSave = 5;
    static const size_t IntegrityLayerCountSaveDataMeta = 4;
    static const size_t IntegrityMaxLayerCount = 7;
    static const size_t IntegrityMinLayerCount = 2;

    /*!
    * @brief バッファセットを定義する構造体です。
    */
    struct FilesystemBufferManagerSet
    {
        //! ハッシュに割り当てるキャッシュ
        IBufferManager* pBuffer[IntegrityMaxLayerCount];
    };
    NN_STATIC_ASSERT(std::is_pod<FilesystemBufferManagerSet>::value);

}}}

namespace nn { namespace fssystem { namespace save {

/**
* @brief   キャッシュ込みの読み書きを行うための機構です。
*
* @details 完全性検証レイヤー向けに、データの読み書きをブロック単位で行う機構を提供します。
*          また、一度読み込んだ(=検証が済んだデータであることを意味します)結果を
*          RAMへキャッシュする機能を提供します。
*
*          - 最大連続書き込み長制限
*          - 非自立的なバッファフラッシュ
*            (BufferdFileを積み重ねたときに、コールスタックが深くなりすぎるのを回避するため)
*
*          という目的のためそれ単体ではIFileとして使えません。
*          書き込み最大長制限と、書き込み毎のバッファフラッシュ処理を外部に付け足す
*          アタッチメントクラスを用意します。
*/
class BlockCacheBufferedStorage : public fs::IStorage
{
public:
    static const size_t DefaultMaxCacheEntryCount = 24; //! キャッシュするエントリー数のデフォルト

public:
    /**
    * @brief       コンストラクタ
    */
    BlockCacheBufferedStorage() NN_NOEXCEPT;

    /**
    * @brief       デストラクタ
    */
    virtual ~BlockCacheBufferedStorage() NN_NOEXCEPT;

    /**
    * @brief       バッファストレージを初期化します。
    *
    * @param[in]   pBuffer バッファマネージャ
    * @param[in]   pLocker 排他制御用のロックオブジェクト
    * @param[in]   pData データを含んだストレージ
    * @param[in]   sizeData ストレージのサイズ
    * @param[in]   sizeVerificationBlock ブロックのサイズ
    * @param[in]   maxCacheEntries キャッシュするエントリー数
    * @param[in]   isRealDataCache 実データに対するキャッシュかどうか
    * @param[in]   bufferLevel バッファの登録レベル
    * @param[in]   isKeepBurstMode バースト転送維持モード
    *
    * @return      関数の処理結果を返します。
    */
    Result Initialize(
                IBufferManager* pBuffer,
                os::Mutex* pLocker,
                IStorage* pData,
                int64_t sizeData,
                size_t sizeVerificationBlock,
                int maxCacheEntries,
                bool isRealDataCache,
                int8_t bufferLevel,
                bool isKeepBurstMode,
                fs::StorageType storageType
            ) NN_NOEXCEPT;

    /**
    * @brief      バッファストレージを閉じます。
    */
    void Finalize() NN_NOEXCEPT;

    /**
    * @brief      ストレージの内容をバッファに読み込みます。
    *
    * @param[in]  offset  読み込み開始位置
    * @param[out] buffer  読み込んだ内容をコピーするバッファ
    * @param[in]  size    読み込むデータサイズ
    *
    * @return     関数の処理結果を返します。
    */
    virtual Result Read(
                       int64_t offset,
                       void* buffer,
                       size_t size
                   ) NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief       バッファの内容をストレージに書き込みます。
    *
    * @param[in]   offset  書き込み開始位置
    * @param[in]   buffer  書き込むデータ
    * @param[in]   size    書き込むデータサイズ
    *
    * @return      関数の処理結果を返します。
    */
    virtual Result Write(
                       int64_t offset,
                       const void* buffer,
                       size_t size
                   ) NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief       バッファストレージのサイズを取得します。
    *
    * @param[out]  outValue バッファストレージのサイズ
    *
    * @return      関数の処理結果を返します。
    */
    virtual Result GetSize(int64_t* outValue) NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief       バッファストレージのサイズは変更できません。初期化時のみ指定できます。
    *
    * @return      常に nn::fs::ResultUnsupportedOperation
    */
    virtual Result SetSize(int64_t) NN_NOEXCEPT NN_OVERRIDE
    {
        return nn::fs::ResultUnsupportedOperation();
    }

    /**
    * @brief        フラッシュします。
    *
    * @return       関数の処理結果を返します。
    */
    virtual Result Flush() NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief       範囲指定処理を行います。
    *
    * @param[out]  outBuffer        範囲指定処理の結果を格納するバッファ
    * @param[in]   outBufferSize    範囲指定処理の結果を格納するバッファのサイズ
    * @param[in]   operationId      範囲指定処理の種類
    * @param[in]   offset           範囲指定処理開始位置
    * @param[in]   size             範囲指定処理を行うデータサイズ
    * @param[in]   inBuffer         範囲指定処理に渡すバッファ
    * @param[in]   inBufferSize     範囲指定処理に渡すバッファのサイズ
    *
    * @return      関数の処理結果を返します。
    */
    virtual Result OperateRange(
                       void* outBuffer,
                       size_t outBufferSize,
                       fs::OperationId operationId,
                       int64_t offset,
                       int64_t size,
                       const void* inBuffer,
                       size_t inBufferSize
                   ) NN_NOEXCEPT NN_OVERRIDE;

    using IStorage::OperateRange;

    /**
    * @brief        コミット処理を行います。
    *
    * @return       関数の処理結果を返します。
    */
    Result Commit() NN_NOEXCEPT;

    /**
    * @brief        ロールバック時の処理です。
    *
    * @return       関数の処理結果を返します。
    *
    * @details      全てのキャッシュをストアせずに破棄します。
    */
    Result OnRollback() NN_NOEXCEPT;

    /**
    * @brief        バースト転送維持モードを設定します。
    */
    void SetKeepBurstMode(bool isEnable) NN_NOEXCEPT
    {
        if (isEnable)
        {
            m_Flags |= Flag_KeepBurstMode;
        }
        else
        {
            m_Flags &= ~Flag_KeepBurstMode;
        }
    }

    /**
    * @brief       バースト転送維持モードが有効かどうか問い合せます。
    *
    * @return      バースト転送維持モードが有効の場合は true を返します。
    */
    bool IsEnabledKeepBurstMode() const NN_NOEXCEPT
    {
        return (0 != (m_Flags & Flag_KeepBurstMode));
    }

    /**
    * @brief       実データに対するキャッシュかどうかを設定します。
    */
    void SetRealDataCache(bool isEnable) NN_NOEXCEPT
    {
        if (isEnable)
        {
            m_Flags |= Flag_RealData;
        }
        else
        {
            m_Flags &= ~Flag_RealData;
        }
    }

    /**
    * @brief       実データに対するキャッシュかどうかを取得します。
    *
    * @return      実データに対するキャッシュの場合は true を返します。
    */
    bool IsRealDataCache() const NN_NOEXCEPT
    {
        return (0 != (m_Flags & Flag_RealData));
    }

private:
    //! キャッシュエントリー
    struct CacheEntry
    {
        size_t size;                                    //! キャッシュしている領域のサイズ
        bool isValid;                                   //! 有効なエントリーかどうか
        bool isWriteBack;                               //! 書き込み待ちかどうか
        bool isCached;                                  //! キャッシュされているエントリーかどうか
        bool isFlushing;                                //! キャッシュされているエントリーかどうか
        int64_t offset;                                 //! キャッシュしている領域のオフセット
        IBufferManager::CacheHandle handle;             //! キャッシュハンドル
        uintptr_t memoryAddress;                        //! メモリ範囲の開始アドレス
        size_t memorySize;                              //! メモリ範囲のサイズ
    };
    NN_STATIC_ASSERT(std::is_pod<CacheEntry>::value);

    typedef int CacheIndex;
    typedef std::pair<uintptr_t, size_t> MemoryRange;

private:
    //! 動作フラグの定義
    enum Flag : int32_t
    {
        Flag_KeepBurstMode         = (1 << 8),  //! バースト転送維持モード
        Flag_RealData              = (1 << 10), //! 実データに対するキャッシュかどうか
    };

private:
    /*!
    * @brief       エントリー可能なキャッシュの最大数を取得します。
    *
    * @return      エントリー可能なキャッシュの最大数を返します。
    *
    * @details     Initialize 時に指定した値を返します。
    */
    int GetMaxCacheEntryCount() const NN_NOEXCEPT
    {
        return m_MaxCacheEntryCount;
    }

    /**
    * @brief       範囲を指定してゼロフィルします。
    *
    * @param[in]   offset  ゼロフィル開始位置
    * @param[in]   size    ゼロフィルするデータサイズ
    *
    * @return      関数の処理結果を返します。
    */
    Result FillZeroImpl(
               int64_t offset,
               int64_t size
           ) NN_NOEXCEPT;

    /**
    * @brief       範囲を指定してハッシュを破壊します。
    *
    * @param[in]   offset  ハッシュを破壊する開始位置
    * @param[in]   size    ハッシュを破壊するデータサイズ
    *
    * @return      関数の処理結果を返します。
    */
    Result DestroySignatureImpl(
               int64_t offset,
               int64_t size
           ) NN_NOEXCEPT;

    /**
    * @brief       範囲を指定してキャッシュを無効化します。
    *
    * @param[in]   offset  キャッシュ無効化開始位置
    * @param[in]   size    無効化するデータサイズ
    *
    * @return      関数の処理結果を返します。
    */
    Result InvalidateImpl(
               int64_t offset,
               int64_t size
           ) NN_NOEXCEPT;

    /**
    * @brief       指定範囲にアクセスする際に使用する機能の情報を取得します。
    *
    * @param[out]  outBuffer        情報を保存するバッファ
    * @param[in]   outBufferSize    情報を保存するバッファのサイズ
    * @param[in]   offset           範囲指定の開始位置
    * @param[in]   size             範囲指定のデータサイズ
    *
    * @return      関数の処理結果を返します。
    */
    Result QueryRangeImpl(
               void* outBuffer,
               size_t outBufferSize,
               int64_t offset,
               int64_t size
           ) NN_NOEXCEPT;

    /**
    * @brief       指定のキャッシュエントリと重複するエントリが存在するか確認します。
    *
    * @param[in]   entry   キャッシュエントリ
    *
    * @return      重複エントリが見つかれば true を返します。
    */
    bool ExistsRedundantCacheEntry(const CacheEntry& entry) const NN_NOEXCEPT;

    /**
    * @brief       指定したオフセット、サイズを含んだキャッシュエントリとそのエントリーが示す範囲を取得します。
    *
    * @param[out]  outRange            範囲オブジェクト
    * @param[out]  outEntry            キャッシュエントリ
    * @param[in]   offset              領域の開始オフセット
    * @param[in]   sizeIdeal           領域のサイズ
    * @param[in]   isAllocateForWrite  書き込み用かどうか
    *
    * @return      関数の処理結果を返します。
    */
    Result GetAssociateBuffer(
               MemoryRange* outRange,
               CacheEntry* outEntry,
               int64_t offset,
               size_t sizeIdeal,
               bool isAllocateForWrite
           ) NN_NOEXCEPT;

    /**
    * @brief       バッファを破棄します。
    *
    * @param[in]   pEntry  破棄するキャッシュエントリ
    * @param[in]   range   破棄するバッファの範囲
    *
    * @return      関数の処理結果を返します。
    */
    void DestroyBuffer(CacheEntry* pEntry, const MemoryRange& range) NN_NOEXCEPT;

    /**
    * @brief       検証済みのバッファをキャッシュと関連付けします。
    *
    * @param[out]  outValue 関連付けられたキャッシュエントリのインデクス
    * @param[in]   range    関連付けするバッファの範囲
    * @param[in]   pEntry   関連付けするキャッシュエントリ
    *
    * @return      関数の処理結果を返します。
    */
    Result StoreAssociateBuffer(
               CacheIndex* outValue,
               const MemoryRange& range,
               const CacheEntry& entry
           ) NN_NOEXCEPT;

    /**
    * @brief       検証済みのバッファをキャッシュと関連付けします。
    *
    * @param[in]   range   関連付けするバッファの範囲
    * @param[in]   pEntry  関連付けするキャッシュエントリ
    *
    * @return      関数の処理結果を返します。
    */
    Result StoreAssociateBuffer(const MemoryRange& range, const CacheEntry& entry) NN_NOEXCEPT
    {
        CacheIndex cacheIndex;
        return StoreAssociateBuffer(&cacheIndex, range, entry);
    }

    /**
    * @brief       検証済みのバッファをキャッシュと関連付けし、それが失敗したらバッファを破棄します。
    *
    * @param[in]   range   関連付けするバッファの範囲
    * @param[in]   pEntry  関連付けするキャッシュエントリ
    *
    * @return      関数の処理結果を返します。
    */
    Result StoreOrDestroyBuffer(const MemoryRange& range, CacheEntry* pEntry) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pEntry);
        const auto result = StoreAssociateBuffer(range, *pEntry);
        if( result.IsFailure() )
        {
            (void)DestroyBuffer(pEntry, range);
            NN_RESULT_THROW(result);
        }
        NN_RESULT_SUCCESS;
    }

    /**
    * @brief        キャッシュエントリをフラッシュします。
    *
    * @param[in]    index               フラッシュするキャッシュエントリ
    * @param[in]    isWithInvalidate    フラッシュしたキャッシュを無効かするかどうか
    * @param[in]    option              書き込みオプション
    *
    * @return       関数の処理結果を返します。
    */
    Result FlushCacheEntry(
               CacheIndex index,
               bool isWithInvalidate
           ) NN_NOEXCEPT;

    /**
    * @brief        範囲に含まれるキャッシュエントリをフラッシュします。
    *
    * @param[in]    offset              フラッシュ範囲の始点を示すオフセット
    * @param[in]    size                フラッシュ範囲
    * @param[in]    isWithInvalidate    フラッシュしたキャッシュを無効かするかどうか
    * @param[in]    option              書き込みオプション
    *
    * @return       関数の処理結果を返します。
    */
    Result FlushRangeCacheEntries(
               int64_t offset,
               int64_t size,
               bool isWithInvalidate
           ) NN_NOEXCEPT;

    /**
    * @brief        範囲に含まれるキャッシュエントリを無効化します。
    *
    * @param[in]    offset              無効化範囲の始点を示すオフセット
    * @param[in]    size                無効化範囲
    *
    * @return       関数の処理結果を返します。
    *
    * @details      キャッシュをフラッシュしません。
    *               フラッシュしてから無効化するためには FlushRangeCacheEntries() を使用します。
    */
    void InvalidateRangeCacheEntries(
             int64_t offset,
             int64_t size
         ) NN_NOEXCEPT;

    /**
    * @brief       全てのキャッシュエントリをフラッシュします。
    *
    * @return      関数の処理結果を返します。
    */
    Result FlushAllCacheEntries() NN_NOEXCEPT;

    /**
    * @brief        全てのキャッシュエントリを無効化します。
    *
    * @return       関数の処理結果を返します。
    */
    Result InvalidateAllCacheEntries() NN_NOEXCEPT;

    /**
    * @brief        ダーティバッファをため込み過ぎないように、必要ならキャッシュをフラッシュします。
    *
    * @return       関数の処理結果を返します。
    */
    Result ControlDirtiness() NN_NOEXCEPT;

    /**
    * @brief        読み書き結果の検査関数
    *
    * @param[in]    result        発生したリザルト
    *
    * @return       関数の処理結果を返します。
    */
    Result UpdateLastResult(nn::Result result) NN_NOEXCEPT;

    /**
    * @brief      読み込む領域の先頭にある連続キャッシュを読み込み、読み込む領域を狭めます。
    *
    * @param[out] pOutMemoryRange       最後に読み込んだキャッシュのバッファ範囲
    * @param[out] pOutEntry             最後に読み込んだキャッシュのエントリー
    * @param[out] pOutIsCacheNeeded     最後に読み込んだキャッシュを新たにキャッシュすべきか
    * @param[in]  pOffset               読み込み開始位置のポインタ
    * @param[in]  pOffsetAligned        アライメントを揃えた読み込み開始位置のポインタ
    * @param[in]  offsetAlignedEnd      アライメントを揃えた読み込み終了位置
    * @param[in]  pBuffer               読み込んだ内容をコピーするバッファのポインタ
    * @param[in]  pSize                 読み込むデータサイズのポインタ
    *
    * @return     関数の処理結果を返します。
    */
    Result ReadHeadCache(
               MemoryRange* pOutMemoryRange,
               CacheEntry* pOutEntry,
               bool* pOutIsCacheNeeded,
               int64_t* pOffset,
               int64_t* pOffsetAligned,
               int64_t offsetAlignedEnd,
               char** pBuffer,
               size_t* pSize
           ) NN_NOEXCEPT;

    /**
    * @brief      読み込む領域の末尾にある連続キャッシュを読み込み、読み込む領域を狭めます。
    *
    * @param[out] pOutMemoryRange       最後に読み込んだキャッシュのバッファ範囲
    * @param[out] pOutEntry             最後に読み込んだキャッシュのエントリー
    * @param[out] pOutIsCacheNeeded     最後に読み込んだキャッシュを新たにキャッシュすべきか
    * @param[in]  offset                読み込み開始位置
    * @param[in]  offsetAligned         アライメントを揃えた読み込み開始位置
    * @param[in]  pOffsetAlignedEnd     アライメントを揃えた読み込み終了位置のポインタ
    * @param[in]  buffer                読み込んだ内容をコピーするバッファ
    * @param[in]  pSize                 読み込むデータサイズのポインタ
    *
    * @return     関数の処理結果を返します。
    */
    Result ReadTailCache(
               MemoryRange* pOutMemoryRange,
               CacheEntry* pOutEntry,
               bool* pOutIsCacheNeeded,
               int64_t offset,
               int64_t offsetAligned,
               int64_t* pOffsetAlignedEnd,
               char* buffer,
               size_t* pSize
           ) NN_NOEXCEPT;

    /**
    * @brief      ストレージの内容をバッファに一括読み込みします。
    *
    * @param[in]  offset  読み込み開始位置
    * @param[out] buffer  読み込んだ内容をコピーするバッファ
    * @param[in]  size    読み込むデータサイズ
    *
    * @return     関数の処理結果を返します。
    */
    Result BulkRead(
               int64_t offset,
               void* buffer,
               size_t size,
               MemoryRange* pMemoryRangeHead,
               MemoryRange* pMemoryRangeTail,
               CacheEntry* pEntryHead,
               CacheEntry* pEntryTail,
               bool isHeadCacheNeeded,
               bool isTailCacheNeeded
           ) NN_NOEXCEPT;

private:
    IBufferManager* m_pBufferManager;                                 //!< バッファマネージャ
    os::Mutex* m_pLocker;                                             //!< ロックオブジェクト
    std::unique_ptr<CacheEntry[], nn::fs::detail::Deleter> m_pEntry;  //!< キャッシュエントリーを管理するテーブル
    IStorage* m_pStorageData;                                         //!< データを含んだストレージ
    Result m_LastResult;                                              //!< 最後に発生したエラーを記憶しておく
    int64_t m_SizeBytesData;                                          //!< ストレージのサイズ
    size_t m_SizeBytesVerificationBlock;                              //!< ブロックサイズ
    size_t m_ShiftBytesVerificationBlock;                             //!< ブロック数を求めるためのシフト値
    CacheIndex m_IndexInvalidate;                                     //!< 次回フラッシュするインデックス
    int m_MaxCacheEntryCount;                                         //!< キャッシュするエントリー数
    int32_t m_Flags;                                                  //!< 各種フラグ
    int m_BufferLevel;                                                //!< バッファの登録レベル
    fs::StorageType m_StorageType;                                    //!< 使用するストレージのタイプ

private:
    friend class FsBlockCacheBufferedStorageTest;
};

}}}

