﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Allocator.h>
#include <nn/fs/fs_IStorage.h>
#include <nn/fs/fs_SubStorage.h>
#include <nn/fssystem/fs_AsynchronousAccess.h>

namespace nn { namespace fssystem {

/**
* @brief         キャッシュ付きストレージクラス
*
* @details       ライトスルー形式のキャッシュを提供するストレージクラスです。
*                単一スレッドから使用することを想定しており、
*                スレッドセーフな実装にはなっていません。
*/
class WriteThroughCacheStorage : public fs::IStorage
{
    NN_DISALLOW_COPY(WriteThroughCacheStorage);

public:
    /**
    * @brief      コンストラクタです。
    */
    WriteThroughCacheStorage() NN_NOEXCEPT;

    /**
    * @brief      デストラクタです。
    */
    virtual ~WriteThroughCacheStorage() NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief      キャッシュが有効なストレージとして初期化します。
    *
    * @param[in]  pAllocator        キャッシュに使用するメモリを確保するアロケータ
    * @param[in]  baseStorage       キャッシュ対象ストレージ
    * @param[in]  blockSize         キャッシュするブロックのサイズ
    * @param[in]  bufferCount       キャッシュバッファ数
    *
    * @return     関数の処理結果を返します。
    *
    * @pre
    *             - 0 < blockSize
    *             - blockSize が 2 のべき乗
    *             - 0 < bufferCount
    */
    Result Initialize(
               nn::MemoryResource* pAllocator,
               fs::SubStorage baseStorage,
               size_t blockSize,
               int bufferCount
           ) NN_NOEXCEPT;

    /**
    * @brief      終了処理を行います。
    */
    void Finalize() NN_NOEXCEPT;

    /**
    * @brief        ストレージの内容をバッファに読み込みます。
    *
    * @param[in]    offset  読み込み開始位置
    * @param[out]   buffer  読み込んだ内容をコピーするバッファ
    * @param[in]    size    読み込むデータサイズ
    *
    * @return       関数の処理結果を返します。
    *
    * @pre
    *               - Initialize に成功している
    */
    virtual Result Read(
                       int64_t offset,
                       void* buffer,
                       size_t size
                   ) NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief        バッファの内容をストレージに書き込みます。
    *
    * @param[in]    offset  書き込み開始位置
    * @param[in]    buffer  書き込むデータ
    * @param[in]    size    書き込むデータサイズ
    *
    * @return       関数の処理結果を返します。
    *
    * @pre
    *               - Initialize に成功している
    */
    virtual Result Write(
                       int64_t offset,
                       const void* buffer,
                       size_t size
                   ) NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief        ストレージサイズを取得します。
    *
    * @param[out]   outValue    ストレージサイズ
    *
    * @return       関数の処理結果を返します。
    *
    * @pre
    *               - Initialize に成功している
    */
    virtual Result GetSize(int64_t* outValue) NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief        ストレージサイズを変更します。
    *
    * @param[in]    size    変更後のストレージサイズ
    *
    * @return       関数の処理結果を返します。
    *
    * @pre
    *               - Initialize に成功している
    *               - size >= 0
    */
    virtual Result SetSize(int64_t size) NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief        フラッシュします。
    *
    * @return       関数の処理結果を返します。
    *
    * @pre
    *               - Initialize に成功している
    */
    virtual Result Flush() NN_NOEXCEPT NN_OVERRIDE;

    /**
    * @brief       範囲指定処理を行います。
    *
    * @param[out]  outBuffer        出力バッファ
    * @param[in]   outBufferSize    出力バッファサイズ
    * @param[in]   operationId      範囲指定処理の種類
    * @param[in]   offset           範囲指定処理開始位置
    * @param[in]   size             範囲指定処理を行うデータサイズ
    * @param[in]   inBuffer         入力バッファ
    * @param[in]   inBufferSize     入力バッファサイズ
    *
    * @return      関数の処理結果を返します。
    *
    * @pre
    *               - Initialize に成功している
    */
    virtual Result OperateRange(
                       void* outBuffer,
                       size_t outBufferSize,
                       fs::OperationId operationId,
                       int64_t offset,
                       int64_t size,
                       const void* inBuffer,
                       size_t inBufferSize
                   ) NN_NOEXCEPT NN_OVERRIDE;

    //! ベースストレージを取得します
    fs::SubStorage& GetBaseStorage() NN_NOEXCEPT
    {
        return m_BaseStorage;
    }

    //! キャッシュのブロックサイズを取得します
    size_t GetBlockSize() const NN_NOEXCEPT
    {
        return m_BlockSize;
    }

private:
    struct Cache;

    /**
    * @brief       キャッシュをリストの先頭に移動します。
    *
    * @param[in]   pCache       範囲指定処理の種類
    */
    void MoveToHead(Cache* pCache) NN_NOEXCEPT;

    /**
    * @brief       キャッシュをリストの末尾に移動します。
    *
    * @param[in]   pCache       範囲指定処理の種類
    */
    void MoveToTail(Cache* pCache) NN_NOEXCEPT;

    /**
    * @brief       指定したオフセットのキャッシュを取得します。
    *
    * @param[in]   offset       オフセット
    *
    * @return      キャッシュへのポインタを返します。
    */
    Cache* GetCache(int64_t offset) NN_NOEXCEPT;

    /**
    * @brief       使用可能なキャッシュを取得します。
    *
    * @param[in]   offset       オフセット
    *
    * @return      キャッシュへのポインタを返します。
    */
    Cache* GetFreeCache() NN_NOEXCEPT;

    /**
    * @brief       指定した範囲のキャッシュを無効化します。
    *
    * @param[in]   offset       無効化開始オフセット
    * @param[in]   size         無効化する範囲
    */
    void InvalidateCache(int64_t offset, int64_t size) NN_NOEXCEPT;

    /**
    * @brief       すべてのキャッシュを無効化します。
    */
    void InvalidateCacheAll() NN_NOEXCEPT;

    /**
    * @brief       指定した範囲のキャッシュが保持するデータを更新します。
    *
    * @param[in]   offset       更新開始オフセット
    * @param[in]   buffer       更新に使用するデータ
    * @param[in]   size         更新する範囲
    */
    void UpdateCache(int64_t offset, const char* buffer, size_t size) NN_NOEXCEPT;

    /**
    * @brief 初期化済みかどうか
    */
    bool IsInitialized() NN_NOEXCEPT
    {
        return (m_pCacheList != nullptr);
    }

private:
    nn::MemoryResource* m_pAllocator;   //!< キャッシュのメモリ確保先アロケータ
    PooledBuffer m_BufferCache;         //!< バッファキャッシュ用メモリ管理
    fs::SubStorage m_BaseStorage;       //!< ベースストレージ
    size_t m_BlockSize;                 //!< キャッシュブロックサイズ
    Cache* m_pCacheList;                //!< キャッシュリスト
    Cache* m_pCacheListHead;            //!< キャッシュリストの先頭
    Cache* m_pCacheListTail;            //!< キャッシュリストの末尾
    int m_CacheCount;                   //!< キャッシュ数
};

}}

