﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <algorithm>
#include <nn/os/os_SdkThread.h>

namespace nn { namespace fssystem {

//! @brief スレッドの優先度を変更するクラス
class ScopedThreadPriorityChanger
{
public:
    enum class Mode
    {
        Absolute,
        Relative
    };

public:
    explicit ScopedThreadPriorityChanger(int priority, Mode mode) NN_NOEXCEPT
        : m_pThread(nn::os::GetCurrentThread())
        , m_Priority(0)
    {
        NN_SDK_ASSERT_NOT_NULL(m_pThread);
        const auto targetPriority = std::min(
            (mode == Mode::Relative
                ? os::GetThreadPriority(m_pThread) + priority
                : priority),
            os::LowestSystemThreadPriority
            );
        m_Priority = os::ChangeThreadPriority(m_pThread, targetPriority);
    }

    ~ScopedThreadPriorityChanger() NN_NOEXCEPT
    {
        (void)os::ChangeThreadPriority(m_pThread, m_Priority);
    }

private:
    os::ThreadType* m_pThread;
    int m_Priority;
};

//! @brief アクセス優先度に基づいてスレッドの優先度を変更するクラス
class ScopedThreadPriorityChangerByAccessPriority
{
public:
    enum class AccessMode
    {
        Read,
        Write
    };

public:
    explicit ScopedThreadPriorityChangerByAccessPriority(AccessMode accessMode) NN_NOEXCEPT
        : m_Changer(GetThreadPriorityByAccessPriority(accessMode), ScopedThreadPriorityChanger::Mode::Absolute)
    {
    }

private:
    static int GetThreadPriorityByAccessPriority(AccessMode accessMode) NN_NOEXCEPT;

private:
    ScopedThreadPriorityChanger m_Changer;
};

}}
