﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/fs/fs_SpeedEmulation.h>
#include <nn/fs/fs_IStorage.h>
#include <nn/fs/detail/fs_Newable.h>
#include <nn/fs/fsa/fs_IFile.h>
#include <nn/fs/fsa/fs_IFileSystem.h>
#include <nn/fssystem/fs_ServiceContext.h>

namespace nn { namespace fssystem {

enum StorageFlag
{
    StorageFlag_Mmc         = 1 << 0,
    StorageFlag_SdCard      = 1 << 1,
    StorageFlag_GameCard    = 1 << 2,
    StorageFlag_Usb         = 1 << 3,
};

inline bool IsStorageFlagValid(int flag) NN_NOEXCEPT
{
    if( flag == 0 )
    {
        return false;
    }

    return true;
}

class ScopedStorageLayoutTypeSetter
{
public:
    explicit ScopedStorageLayoutTypeSetter(int flag) NN_NOEXCEPT
        : m_StorageFlagPrevious(0)
#if defined(NN_BUILD_CONFIG_OS_WIN)
        , m_Context()
        , m_IsContextRegistered(false)
#endif
    {
        NN_SDK_ASSERT(IsStorageFlagValid(flag));

        auto pContext = GetServiceContext();
        if( pContext != nullptr )
        {
            m_StorageFlagPrevious = pContext->GetStorageFlag();
            const auto newFlag = m_StorageFlagPrevious | flag;
            NN_SDK_ASSERT(IsStorageFlagValid(newFlag));
            pContext->SetStorageFlag(newFlag);
        }
#if defined(NN_BUILD_CONFIG_OS_WIN)
        else
        {
            m_Context.SetStorageFlag(flag);
            RegisterServiceContext(&m_Context);
            m_IsContextRegistered = true;
        }
#endif
    }

    ~ScopedStorageLayoutTypeSetter() NN_NOEXCEPT
    {
        auto pContext = GetServiceContext();
        NN_SDK_ASSERT_NOT_NULL(pContext);
        if( pContext != nullptr )
        {
#if defined(NN_BUILD_CONFIG_OS_WIN)
            if( m_IsContextRegistered )
            {
                UnregisterServiceContext();
            }
            else
#endif
            {
                pContext->SetStorageFlag(m_StorageFlagPrevious);
            }
        }
    }

private:
    int m_StorageFlagPrevious;
#if defined(NN_BUILD_CONFIG_OS_WIN)
    ServiceContext m_Context;
    bool m_IsContextRegistered;
#endif
};

class StorageLayoutTypeSetStorage : public nn::fs::IStorage, public nn::fs::detail::Newable
{
public:
    StorageLayoutTypeSetStorage(std::shared_ptr<IStorage> pBaseStorage, int storageFlag) NN_NOEXCEPT
        : m_pBaseStorage(pBaseStorage)
        , m_StorageFlag(storageFlag)
    {
        NN_SDK_ASSERT(IsStorageFlagValid(storageFlag));
    }

    virtual ~StorageLayoutTypeSetStorage() NN_NOEXCEPT NN_OVERRIDE;

    virtual Result Read(int64_t offset, void* buffer, size_t size) NN_NOEXCEPT NN_OVERRIDE;
    virtual Result Write(int64_t offset, const void* buffer, size_t size) NN_NOEXCEPT NN_OVERRIDE;
    virtual Result Flush() NN_NOEXCEPT NN_OVERRIDE;
    virtual Result SetSize(int64_t size) NN_NOEXCEPT NN_OVERRIDE;
    virtual Result GetSize(int64_t* outValue) NN_NOEXCEPT NN_OVERRIDE;
    virtual Result OperateRange(
        void* outBuffer,
        size_t outBufferSize,
        nn::fs::OperationId operationId,
        int64_t offset,
        int64_t size,
        const void* inBuffer,
        size_t inBufferSize) NN_NOEXCEPT NN_OVERRIDE;

private:
    std::shared_ptr<IStorage> m_pBaseStorage;
    int m_StorageFlag;
};

class StorageLayoutTypeSetFile : public nn::fs::fsa::IFile, public nn::fs::detail::Newable
{
public:
    StorageLayoutTypeSetFile(std::unique_ptr<IFile>&& pBaseFile, int storageFlag) NN_NOEXCEPT
        : m_pBaseFile(pBaseFile.get())
        , m_pUniqueBaseFile(std::move(pBaseFile))
        , m_StorageFlag(storageFlag)
    {
        NN_SDK_ASSERT(IsStorageFlagValid(storageFlag));
    }

    virtual ~StorageLayoutTypeSetFile() NN_NOEXCEPT NN_OVERRIDE;

    StorageLayoutTypeSetFile(std::shared_ptr<IFile>&& pBaseFile, int storageFlag) NN_NOEXCEPT
        : m_pBaseFile(pBaseFile.get())
        , m_pSharedBaseFile(pBaseFile)
        , m_StorageFlag(storageFlag)
    {
        NN_SDK_ASSERT(IsStorageFlagValid(storageFlag));
    }

    virtual Result DoRead(size_t* outValue, int64_t offset, void* buffer, size_t size, const nn::fs::ReadOption& option) NN_NOEXCEPT NN_OVERRIDE;
    virtual Result DoWrite(int64_t offset, const void* buffer, size_t size, const nn::fs::WriteOption& option) NN_NOEXCEPT NN_OVERRIDE;
    virtual Result DoFlush() NN_NOEXCEPT NN_OVERRIDE;
    virtual Result DoSetSize(int64_t size) NN_NOEXCEPT NN_OVERRIDE;
    virtual Result DoGetSize(int64_t* outValue) NN_NOEXCEPT NN_OVERRIDE;
    virtual Result DoOperateRange(
        void* outBuffer,
        size_t outBufferSize,
        nn::fs::OperationId operationId,
        int64_t offset,
        int64_t size,
        const void* inBuffer,
        size_t inBufferSize) NN_NOEXCEPT NN_OVERRIDE;

private:
    nn::fs::fsa::IFile* m_pBaseFile;
    std::unique_ptr<IFile> m_pUniqueBaseFile;
    std::shared_ptr<IFile> m_pSharedBaseFile;
    int m_StorageFlag;
};

class StorageLayoutTypeSetFileSystem : public nn::fs::fsa::IFileSystem, public nn::fs::detail::Newable
{
public:
    StorageLayoutTypeSetFileSystem(std::shared_ptr<IFileSystem> pBaseFileSystem, int storageFlag) NN_NOEXCEPT
        : m_pBaseFileSystem(pBaseFileSystem)
        , m_StorageFlag(storageFlag)
    {
        NN_SDK_ASSERT(IsStorageFlagValid(storageFlag));
    }

    virtual ~StorageLayoutTypeSetFileSystem() NN_NOEXCEPT NN_OVERRIDE;

private:
    virtual Result DoCreateFile(const char* path, int64_t size, int option) NN_NOEXCEPT NN_OVERRIDE;
    virtual Result DoDeleteFile(const char* path) NN_NOEXCEPT NN_OVERRIDE;
    virtual Result DoCreateDirectory(const char* path) NN_NOEXCEPT NN_OVERRIDE;
    virtual Result DoDeleteDirectory(const char* path) NN_NOEXCEPT NN_OVERRIDE;
    virtual Result DoDeleteDirectoryRecursively(const char* path) NN_NOEXCEPT NN_OVERRIDE;
    virtual Result DoCleanDirectoryRecursively(const char* path) NN_NOEXCEPT NN_OVERRIDE;
    virtual Result DoRenameFile(const char* currentPath, const char* newPath) NN_NOEXCEPT NN_OVERRIDE;
    virtual Result DoRenameDirectory(const char* currentPath, const char* newPath) NN_NOEXCEPT NN_OVERRIDE;
    virtual Result DoGetEntryType(nn::fs::DirectoryEntryType* outValue, const char* path) NN_NOEXCEPT NN_OVERRIDE;
    virtual Result DoGetFreeSpaceSize(int64_t* outValue, const char* path) NN_NOEXCEPT NN_OVERRIDE;
    virtual Result DoGetTotalSpaceSize(int64_t* outValue, const char* path) NN_NOEXCEPT NN_OVERRIDE;
    virtual Result DoOpenFile(std::unique_ptr<nn::fs::fsa::IFile>* outValue, const char* path, nn::fs::OpenMode mode) NN_NOEXCEPT NN_OVERRIDE;
    virtual Result DoOpenDirectory(std::unique_ptr<nn::fs::fsa::IDirectory>* outValue, const char* path, nn::fs::OpenDirectoryMode mode) NN_NOEXCEPT NN_OVERRIDE;
    virtual Result DoCommit() NN_NOEXCEPT NN_OVERRIDE;
    virtual Result DoCommitProvisionally(int64_t counter) NN_NOEXCEPT NN_OVERRIDE;
    virtual Result DoRollback() NN_NOEXCEPT NN_OVERRIDE;
    virtual Result DoFlush() NN_NOEXCEPT NN_OVERRIDE;
    virtual Result DoGetFileTimeStampRaw(nn::fs::FileTimeStampRaw* outTimeStamp, const char* path) NN_NOEXCEPT NN_OVERRIDE;
    virtual Result DoQueryEntry(char* outBuffer, size_t outBufferSize, const char* inBuffer, size_t inBufferSize, nn::fs::fsa::QueryId queryId, const char* path) NN_NOEXCEPT NN_OVERRIDE;

private:
    std::shared_ptr<IFileSystem> m_pBaseFileSystem;
    int m_StorageFlag;
};

}}
