﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/fs/fs_PriorityPrivate.h>
#include <nn/result/result_HandlingUtility.h>

namespace nn { namespace fssystem {

class ServiceContext
{
public:
    static const Bit64 InvalidProcessId = 0;
    static const int InvalidPrioritySessionType = -1;

public:
    ServiceContext() NN_NOEXCEPT
        : m_Priority(fs::PriorityRaw_Normal)
        , m_StorageFlag(0)
        , m_pRequestHookContext(nullptr)
    {
    }

public:
    fs::PriorityRaw GetPriority() const NN_NOEXCEPT
    {
        return m_Priority;
    }

    void SetPriority(fs::PriorityRaw priority) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(
            priority == fs::PriorityRaw_Background
            || priority == fs::PriorityRaw_Low
            || priority == fs::PriorityRaw_Normal
            || priority == fs::PriorityRaw_Realtime
        );
        m_Priority = priority;
    }

    void ResetDeferredProcessContext(bool isSessionForPriorityAcquired) NN_NOEXCEPT
    {
        {
            DeferredProcessContextForDeviceError context = { InvalidProcessId, false, false };
            m_DeferredProcessContextForDeviceError = context;
        }
        {
            DeferredProcessContextForPriority context =
            {
                isSessionForPriorityAcquired ? m_DeferredProcessContextForPriority.prioritySessionType : InvalidPrioritySessionType,
                false,
                isSessionForPriorityAcquired
            };
            m_DeferredProcessContextForPriority = context;
        }
    }

    bool IsProcessDeferredByDeviceError(Bit64 *outValue) const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(outValue);

        if (m_DeferredProcessContextForDeviceError.isProcessDeferred)
        {
            *outValue = m_DeferredProcessContextForDeviceError.processId;
            return true;
        }
        *outValue = InvalidProcessId;
        return false;
    }

    void SetDeferredProcessProcessId(Bit64 processId) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_EQUAL(InvalidProcessId, processId);
        DeferredProcessContextForDeviceError context = { processId, true, false };
        m_DeferredProcessContextForDeviceError = context;
    }

    bool GetProcessIdRequestedToInvokeDeferred(Bit64 *outValue) const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(outValue);

        if (m_DeferredProcessContextForDeviceError.isRequestedToInvokeDeferred)
        {
            *outValue = m_DeferredProcessContextForDeviceError.processId;
            return true;
        }
        *outValue = InvalidProcessId;
        return false;
    }

    void RequestToInvokeDeferred(Bit64 processId) NN_NOEXCEPT
    {
        DeferredProcessContextForDeviceError context = { processId, false, true };
        m_DeferredProcessContextForDeviceError = context;
    }

    bool IsProcessDeferredByPriority(int* outValue) const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(outValue);

        if( m_DeferredProcessContextForPriority.isProcessDeferred )
        {
            *outValue = m_DeferredProcessContextForPriority.prioritySessionType;
            return true;
        }
        *outValue = InvalidPrioritySessionType;
        return false;
    }

    void SetDeferredProcessPrioritySessionType(int prioritySessionType) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(InvalidPrioritySessionType == prioritySessionType || 0 <= prioritySessionType);
        DeferredProcessContextForPriority context = { prioritySessionType, true, false };
        m_DeferredProcessContextForPriority = context;
    }

    void SetAcquiredPrioritySessionType(int prioritySessionType) NN_NOEXCEPT
    {
        DeferredProcessContextForPriority context = { prioritySessionType, false, true };
        m_DeferredProcessContextForPriority = context;
    }

    void ResetAcquiredPrioritySessionType() NN_NOEXCEPT
    {
        DeferredProcessContextForPriority context = { InvalidPrioritySessionType, false, false };
        m_DeferredProcessContextForPriority = context;
    }

    bool GetAcquiredPrioritySessionType(int* outValue) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(outValue);

        if( m_DeferredProcessContextForPriority.isAcquired )
        {
            *outValue = m_DeferredProcessContextForPriority.prioritySessionType;
            return true;
        }
        *outValue = InvalidPrioritySessionType;
        return false;
    }

    int GetStorageFlag() const NN_NOEXCEPT
    {
        return m_StorageFlag;
    }

    void SetStorageFlag(int flag) NN_NOEXCEPT
    {
        m_StorageFlag = flag;
    }

    void* GetRequestHookContext() const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(m_pRequestHookContext);
        return m_pRequestHookContext;
    }

    void SetRequestHookContext(void* pContext) NN_NOEXCEPT
    {
        m_pRequestHookContext = pContext;
    }

private:
    struct DeferredProcessContextForDeviceError
    {
        Bit64 processId;
        bool isProcessDeferred;
        bool isRequestedToInvokeDeferred;
    };
    NN_STATIC_ASSERT(std::is_pod<DeferredProcessContextForDeviceError>::value);

    struct DeferredProcessContextForPriority
    {
        int prioritySessionType;
        bool isProcessDeferred;
        bool isAcquired;
    };
    NN_STATIC_ASSERT(std::is_pod<DeferredProcessContextForPriority>::value);

private:
    fs::PriorityRaw m_Priority;
    DeferredProcessContextForDeviceError m_DeferredProcessContextForDeviceError;
    DeferredProcessContextForPriority m_DeferredProcessContextForPriority;
    int m_StorageFlag;
    void* m_pRequestHookContext;
};

void RegisterServiceContext(ServiceContext* pContext) NN_NOEXCEPT;

void UnregisterServiceContext() NN_NOEXCEPT;

ServiceContext* GetServiceContext() NN_NOEXCEPT;

const ServiceContext& GetServiceContextReadOnly() NN_NOEXCEPT;

}}
