﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/fssystem/fs_BucketTree.h>

namespace nn { namespace fssystem {

namespace utilTool {
    class IndirectStorageChecker;
}

/**
 * @brief   仮想オフセットを用いて複数のストレージから選択的にデータを読み込むクラスです。
 */
class IndirectStorage : public fs::IStorage, public fs::detail::Newable
{
    NN_DISALLOW_COPY(IndirectStorage);

public:
    typedef MemoryResource IAllocator;

public:
    /**
     * @brief   テーブルのエントリです。
     */
    struct Entry
    {
        char virtualOffset[sizeof(int64_t)];    //!< 仮想オフセット
        char physicalOffset[sizeof(int64_t)];   //!< 実データのオフセット
        int32_t storageIndex;                   //!< 実データを格納したストレージのインデックス

        //! 仮想オフセットを設定します。
        void SetVirtualOffset(const int64_t& offset) NN_NOEXCEPT
        {
            // 8 バイトアライメント対策
            std::memcpy(this->virtualOffset, &offset, sizeof(int64_t));
        }

        //! 仮想オフセットを取得します。
        int64_t GetVirtualOffset() const NN_NOEXCEPT
        {
            // 8 バイトアライメント対策
            int64_t offset;
            std::memcpy(&offset, this->virtualOffset, sizeof(int64_t));
            return offset;
        }

        //! 実データオフセットを設定します。
        void SetPhysicalOffset(const int64_t& offset) NN_NOEXCEPT
        {
            std::memcpy(this->physicalOffset, &offset, sizeof(int64_t));
        }

        //! 実データオフセットを取得します。
        int64_t GetPhysicalOffset() const NN_NOEXCEPT
        {
            int64_t offset;
            std::memcpy(&offset, this->physicalOffset, sizeof(int64_t));
            return offset;
        }
    };
    NN_STATIC_ASSERT(std::is_pod<Entry>::value);

    /**
     * @brief   エントリ情報です。
     */
    struct EntryData
    {
        int64_t virtualOffset;
        int64_t physicalOffset;
        int storageIndex;

        void Set(const Entry& entry) NN_NOEXCEPT
        {
            this->virtualOffset = entry.GetVirtualOffset();
            this->physicalOffset = entry.GetPhysicalOffset();
            this->storageIndex = entry.storageIndex;
        }
    };
    NN_STATIC_ASSERT(std::is_pod<EntryData>::value);

    static const int StorageCount = 2; //!< ストレージの数
    static const size_t NodeSize = 16 * 1024; //!< BucketTree のノードサイズ

public:
    /**
     * @brief   ヘッダ部分のストレージのサイズを取得します。
     *
     * @return  ヘッダ部分のストレージサイズを返します。
     */
    static int64_t QueryHeaderStorageSize() NN_NOEXCEPT
    {
        return BucketTree::QueryHeaderStorageSize();
    }

    /**
     * @brief   ノード部分のストレージのサイズを取得します。
     *
     * @param[in]   entryCount  エントリの総数
     *
     * @return  ノード部分のストレージサイズを返します。
     */
    static int64_t QueryNodeStorageSize(int entryCount) NN_NOEXCEPT
    {
        return BucketTree::QueryNodeStorageSize(NodeSize, sizeof(Entry), entryCount);
    }

    /**
     * @brief   エントリ部分のストレージサイズを取得します。
     *
     * @param[in]   entryCount  エントリの総数
     *
     * @return  エントリ部分のストレージサイズを返します。
     */
    static int64_t QueryEntryStorageSize(int entryCount) NN_NOEXCEPT
    {
        return BucketTree::QueryEntryStorageSize(NodeSize, sizeof(Entry), entryCount);
    }

public:
    /**
     * @brief   コンストラクタです。
     */
    IndirectStorage() NN_NOEXCEPT;

    /**
     * @brief   デストラクタです。
     */
    virtual ~IndirectStorage() NN_NOEXCEPT NN_OVERRIDE
    {
        Finalize();
    }

    /**
     * @brief   初期化します。（テスト向け）
     *
     * @param[in]   pAllocator      アロケータのポインタ
     * @param[in]   tableStorage    論物変換テーブルを格納したストレージ
     *
     * @retval  ResultSuccess                   正常に処理が終了しました。
     * @retval  ResultBufferAllocationFailed    メモリの確保に失敗しました。
     * @retval  上記以外                        ストレージの読み込みに失敗しました。
     *
     * @pre
     *      - pAllocator != nullptr
     *      - 未初期化
     */
    Result Initialize(IAllocator* pAllocator, fs::SubStorage tableStorage) NN_NOEXCEPT;

    /**
     * @brief   初期化します。
     *
     * @param[in]   pAllocator      アロケータのポインタ
     * @param[in]   nodeStorage     テーブル検索のための木構造を格納したストレージ
     * @param[in]   entryStorage    論物変換用のテーブルデータを格納したストレージ
     * @param[in]   entryCount      テーブルデータのエントリの総数
     *
     * @retval  ResultSuccess                   正常に処理が終了しました。
     * @retval  ResultBufferAllocationFailed    メモリの確保に失敗しました。
     * @retval  上記以外                        ストレージの読み込みに失敗しました。
     *
     * @pre
     *      - pAllocator != nullptr
     *      - 0 < entryCount
     *      - 未初期化
     *
     * nodeStorage, entryStorage で使用するストレージは、4KB アライメントすることを強く推奨します。
     */
    Result Initialize(
               IAllocator* pAllocator,
               fs::SubStorage nodeStorage,
               fs::SubStorage entryStorage,
               int entryCount
           ) NN_NOEXCEPT
    {
        return m_Table.Initialize(
                           pAllocator,
                           nodeStorage,
                           entryStorage,
                           NodeSize,
                           sizeof(Entry),
                           entryCount
                       );
    }

    /**
     * @brief   終了処理をします。
     */
    void Finalize() NN_NOEXCEPT;

    /**
     * @brief   初期化済みかどうかを取得します。
     *
     * @retval true     初期化済みです。
     * @retval false    初期化していません。
     */
    bool IsInitialized() const NN_NOEXCEPT
    {
        return m_Table.IsInitialized();
    }

    /**
     * @brief   データが存在するストレージを設定します。
     *
     * @param[in]   index       ストレージのインデックス
     * @param[in]   storage     ストレージ
     */
    void SetStorage(int index, fs::SubStorage storage) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_RANGE(index, 0, StorageCount);

        m_DataStorage[index] = storage;
    }

    /**
     * @brief   データが存在するストレージを設定します。
     *
     * @param[in]   index       ストレージのインデックス
     * @param[in]   pStorage    ストレージのポインタ
     * @param[in]   offset      ストレージのオフセット
     * @param[in]   size        ストレージのサイズ
     */
    template< typename T >
    void SetStorage(int index, T pStorage, int64_t offset, int64_t size) NN_NOEXCEPT
    {
        // その他の引数の事前検証は SubStorage に任せる
        NN_SDK_REQUIRES_RANGE(index, 0, StorageCount);

        m_DataStorage[index] = fs::SubStorage(pStorage, offset, size);
    }

    /**
     * @brief   データを読み込みます。
     *
     * @param[in]   offset  読み込むオフセット
     * @param[out]  buffer  読み込んだデータを格納するバッファ
     * @param[in]   size    読み込むサイズ
     *
     * @return  関数の実行結果を返します。
     *
     * @pre
     *      - 0 <= offset
     *      - size == 0 || buffer != nullptr
     *      - 初期化済み
     *
     * この関数は、以下の条件を満たした場合に限りスレッドセーフとなります。
     *      - Initialize() で指定したストレージすべてがスレッドセーフ
     *      - AddStorage() で指定したストレージすべてがスレッドセーフ
     */
    virtual Result Read(int64_t offset, void* buffer, size_t size) NN_NOEXCEPT NN_OVERRIDE;

    virtual Result Flush() NN_NOEXCEPT NN_OVERRIDE
    {
        NN_RESULT_SUCCESS;
    }

    /**
     * @brief   ストレージのサイズを取得します。
     *
     * @return  関数の実行結果を返します。
     *
     * @pre
     *      - pOutSize != nullptr
     */
    virtual Result GetSize(int64_t* pOutSize) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_REQUIRES_NOT_NULL(pOutSize);

        *pOutSize = m_Table.GetSize();

        NN_RESULT_SUCCESS;
    }

    /**
     * @brief       範囲指定処理を行います。
     *
     * @param[out]  outBuffer        範囲指定処理の結果を格納するバッファ
     * @param[in]   outBufferSize    範囲指定処理の結果を格納するバッファのサイズ
     * @param[in]   operationId      範囲指定処理の種類
     * @param[in]   offset           範囲指定処理開始位置
     * @param[in]   size             範囲指定処理を行うデータサイズ
     * @param[in]   inBuffer         範囲指定処理に渡すバッファ
     * @param[in]   inBufferSize     範囲指定処理に渡すバッファのサイズ
     *
     * @return      関数の処理結果を返します。
     */
    virtual Result OperateRange(
                       void* outBuffer,
                       size_t outBufferSize,
                       fs::OperationId operationId,
                       int64_t offset,
                       int64_t size,
                       const void* inBuffer,
                       size_t inBufferSize
                   ) NN_NOEXCEPT NN_OVERRIDE;

    /**
     * @brief   指定した範囲に対応するエントリリストを取得します。（AuthoringTool 向け）
     *
     * @param[out]  outEntries     エントリリストを格納するポインタ
     * @param[out]  outEntryCount  エントリリストの長さ
     * @param[in]   entryCount     取得するエントリリストの長さ
     * @param[in]   offset         範囲のオフセット
     * @param[in]   size           範囲のサイズ
     *
     * @return  関数の実行結果を返します。
     *
     * @pre
     *      - 0 <= offset
     *      - outEntryCount != nullptr
     *      - entryCount == 0 || outEntries != nullptr
     *      - outEntries が指す格納領域のサイズが sizeof(Entry) * entryCount 以上
     *      - 初期化済み
     *
     * entryCount == 0 で実行した場合、outEntryCount にエントリリストの長さが返ります。
     * entryCount != 0 で実行した場合、outEntryCount には取得したエントリリストの長さが返ります。
     */
    Result GetEntryList(Entry* outEntries, int* outEntryCount, int entryCount, int64_t offset, int64_t size) NN_NOEXCEPT;

protected:
    BucketTree& GetEntryTable() NN_NOEXCEPT
    {
        return m_Table;
    }

    fs::SubStorage& GetDataStorage(int index) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_RANGE(index, 0, StorageCount);

        return m_DataStorage[index];
    }

    /**
     * @brief       指定範囲に対してエントリごとに処理を行います
     *
     * @param[in]   offset  処理を行う範囲の開始位置
     * @param[in]   size    処理を行うデータサイズ
     * @param[in]   func    各エントリに対して実行する関数
     *
     * @return      関数の処理結果を返します。
     */
    template< bool IsContinuousCheckNeeded, typename TFunc >
    Result OperatePerEntry(int64_t offset, int64_t size, TFunc func) NN_NOEXCEPT;

private:
    /**
     * @brief   データをまとめて読み込む処理の解析に使用するエントリ情報です。
     */
    struct ContinuousReadingEntry
    {
        static const int FragmentSizeMax = 4 * 1024;

        IndirectStorage::Entry entry;

        int64_t GetVirtualOffset() const NN_NOEXCEPT
        {
            return this->entry.GetVirtualOffset();
        }

        int64_t GetPhysicalOffset() const NN_NOEXCEPT
        {
            return this->entry.GetPhysicalOffset();
        }

        bool IsFragment() const NN_NOEXCEPT
        {
            return this->entry.storageIndex != 0;
        }
    };
    NN_STATIC_ASSERT(std::is_pod<ContinuousReadingEntry>::value);

private:
    BucketTree m_Table;
    fs::SubStorage m_DataStorage[StorageCount];

    friend class IndirectStorageTest;
    friend class IndirectStorageBuilder;
    friend class utilTool::IndirectStorageChecker;
};

}}
