﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>

#include <nn/fssystem/fs_DbmRomTypes.h>
#include "fs_DbmRomPathTool.h"
#include "fs_DbmKeyValueRomStorageTemplate.h"

#ifdef __cplusplus

namespace nn { namespace fssystem {

/*!
    :private

    @brief ROM ファイルシステム専用ファイル、ディレクトリエントリー管理テーブルクラスです。


    以下のような操作を行った場合はエラーを返します。

    ・パスに存在しないディレクトリが含まれている場合は ResultDbmDirectoryNotFound を返します。
    ・パスに含まれているディレクトリ名が長すぎる場合は ResultDbmDirectoryNameTooLong を返します。
    ・パスに含まれているファイル名が長すぎる場合は ResultDbmFileNameTooLong を返します。
    ・パスに含まれている区切り文字が複数回連続している場合には ResultDbmInvalidPathFormat を返します。
    ・パスの 1 文字目が "/" でない場合は ResultDbmInvalidPathFormat を返します。
    ・ファイルに対してディレクトリ操作を行おうとした場合は ResultDbmInvalidOperation を返します。
    ・ディレクトリに対してファイル操作を行おうとした場合は ResultDbmInvalidOperation を返します。

    作成処理用エラー
    ・すでに存在しているディレクトリ、ファイルを作成しようとした場合は ResultDbmAlreadyExists を返します。
*/
template <
    typename DirectoryBucketStorage_,
    typename DirectoryEntryStorage_,
    typename FileBucketStorage_,
    typename FileEntryStorage_
>
class HierarchicalRomFileTableTemplate
{
private:
    typedef DirectoryBucketStorage_ DirectoryBucketStorage;
    typedef DirectoryEntryStorage_ DirectoryEntryStorage;
    typedef FileBucketStorage_ FileBucketStorage;
    typedef FileEntryStorage_ FileEntryStorage;

public:
    //!< オフセット
    typedef uint32_t StoragePosition;

    //!< ファイル探索位置指示構造体です。
    struct FindPosition
    {
        StoragePosition nextPositionDirectory;     //!< 探索中のディレクトリインデックス
        StoragePosition nextPositionFile;          //!< 探索中のファイルインデックス
    };
    NN_STATIC_ASSERT(std::is_pod<FindPosition>::value);

    typedef RomDirectoryInfo DirectoryInfo;
    typedef RomFileInfo FileInfo;

    //<! 位置データをファイル ID に変換します。
    static inline RomFileId PositionToFileId(StoragePosition pos)
    {
        return static_cast<RomFileId>(pos);
    }

private:
    //!< 無効な位置です。
    // (KeyValueStorageTemplate::KVSTORAGE_FREEENTRY と同じ値を使用)
    static const StoragePosition POSITION_NONE = 0xFFFFFFFF;

    //!< ルートディレクトリを示す位置です。
    static const StoragePosition POSITION_ROOT_DIRECTORY = 0;

    //<! 位置データをディレクトリ ID に変換します。
    static inline RomDirectoryId PositionToDirectoryId(StoragePosition pos)
    {
        return static_cast<RomDirectoryId>(pos);
    }

    //!< ディレクトリ ID を位置データに変換します。
    static inline StoragePosition DirectoryIdToPosition(RomDirectoryId dirId)
    {
        return static_cast<StoragePosition>(dirId);
    }

    //!< ファイル ID を位置データに変換します。
    static inline StoragePosition FileIdToPosition(RomFileId fileId)
    {
        return static_cast<StoragePosition>(fileId);
    }

    //!< ディレクトリエントリーです。
    struct DirectoryEntry
    {
        StoragePosition  posNext;           //!< 兄弟へのインデックス
        StoragePosition  posDirectory;      //!< 子ディレクトリー内にあるディレクトリのインデックス
        StoragePosition  posFile;           //!< 子ディレクトリー内にあるファイルのインデックス
#ifdef NN_DBM_ENABLE_DIRECTORY_METADATA
        DirectoryInfo    info;              //!< ディレクトリ付加情報
#endif
    };
    NN_STATIC_ASSERT(std::is_pod<DirectoryEntry>::value);

    //!< ファイルエントリーです。
    struct FileEntry
    {
        StoragePosition  posNext;           //!< 兄弟へのインデックス
        FileInfo         info;              //!< ファイル付加情報
    };
    NN_STATIC_ASSERT(std::is_pod<FileEntry>::value);

    //!< エントリーマップテーブルの共通実装
    static const uint32_t MAX_KEY_LENGTH = RomPathTool::MAX_PATH_LENGTH;
    template <
        typename BucketStorage_,
        typename EntryStorage_,
        typename Key_,
        typename Key2_,
        typename Value_
    >
    class EntryMapTable
        : public KeyValueRomStorageTemplate<
                     BucketStorage_,
                     EntryStorage_,
                     Key_,
                     Value_,
                     MAX_KEY_LENGTH
                 >
    {
    public:
        typedef Key_ Key;
        typedef Key2_ Key2;
        typedef Value_ Value;
        typedef uint32_t Position;
        typedef KeyValueRomStorageTemplate <
                     BucketStorage_,
                     EntryStorage_,
                     Key_,
                     Value_,
                     MAX_KEY_LENGTH
                >
                BaseClass;

    public:
        //!< KeyValueStorageTemplate::AddInternal を公開します。
        inline Result Add(Position* pOutPosition, const Key2& key, const Value& value)
        {
            return BaseClass::AddInternal(
                       pOutPosition,
                       key.key,
                       key.Hash(),
                       key.name.path,
                       key.name.length * sizeof(RomPathChar),
                       value
                   );
        }

        //!< KeyValueStorageTemplate::GetInternal を公開します。
        inline Result Get(Position* pOutPosition, Value* pValue, const Key2& key) const
        {
            return BaseClass::GetInternal(
                       pOutPosition,
                       pValue,
                       key.key,
                       key.Hash(),
                       key.name.path,
                       key.name.length * sizeof(RomPathChar)
                   );
        }

        //!< KeyValueStorageTemplate::GetByPosition を公開します。
        inline Result GetByPosition(Key* pKey, Value* pValue, void* pExtraKey, size_t* pExtraSize, Position pos) const
        {
            return BaseClass::GetByPosition(pKey, pValue, pExtraKey, pExtraSize, pos);
        }

        //!< KeyValueStorageTemplate::SetByPosition を公開します。
        inline Result SetByPosition(Position pos, const Value& value, nn::fs::WriteOption option) const
        {
            return BaseClass::SetByPosition(pos, value, option);
        }

    };

    //!< エントリーを管理するハッシュテーブル用固定長キー
    struct RomEntryKey
    {
        StoragePosition parentDir;          //!< 親ディレクトリ

        //!< キーの比較を行います。
        inline bool IsEqual(const RomEntryKey& rKey, const void* pExtraKey1, size_t extraSize1, const void* pExtraKey2, size_t extraSize2) const
        {
            if (this->parentDir != rKey.parentDir)
            {
                return false;
            }
            if (extraSize1 != extraSize2)
            {
                return false;
            }
            return RomPathTool::IsEqualPath(reinterpret_cast<const RomPathChar *>(pExtraKey1), reinterpret_cast<const RomPathChar *>(pExtraKey2), (extraSize1 / sizeof(RomPathChar)));
        }
    };
    NN_STATIC_ASSERT(std::is_pod<RomEntryKey>::value);

    //!< エントリーを管理するハッシュテーブル用キー
    struct EntryKey
    {
        RomEntryKey key;                    //!< キ－
        RomPathTool::RomEntryName name;     //!< 名前

        //!< ハッシュ値を取得します。
        inline uint32_t Hash() const
        {
            uint32_t v = 123456789;
            v ^= key.parentDir;
            const RomPathChar* pName = name.path;
            const RomPathChar* pNameEnd = pName + name.length;
            while (pName < pNameEnd)
            {
                // *pName の値を符号なしに統一します。
                const auto unsignedNameChar
                    = static_cast<std::make_unsigned<RomPathChar>::type>(*pName);
                v = ((v >> 5) | (v << (32 - 5))) ^ unsignedNameChar;
                pName ++;
            }
            return v;
        }
    };
    NN_STATIC_ASSERT(std::is_pod<EntryKey>::value);

    //!< ディレクトリエントリーを管理するハッシュテーブル
    typedef EntryMapTable<DirectoryBucketStorage, DirectoryEntryStorage, RomEntryKey, EntryKey, DirectoryEntry> DirectoryEntryMapTable;

    //!< ファイルエントリーを管理するハッシュテーブル
    typedef EntryMapTable<FileBucketStorage, FileEntryStorage, RomEntryKey, EntryKey, FileEntry> FileEntryMapTable;

    DirectoryEntryMapTable m_TableDirectory;    //!< ディレクトリ管理テーブル
    FileEntryMapTable m_TableFile;              //!< ファイル管理テーブル

public:

#ifdef NN_DBM_CREATE_METADATA

    /*!
        @brief ディレクトリエントリーに必要なサイズを求めます。

               初回ファイルツリー作成(メタデータ構築)時に使用します。

        @param[in] entryNameLength エントリー名の長さ(終端のNULL文字を含まない)

        @return ディレクトリエントリーに必要なサイズ
    */
    static uint32_t QueryDirectoryEntrySize(uint32_t entryNameLength);

    /*!
        @brief ファイルエントリーに必要なサイズを求めます。

               初回ファイルツリー作成(メタデータ構築)時に使用します。

        @param[in] entryNameLength エントリー名の長さ(終端のNULL文字を含まない)

        @return ファイルエントリーに必要なサイズ
    */
    static uint32_t QueryFileEntrySize(uint32_t entryNameLength);

    /*!
        @brief バケット数から必要なストレージのサイズを求めます。

        ・バケット

           ・"バケット数"とは、ファイル名の探索に使用するハッシュテーブルのサイズです。

           ・バケット数の大きさに比例し、ファイルの探索速度が向上します。

           ・総ディレクトリ数や総ファイル数の数分の一を目安で与えます。

           ・バケット数はフォーマット以降拡張できません。

        @param[in] countDirectoryBucket ディレクトリバケット数
        @param[in] countFileBucket ファイルバケット数

        @return 必要なストレージのサイズ
    */
    static uint32_t QueryDirectoryEntryBucketStorageSize(uint32_t countDirectoryBucket);
    static uint32_t QueryFileEntryBucketStorageSize(uint32_t countFileBucket);

    /*!
        @brief ファイル管理テーブルをフォーマットします。

               IStorage を初期化します。使用する前に一度呼び出します。

        @param[in] pDirectoryBucket ディレクトリエントリーバケット用ストレージ
        @param[in] offsetDirectoryBucket ディレクトリエントリーバケット用ストレージに割り当てる領域のオフセット
        @param[in] sizeDirectoryBucket ディレクトリエントリーバケット用ストレージに割り当てる領域のサイズ(要素数)
        @param[in] pDirectoryEntry ディレクトリエントリー用ストレージ
        @param[in] offsetDirectoryEntry ディレクトリエントリー用ストレージに割り当てる領域のオフセット
        @param[in] sizeDirectoryEntry ディレクトリエントリー用ストレージに割り当てる領域のサイズ(バイト数)
        @param[in] pFileBucket ファイルエントリーバケット用ストレージ
        @param[in] offsetFileBucket ファイルエントリーバケット用ストレージに割り当てる領域のオフセット
        @param[in] sizeFileBucket ファイルエントリーバケット用ストレージに割り当てる領域のサイズ(要素数)
        @param[in] pFileEntry ファイルエントリー用ストレージ
        @param[in] offsetFileEntry ファイルエントリー用ストレージに割り当てる領域のオフセット
        @param[in] sizeFileEntry ファイルエントリー用ストレージに割り当てる領域のサイズ(バイト数)

        @return 関数の処理結果を返します。
    */
    static Result Format(
                      DirectoryBucketStorage* pDirectoryBucket,
                      int64_t offsetDirectoryBucket,
                      uint32_t sizeDirectoryBucket,
                      DirectoryEntryStorage* pDirectoryEntry,
                      int64_t offsetDirectoryEntry,
                      uint32_t sizeDirectoryEntry,
                      FileBucketStorage* pFileBucket,
                      int64_t offsetFileBucket,
                      uint32_t sizeFileBucket,
                      FileEntryStorage* pFileEntry,
                      int64_t offsetFileEntry,
                      uint32_t sizeFileEntry
                 );
#endif // NN_DBM_CREATE_METADATA

    /*!
        @brief  コンストラクタです。
    */
    HierarchicalRomFileTableTemplate();

    /*!
        @brief  ストレージをマウントします。

        @param[in] pDirectoryBucket ディレクトリバケット用ストレージ
        @param[in] offsetDirectoryBucket ディレクトリバケットに割り当てる領域のオフセット
        @param[in] sizeDirectoryBucket ディレクトリバケットに割り当てる領域のサイズ(要素数)
        @param[in] pDirectoryEntry ディレクトリエントリー用ストレージ
        @param[in] offsetDirectoryEntry ディレクトリエントリーに割り当てる領域のオフセット
        @param[in] sizeDirectoryEntry ディレクトリエントリーに割り当てる領域のサイズ(バイト数)
        @param[in] pFileBucket ファイルエントリーバケット用ストレージ
        @param[in] offsetFileBucket ファイルエントリーバケットに割り当てる領域のオフセット
        @param[in] sizeFileBucket ファイルエントリーバケットに割り当てる領域のサイズ(要素数)
        @param[in] pFileEntry ファイルエントリー用ストレージ
        @param[in] offsetFileEntry ファイルエントリーに割り当てる領域のオフセット
        @param[in] sizeFileEntry ファイルエントリーに割り当てる領域のサイズ(バイト数)

        @return 関数の処理結果を返します。
    */
    Result Initialize(
               DirectoryBucketStorage* pDirectoryBucket,
               int64_t offsetDirectoryBucket,
               uint32_t sizeDirectoryBucket,
               DirectoryEntryStorage* pDirectoryEntry,
               int64_t offsetDirectoryEntry,
               uint32_t sizeDirectoryEntry,
               FileBucketStorage* pFileBucket,
               int64_t offsetFileBucket,
               uint32_t sizeFileBucket,
               FileEntryStorage* pFileEntry,
               int64_t offsetFileEntry,
               uint32_t sizeFileEntry
           );

    /*!
        @brief  ストレージをアンマウントします。
    */
    void Finalize(void);

#ifdef NN_DBM_CREATE_METADATA
    /*!
        @brief ルートディレクトリエントリーを作成します。

        @return 関数の処理結果を返します。
    */
    Result CreateRootDirectory();

    /*!
        @brief ディレクトリを作成します。

        @param[out] pDirId ディレクトリ ID
        @param[in] pFullPath ディレクトリパス
        @param[in] dirInfo ディレクトリ情報

        @return 関数の処理結果を返します。
    */
    Result CreateDirectory(RomDirectoryId* pDirId, const RomPathChar* pFullPath, const DirectoryInfo& dirInfo);

    /*!
        @brief ファイルを作成します。

        @param[out] pFileId ディレクトリ ID
        @param[in] pFullPath ファイルパス
        @param[in] fileInfo ファイル情報

        @return 関数の処理結果を返します。
    */
    Result CreateFile(RomFileId* pFileId, const RomPathChar* pFullPath, const FileInfo& fileInfo);
#endif // NN_DBM_CREATE_METADATA

    /*!
        @brief 指定したフルパスをディレクトリ ID に変換します。

        @param[out] pDIrID ディレクトリ ID
        @param[in] pFullPath ディレクトリパス

        @return 関数の処理結果を返します。
    */
    Result ConvertPathToDirectoryId(RomDirectoryId* pDirId, const RomPathChar* pFullPath) const;

    /*!
        @brief 指定したフルパスをファイル ID に変換します。

        @param[out] pFileId ファイル ID
        @param[in] pFullPath ファイルパス

        @return 関数の処理結果を返します。
    */
    Result ConvertPathToFileId(RomFileId* pFileId, const RomPathChar* pFullPath) const;

    /*!
        @brief 指定したディレクトリ情報を取得します。

        @param[out] pDirInfo ディレクトリ情報
        @param[in] pFullPath ディレクトリパス
        @param[in] dirId ディレクトリ ID

        @return 関数の処理結果を返します。
    */
    Result GetDirectoryInformation(DirectoryInfo* pDirInfo, const RomPathChar* pFullPath) const;
    Result GetDirectoryInformation(DirectoryInfo* pDirInfo, RomDirectoryId dirId) const;

    /*!
        @brief ファイルを開きます。

        @param[out] pFullPath ファイル情報
        @param[in] pFullPath ファイルパス
        @param[in] fileId ファイル ID

        @return 関数の処理結果を返します。
    */
    Result OpenFile(FileInfo* pFileInfo, const RomPathChar* pFullPath) const;
    Result OpenFile(FileInfo* pFileInfo, RomFileId fileId) const;

    /*!
        @brief 指定したディレクトリ以下のファイル列挙を開始します。

               イテレータを @ref FindNextDirectory、@ref FindNextFile に
               渡すことでディレクトリ名、ファイル名をイテレーションできます。
               また、イテレーションはファイル、ディレクトリ各々に対して行うことができます。

        @param[in,out] pfi イテレータ
        @param[in] pFullPath ディレクトリパス
        @param[in] dirId ディレクトリ ID

        @return 関数の処理結果を返します。
    */
    Result FindOpen(FindPosition* pfi, const RomPathChar* pFullPath) const;
    Result FindOpen(FindPosition* pfi, RomDirectoryId dirId) const;

    /*!
        @brief ディレクトリ以下の次のディレクトリを取得します。

               この関数の前に @ref FindOpen でイテレータを初期化する必要があります。

        @param[out] pDirectoryName ディレクトリ名を格納するバッファm_TableFile
                                   サイズ RomPathTool::MAX_PATH_LENGTH + 1 のバッファを渡してください。
        @param[in,out] pfi イテレータ

        @return 関数の処理結果を返します。
                イテレーション完了時には ResultDbmFindFinished を返します。
    */
    Result FindNextDirectory(RomPathChar* pDirectoryName, FindPosition* pfi) const;

    /*!
        @brief ディレクトリ以下の次のファイルを取得します。

               この関数の前に @ref FindOpen でイテレータを初期化する必要があります。

        @param[out] pFileName ベースファイル名
                              サイズ RomPathTool::MAX_PATH_LENGTH + 1 のバッファを渡してください。
        @param[in,out] pfi イテレータ

        @return 関数の処理結果を返します。
                イテレーション完了時には ResultDbmFindFinished を返します。
    */
    Result FindNextFile(RomPathChar* pFileName, FindPosition* pfi) const;

#ifdef NN_DBM_CREATE_METADATA
    /*!
        @brief ファイルシステム構築に必要なバッファサイズを取得します。

        @param[out] pSize ROM ファイルシステムに必要なサイズ

        @return 関数の処理結果を返します。
    */
    Result QueryRomFileSystemSize(
               uint32_t* pSizeDirectoryEntry,
               uint32_t* pSizeFileEntry
           );

    //!< ディレクトリエントリー数を取得します。
    inline uint32_t GetDirectoryEntryCount() const
    {
        return m_TableDirectory.GetEntryCount();
    }

    //!< ファイルエントリー数を取得します。
    inline uint32_t GetFileEntryCount() const
    {
        return m_TableFile.GetEntryCount();
    }
#endif // NN_DBM_CREATE_METADATA

#if !defined(NN_SWITCH_DISABLE_DEBUG_PRINT_FOR_SDK) && !defined(NN_COMPATIBLE_SDK) && defined(NN_DBM_CREATE_METADATA) && defined(_MSC_VER)
    /*!
        @brief ファイル構成をダンプします。

               デバッグ用途の関数です。内部でスタックを多く消費します。

        @return 関数の処理結果を返します。
    */
    Result DumpTree();
#endif // !defined(NN_SWITCH_DISABLE_DEBUG_PRINT_FOR_SDK) && !defined(NN_COMPATIBLE_SDK) && defined(NN_DBM_CREATE_METADATA) && defined(_MSC_VER)

private:

    /*!
        :private

        @brief 親の親ディレクトリを取得します。

        @param[out] pPosition 親の親ディレクトリエントリーのストレージ位置
        @param[out] pDirKey 親の親ディレクトリキー
        @param[out] pDirEntry 親の親ディレクトリエントリー
        @param[in] pos 親ディレクトリエントリーのストレージ位置
        @param[in] name 親ディレクトリ名
        @param[in] pFileName フルパス

        @return 関数の処理結果を返します。
    */
    Result GetGrandparent(
               StoragePosition* pPosition,
               EntryKey* pDirKey,
               DirectoryEntry* pDirEntry,
               StoragePosition pos,
               RomPathTool::RomEntryName name,
               const RomPathChar* pFullPath
          ) const;

    /*!
        :private

        @brief 指定した親ディレクトリを取得します。

        @param[out] pParentPosition 親ディレクトリエントリーのストレージ位置
        @param[out] pParentDirKey 親ディレクトリキー
        @param[out] pParentDirEntry 親ディレクトリエントリー
        @param[in] parser パスパーサー

        @return 関数の処理結果を返します。
    */
    Result FindParentDirRecursive(
               StoragePosition* pParentPosition,
               EntryKey* pParentDirKey,
               DirectoryEntry* pParentDirEntry,
               RomPathTool::PathParser& parser,
               const RomPathChar* pFullPath
           ) const;

    /*!
        :private

        @brief 指定したパスからディレクトまたはファイルを取得します。

        @param[out] pParentDirKey 親ディレクトリキー
        @param[out] pParentDirEntry 親ディレクトリエントリー
        @param[out] pKey エントリーのキー
        @param[in] bFindDir ディレクトリエントリーかどうか
        @param[in] pFullPath 分離するフルパス

        @return 関数の処理結果を返します。
    */
    Result FindPathRecursive(
               EntryKey* pParentDirKey,
               DirectoryEntry* pParentDirEntry,
               EntryKey* pKey,
               bool bFindDir,
               const RomPathChar* pFullPath
           ) const;

    /*!
        :private

        @brief 指定したパスから親ディレクトとディレクトリキーを取得します。

        @param[out] pParentDirKey 親ディレクトリキー
        @param[out] pParentDirEntry 親ディレクトリエントリー
        @param[out] pDirKey ディレクトリキー
        @param[in] pFullPath 分離するフルパス

        @return 関数の処理結果を返します。
    */
    Result FindDirectoryRecursive(
               EntryKey* pParentDirKey,
               DirectoryEntry* pParentDirEntry,
               EntryKey* pDirKey,
               const RomPathChar* pFullPath
           ) const;

    /*!
        :private

        @brief 指定したパスから親ディレクトリとファイルキーを取得します。

        @param[out] pParentDirKey 親ディレクトリキー
        @param[out] pParentDirEntry 親ディレクトリエントリー
        @param[out] pFileKey ファイルキー
        @param[in] pFullPath 分離するフルパス

        @return 関数の処理結果を返します。
    */
    Result FindFileRecursive(
               EntryKey* pParentDirKey,
               DirectoryEntry* pParentDirEntry,
               EntryKey* pFileKey,
               const RomPathChar* pFullPath
           ) const;

#ifdef NN_DBM_CREATE_METADATA
    /*!
        :private

        @brief 指定したエントリーと同名のファイル、ディレクトリが存在するかどうかを取得します。

        @param[in] key キー
        @param[in] resultIfExist 同名のエントリーが存在したときの結果

        @return 関数の処理結果を返します。
    */
    Result CheckSameEntryExists(
               const EntryKey& key,
               Result resultIfExist
           ) const;
#endif // NN_DBM_CREATE_METADATA

    /*!
        :private

        @brief ディレクトリエントリーを取得します。

        @param[out] pPosition ディレクトリエントリーのストレージ位置
        @param[out] pEntry ディレクトリエントリー
        @param[in] key ディレクトリキー

        @return 関数の処理結果を返します。
    */
    Result GetDirectoryEntry(StoragePosition* pPosition, DirectoryEntry* pEntry, const EntryKey& key) const;

    /*!
        :private

        @brief ディレクトリエントリーを取得します。

        @param[out] pEntry ディレクトリエントリー
        @param[in] dirId ディレクトリ ID

        @return 関数の処理結果を返します。
    */
    Result GetDirectoryEntry(DirectoryEntry* pEntry, RomDirectoryId dirId) const;

    /*!
        :private

        @brief ファイルエントリーを取得します。

        @param[out] pPosition ファイルエントリーのストレージ位置
        @param[out] pEntry ファイルエントリー
        @param[in] key ファイルキー

        @return 関数の処理結果を返します。
    */
    Result GetFileEntry(StoragePosition* pPosition, FileEntry* pEntry, const EntryKey& key) const;

    /*!
        :private

        @brief ファイルエントリーを取得します。

        @param[out] pEntry ファイルエントリー
        @param[in] fileId ファイル ID

        @return 関数の処理結果を返します。
    */
    Result GetFileEntry(FileEntry* pEntry, RomFileId fileId) const;

    /*!
        :private

        @brief 指定したディレクトリ情報を取得します。

        @param[out] pDirInfo ディレクトリ情報
        @param[in] dirKey ディレクトリキー

        @return 関数の処理結果を返します。
    */
    Result GetDirectoryInformation(DirectoryInfo* pDirInfo, const EntryKey& dirKey) const;

    /*!
        :private

        @brief ファイルを開きます。

        @param[out] pFileInfo ファイル情報
        @param[in] fileKey ファイルキー

        @return 関数の処理結果を返します。
    */
    Result OpenFile(FileInfo* pFileInfo, const EntryKey& fileKey) const;

    /*!
        @brief 指定したディレクトリ以下のファイル列挙を開始します。

               イテレータを @ref FindNextDirectory、@ref FindNextFile に
               渡すことでディレクトリ名、ファイル名をイテレーションできます。
               また、イテレーションはファイル、ディレクトリ各々に対して行うことができます。

        @param[in,out] pfi イテレータ
        @param[in] dirKey ディレクトリキー

        @return 関数の処理結果を返します。
    */
    Result FindOpen(FindPosition* pfi, const EntryKey& dirKey) const;

#if !defined(NN_SWITCH_DISABLE_DEBUG_PRINT_FOR_SDK) && !defined(NN_COMPATIBLE_SDK) && defined(NN_DBM_CREATE_METADATA) && defined(_MSC_VER)
    /*!
        :private

        @brief 指定したディレクトリ以下のファイルツリーをデバッグ表示します。

        @param[in] posDirectory ディレクトリ位置
        @param[in] depth ファイル階層の深さ

        @return 関数の処理結果を返します。
    */
    Result DumpDirectory(StoragePosition posDirectory, int depth);
#endif // !defined(NN_SWITCH_DISABLE_DEBUG_PRINT_FOR_SDK) && !defined(NN_COMPATIBLE_SDK) && defined(NN_DBM_CREATE_METADATA) && defined(_MSC_VER)

    //!< テストのため TestSuite に対して実装を公開します。
    friend class HierarchicalRomFileTableTemplateTestSuite;
};

}}

#endif // __cplusplus
