﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/fssystem/fs_AsynchronousAccess.h>
#include <nn/fssystem/fs_BucketTreeUtility.h>

namespace nn { namespace fssystem {

/**
 * @brief   ストレージのデータをまとめて読み込めるか確認します。
 *
 * 物理アドレスが連続した複数のデータを 1 度にまとめて読み込んだ後に、
 * 不連続で小さなデータ（断片）で上書きするための情報を取得します。
 *
 * 該当のデータが断片かどうかは、TEntry の実装に従います。
 */
template< typename TEntry >
Result BucketTree::ScanContinuousReading(
                       ContinuousReadingInfo* pOutInfo,
                       const ContinuousReadingParam<TEntry>& param
                   ) const NN_NOEXCEPT
{
    NN_STATIC_ASSERT(std::is_pod<ContinuousReadingParam<TEntry>>::value);
    NN_SDK_REQUIRES(IsInitialized());
    NN_SDK_REQUIRES_NOT_NULL(pOutInfo);
    NN_SDK_REQUIRES_EQUAL(sizeof(TEntry), m_EntrySize);

    pOutInfo->Reset();

    if( param.size == 0 )
    {
        NN_RESULT_SUCCESS;
    }

    // 断片として扱うデータが先頭ならまとめて読み込まない
    if( param.entry.IsFragment() )
    {
        NN_RESULT_SUCCESS;
    }

    auto entry1 = param.entry;
    auto currentOffset = param.offset;

    NN_RESULT_THROW_UNLESS(
        entry1.GetVirtualOffset() <= currentOffset, fs::ResultOutOfRange());

    PooledBuffer pool(m_NodeSize, 1);
    char* buffer = nullptr;

    // ノードを先読み
    if( m_NodeSize <= pool.GetSize() )
    {
        buffer = pool.GetBuffer();

        const auto offset = param.entrySet.index * static_cast<int64_t>(m_NodeSize);
        NN_RESULT_DO(m_EntryStorage.Read(offset, buffer, m_NodeSize));
    }

    const auto endOffset = currentOffset + static_cast<int64_t>(param.size);
    int64_t physicalOffset = entry1.GetPhysicalOffset();
    auto entryIndex = param.entryIndex;
    int64_t mergeSize = 0;
    bool isMerged = false;
    int64_t readableSize = 0;

    // エントリセットをまたぐ判定はしない
    for( const auto entryCount = param.entrySet.count; entryIndex < entryCount; ++entryIndex )
    {
        // 読み込むデータの終端に到着
        if( endOffset <= currentOffset )
        {
            break;
        }

        const auto entry1Offset = entry1.GetVirtualOffset();

        // entry1Offset(または currentOffset) が不正
        NN_RESULT_THROW_UNLESS(
            entry1Offset <= currentOffset, fs::ResultInvalidIndirectEntryOffset());

        TEntry entry2 = {};
        int64_t entry2Offset;

        if( entryIndex + 1 < entryCount )
        {
            if( buffer != nullptr )
            {
                const auto entryOffset =
                    detail::GetBucketTreeEntryOffset(0, m_EntrySize, entryIndex + 1);

                std::memcpy(&entry2, buffer + entryOffset, m_EntrySize);
            }
            else
            {
                const auto entryOffset = detail::GetBucketTreeEntryOffset(
                                             param.entrySet.index,
                                             m_NodeSize,
                                             m_EntrySize,
                                             entryIndex + 1
                                         );
                NN_RESULT_DO(m_EntryStorage.Read(entryOffset, &entry2, m_EntrySize));
            }

            entry2Offset = entry2.GetVirtualOffset();

            NN_RESULT_THROW_UNLESS(
                IsInclude(entry2Offset), fs::ResultInvalidIndirectEntryOffset());
        }
        else
        {
            entry2Offset = param.entrySet.offset;
        }

        // entry2Offset が不正
        NN_RESULT_THROW_UNLESS(
            currentOffset < entry2Offset, fs::ResultInvalidIndirectEntryOffset());
        // この時点で 0 <= entry1Offset <= currentOffset < entry2Offset が保証される

        const auto dataSize = entry2Offset - currentOffset;
        NN_SDK_ASSERT_LESS(0, dataSize);

        const auto remainingSize = endOffset - currentOffset; // <= param.size
        const auto readSize = static_cast<size_t>(std::min(remainingSize, dataSize));
        NN_SDK_ASSERT_LESS_EQUAL(readSize, param.size);

        if( entry1.IsFragment() )
        {
            // 断片として扱うデータが大きいか終端なら終了
            if( TEntry::FragmentSizeMax <= readSize || remainingSize <= dataSize )
            {
                break;
            }

            mergeSize += readSize;
        }
        else
        {
            // 物理アドレスが連続していないので終了
            if( physicalOffset != entry1.GetPhysicalOffset() )
            {
                break;
            }

            readableSize += mergeSize + readSize;
            NN_SDK_ASSERT_LESS_EQUAL(readableSize, static_cast<int64_t>(param.size));

            isMerged = isMerged || 0 < mergeSize;
            mergeSize = 0;
        }

        currentOffset += readSize;
        NN_SDK_ASSERT_LESS_EQUAL(currentOffset, endOffset);

        physicalOffset += entry2Offset - entry1Offset;

        entry1 = entry2;
    }

    // 断片として扱うデータがまとめて読み込みたいデータに挟まれていれば ok
    if( isMerged )
    {
        pOutInfo->SetReadSize(static_cast<size_t>(readableSize));
    }
    pOutInfo->SetSkipCount(entryIndex - param.entryIndex);

    NN_RESULT_SUCCESS;
} // NOLINT(impl/function_size)

/**
 * @brief   ストレージのデータをまとめて読み込めるか確認します。
 *
 * @param[out]  pOutInfo    まとめ読みの情報を出力するアドレス
 * @param[in]   offset      まとめ読みの開始オフセット
 * @param[in]   size        まとめ読みのサイズ
 *
 * @retval  ResultSuccess   正常に処理が終了しました。
 * @retval  上記以外        ストレージの読み込みに失敗しました。
 */
template< typename TEntry >
Result BucketTree::Visitor::ScanContinuousReading(
                                ContinuousReadingInfo* pOutInfo,
                                int64_t offset,
                                size_t size
                            ) const NN_NOEXCEPT
{
    NN_STATIC_ASSERT(std::is_pod<TEntry>::value);
    NN_SDK_REQUIRES(IsValid());

    ContinuousReadingParam<TEntry> param =
    {
        offset,
        size,
        m_EntrySet.header,
        m_EntryIndex,
    };
    std::memcpy(&param.entry, m_pEntry, sizeof(TEntry));

    return m_pTree->ScanContinuousReading<TEntry>(pOutInfo, param);
}

}}
