﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <cstring>
#include <algorithm>
#include <mutex>
#include <memory>

#include <nn/os.h>
#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/fs/fs_Result.h>

#include <nn/fs/detail/fs_Newable.h>
#include <nn/fs/fs_IStorage.h>
#include <nn/fssystem/fs_AlignmentMatchingStorageImpl.h>
#include <nn/fssystem/fs_AsynchronousAccess.h>
#include <nn/util/util_BitUtil.h>

namespace nn { namespace fssystem {

// アライメント整合のための簡易バッファレイヤ
template <int DataAlign_, int BufferAlign_>
class AlignmentMatchingStorage : public fs::IStorage, public nn::fs::detail::Newable
{
    NN_DISALLOW_COPY(AlignmentMatchingStorage);

public:
    static const int DataAlign = DataAlign_;
    static const int BufferAlign = BufferAlign_;

    static_assert(DataAlign <= 512, "DataAlign must be <= 512"); // スタックでのバッファ確保のため、サイズ制限
    static_assert((DataAlign   & (DataAlign   - 1)) == 0, "DataAlign must be power of 2."); // 2^N
    static_assert((BufferAlign & (BufferAlign - 1)) == 0, "BufferAlign must be power of 2."); // 2^N

    explicit AlignmentMatchingStorage(IStorage* pBaseStorage)
        : m_pBaseStorage(pBaseStorage),
          m_IsBaseStorageSizeDirty(true)
    {
    }

    explicit AlignmentMatchingStorage(std::shared_ptr<IStorage> pBaseStorage)
        : m_pSharedBaseStorage(pBaseStorage),
          m_pBaseStorage(pBaseStorage.get()),
          m_IsBaseStorageSizeDirty(true)
    {
    }

    virtual Result Read(int64_t offset, void* buffer, size_t size) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_ALIGNAS(512) char subBuffer[DataAlign];
        static_assert( (NN_ALIGNOF(subBuffer) & (BufferAlign - 1)) == 0, "subBuffer must be aligned with BufferAlign." );

        if( size == 0 )
        {
            NN_RESULT_SUCCESS;
        }
        if( buffer == nullptr )
        {
            return nn::fs::ResultNullptrArgument();
        }
        int64_t baseStorageSize = 0;
        NN_RESULT_DO(GetSize(&baseStorageSize));
        if( ! IStorage::CheckAccessRange(offset, size, baseStorageSize) )
        {
            return nn::fs::ResultOutOfRange();
        }

        return AlignmentMatchingStorageImpl::Read(m_pBaseStorage, subBuffer, sizeof(subBuffer), DataAlign, BufferAlign, offset, static_cast<char*>(buffer), size);
    }

    virtual Result Write(int64_t offset, const void* buffer, size_t size) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_ALIGNAS(512) char subBuffer[DataAlign];
        static_assert( (NN_ALIGNOF(subBuffer) & (BufferAlign - 1)) == 0, "subBuffer must be aligned with BufferAlign." );

        if( size == 0 )
        {
            NN_RESULT_SUCCESS;
        }
        if( buffer == nullptr )
        {
            return nn::fs::ResultNullptrArgument();
        }
        int64_t baseStorageSize = 0;
        NN_RESULT_DO(GetSize(&baseStorageSize));
        if( ! IStorage::CheckAccessRange(offset, size, baseStorageSize) )
        {
            return nn::fs::ResultOutOfRange();
        }
        return AlignmentMatchingStorageImpl::Write(m_pBaseStorage, subBuffer, sizeof(subBuffer), DataAlign, BufferAlign, offset, static_cast<const char*>(buffer), size);
    }

    virtual Result Flush() NN_NOEXCEPT NN_OVERRIDE
    {
        return m_pBaseStorage->Flush();
    }

    virtual Result SetSize(int64_t size) NN_NOEXCEPT NN_OVERRIDE
    {
        int64_t alignedSize = nn::util::align_up(size, DataAlign);

        // 切り上げたサイズを設定する
        auto result = m_pBaseStorage->SetSize(alignedSize);

        // サイズ更新フラグを立てる
        m_IsBaseStorageSizeDirty = true;

        return result;
    }

    virtual Result GetSize(int64_t* outValue) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_REQUIRES_NOT_NULL(outValue);

        if( m_IsBaseStorageSizeDirty )
        {
            // サイズ更新
            int64_t gotSize;
            NN_RESULT_DO(m_pBaseStorage->GetSize(&gotSize));

            m_IsBaseStorageSizeDirty = false;
            m_BaseStorageSize = gotSize;
        }

        *outValue = m_BaseStorageSize;
        NN_RESULT_SUCCESS;
    }

    virtual Result OperateRange(
                       void* outBuffer,
                       size_t outBufferSize,
                       fs::OperationId operationId,
                       int64_t offset,
                       int64_t size,
                       const void* inBuffer,
                       size_t inBufferSize
                   ) NN_NOEXCEPT NN_OVERRIDE
    {
        if( size == 0 )
        {
            NN_RESULT_SUCCESS;
        }
        int64_t baseStorageSize = 0;
        NN_RESULT_DO(GetSize(&baseStorageSize));
        if( !IStorage::CheckOffsetAndSize(offset, size) )
        {
            return nn::fs::ResultOutOfRange();
        }

        const auto accessibleSize = std::min(size, baseStorageSize - offset);
        const auto alignedOffset = nn::util::align_down(offset, DataAlign);
        const auto alignedOffsetEnd = nn::util::align_up(offset + accessibleSize, DataAlign);
        const auto alignedSize = alignedOffsetEnd - alignedOffset;

        return m_pBaseStorage->OperateRange(
            outBuffer,
            outBufferSize,
            operationId,
            alignedOffset,
            alignedSize,
            inBuffer,
            inBufferSize);
    }

private:
    IStorage* const m_pBaseStorage;
    int64_t m_BaseStorageSize;
    bool m_IsBaseStorageSizeDirty;

    std::shared_ptr<IStorage> m_pSharedBaseStorage;
};

// アライメント整合のための簡易バッファレイヤ、バッファプール版
template <int BufferAlign_>
class AlignmentMatchingStoragePooledBuffer : public fs::IStorage, public nn::fs::detail::Newable
{
    NN_DISALLOW_COPY(AlignmentMatchingStoragePooledBuffer);

public:
    static const int BufferAlign = BufferAlign_;

    static_assert((BufferAlign & (BufferAlign - 1)) == 0, "BufferAlign must be power of 2."); // 2^N

    explicit AlignmentMatchingStoragePooledBuffer(IStorage* pBaseStorage, int dataAlign)
        : m_pBaseStorage(pBaseStorage),
          m_DataAlign(dataAlign),
          m_IsBaseStorageSizeDirty(true)
    {
        NN_SDK_REQUIRES((dataAlign & (dataAlign - 1)) == 0, "DataAlign must be power of 2.");
    }

    virtual Result Read(int64_t offset, void* buffer, size_t size) NN_NOEXCEPT NN_OVERRIDE
    {
        if( size == 0 )
        {
            NN_RESULT_SUCCESS;
        }
        if( buffer == nullptr )
        {
            return nn::fs::ResultNullptrArgument();
        }
        int64_t baseStorageSize = 0;
        NN_RESULT_DO(GetSize(&baseStorageSize));
        if( ! IStorage::CheckAccessRange(offset, size, baseStorageSize) )
        {
            return nn::fs::ResultOutOfRange();
        }

        PooledBuffer pooledBuffer;
        pooledBuffer.AllocateParticularlyLarge(m_DataAlign, m_DataAlign);
        return AlignmentMatchingStorageImpl::Read(m_pBaseStorage, pooledBuffer.GetBuffer(), pooledBuffer.GetSize(), m_DataAlign, BufferAlign, offset, static_cast<char*>(buffer), size);
    }

    virtual Result Write(int64_t offset, const void* buffer, size_t size) NN_NOEXCEPT NN_OVERRIDE
    {
        if( size == 0 )
        {
            NN_RESULT_SUCCESS;
        }
        if( buffer == nullptr )
        {
            return nn::fs::ResultNullptrArgument();
        }
        int64_t baseStorageSize = 0;
        NN_RESULT_DO(GetSize(&baseStorageSize));
        if( ! IStorage::CheckAccessRange(offset, size, baseStorageSize) )
        {
            return nn::fs::ResultOutOfRange();
        }

        PooledBuffer pooledBuffer;
        pooledBuffer.AllocateParticularlyLarge(m_DataAlign, m_DataAlign);
        return AlignmentMatchingStorageImpl::Write(m_pBaseStorage, pooledBuffer.GetBuffer(), pooledBuffer.GetSize(), m_DataAlign, BufferAlign, offset, static_cast<const char*>(buffer), size);
    }

    virtual Result Flush() NN_NOEXCEPT NN_OVERRIDE
    {
        return m_pBaseStorage->Flush();
    }

    virtual Result SetSize(int64_t size) NN_NOEXCEPT NN_OVERRIDE
    {
        int64_t alignedSize = nn::util::align_up(size, m_DataAlign);

        // 切り上げたサイズを設定する
        auto result = m_pBaseStorage->SetSize(alignedSize);

        // サイズ更新フラグを立てる
        m_IsBaseStorageSizeDirty = true;

        return result;
    }

    virtual Result GetSize(int64_t* outValue) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_REQUIRES_NOT_NULL(outValue);

        if( m_IsBaseStorageSizeDirty )
        {
            // サイズ更新
            int64_t gotSize;
            NN_RESULT_DO(m_pBaseStorage->GetSize(&gotSize));

            m_IsBaseStorageSizeDirty = false;
            m_BaseStorageSize = gotSize;
        }

        *outValue = m_BaseStorageSize;
        NN_RESULT_SUCCESS;
    }

    virtual Result OperateRange(
                       void* outBuffer,
                       size_t outBufferSize,
                       fs::OperationId operationId,
                       int64_t offset,
                       int64_t size,
                       const void* inBuffer,
                       size_t inBufferSize
                   ) NN_NOEXCEPT NN_OVERRIDE
    {
        if( size == 0 )
        {
            NN_RESULT_SUCCESS;
        }
        int64_t baseStorageSize = 0;
        NN_RESULT_DO(GetSize(&baseStorageSize));
        if( !IStorage::CheckOffsetAndSize(offset, size) )
        {
            return nn::fs::ResultOutOfRange();
        }

        const auto accessibleSize = std::min(size, baseStorageSize - offset);
        const auto alignedOffset = nn::util::align_down(offset, m_DataAlign);
        const auto alignedOffsetEnd = nn::util::align_up(offset + accessibleSize, m_DataAlign);
        const auto alignedSize = alignedOffsetEnd - alignedOffset;

        return m_pBaseStorage->OperateRange(
            outBuffer,
            outBufferSize,
            operationId,
            alignedOffset,
            alignedSize,
            inBuffer,
            inBufferSize);
    }

private:
    IStorage* const m_pBaseStorage;
    int64_t m_BaseStorageSize;
    int m_DataAlign;
    bool m_IsBaseStorageSizeDirty;
};

// アライメント整合のための簡易バッファレイヤ、一括読み込み版
template <int BufferAlignment>
class AlignmentMatchingStorageInBulkRead : public fs::IStorage, public fs::detail::Newable
{
    NN_DISALLOW_COPY(AlignmentMatchingStorageInBulkRead);

    NN_STATIC_ASSERT((BufferAlignment & (BufferAlignment - 1)) == 0);

public:
    AlignmentMatchingStorageInBulkRead(fs::IStorage* pBaseStorage, int dataAlignment) NN_NOEXCEPT
        : m_pBaseStorage(pBaseStorage),
          m_pBaseStorageShared(),
          m_BaseStorageSize(-1),
          m_DataAlignment(dataAlignment)
    {
        NN_SDK_REQUIRES(
            (dataAlignment & (dataAlignment - 1)) == 0,
            "DataAlignment must be power of 2.");
    }

    AlignmentMatchingStorageInBulkRead(
        std::shared_ptr<fs::IStorage> pBaseStorage,
        int dataAlignment) NN_NOEXCEPT
        : m_pBaseStorage(pBaseStorage.get()),
          m_pBaseStorageShared(pBaseStorage),
          m_BaseStorageSize(-1),
          m_DataAlignment(dataAlignment)
    {
        NN_SDK_REQUIRES(
            (dataAlignment & (dataAlignment - 1)) == 0,
            "DataAlignment must be power of 2.");
    }

    virtual Result Read(int64_t offset, void* buffer, size_t size) NN_NOEXCEPT NN_OVERRIDE;

    virtual Result Write(int64_t offset, const void* buffer, size_t size) NN_NOEXCEPT NN_OVERRIDE
    {
        if( size == 0 )
        {
            NN_RESULT_SUCCESS;
        }
        if( buffer == nullptr )
        {
            return fs::ResultNullptrArgument();
        }
        int64_t baseStorageSize = 0;
        NN_RESULT_DO(GetSize(&baseStorageSize));
        NN_RESULT_THROW_UNLESS(fs::IStorage::CheckAccessRange(offset, size, baseStorageSize), fs::ResultOutOfRange());

        PooledBuffer pooledBuffer(m_DataAlignment, m_DataAlignment);
        return AlignmentMatchingStorageImpl::Write(
            m_pBaseStorage,
            pooledBuffer.GetBuffer(),
            pooledBuffer.GetSize(),
            m_DataAlignment,
            BufferAlignment,
            offset,
            static_cast<const char*>(buffer),
            size);
    }

    virtual Result Flush() NN_NOEXCEPT NN_OVERRIDE
    {
        return m_pBaseStorage->Flush();
    }

    virtual Result SetSize(int64_t size) NN_NOEXCEPT NN_OVERRIDE
    {
        int64_t alignedSize = nn::util::align_up(size, m_DataAlignment);

        auto result = m_pBaseStorage->SetSize(alignedSize);

        m_BaseStorageSize = -1;

        return result;
    }

    virtual Result GetSize(int64_t* outValue) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_REQUIRES_NOT_NULL(outValue);

        if( m_BaseStorageSize < 0 )
        {
            int64_t baseStorageSize = 0;
            NN_RESULT_DO(m_pBaseStorage->GetSize(&baseStorageSize));
            m_BaseStorageSize = baseStorageSize;
        }

        *outValue = m_BaseStorageSize;
        NN_RESULT_SUCCESS;
    }

    virtual Result OperateRange(
                       void* outBuffer,
                       size_t outBufferSize,
                       fs::OperationId operationId,
                       int64_t offset,
                       int64_t size,
                       const void* inBuffer,
                       size_t inBufferSize
                   ) NN_NOEXCEPT NN_OVERRIDE
    {
        if( size == 0 )
        {
            NN_RESULT_SUCCESS;
        }
        int64_t baseStorageSize = 0;
        NN_RESULT_DO(GetSize(&baseStorageSize));
        if( !IStorage::CheckOffsetAndSize(offset, size) )
        {
            return nn::fs::ResultOutOfRange();
        }

        const auto accessibleSize = std::min(size, baseStorageSize - offset);
        const auto alignedOffset = nn::util::align_down(offset, m_DataAlignment);
        const auto alignedOffsetEnd = nn::util::align_up(offset + accessibleSize, m_DataAlignment);
        const auto alignedSize = alignedOffsetEnd - alignedOffset;

        return m_pBaseStorage->OperateRange(
            outBuffer,
            outBufferSize,
            operationId,
            alignedOffset,
            alignedSize,
            inBuffer,
            inBufferSize);
    }

private:
    fs::IStorage* const m_pBaseStorage;
    std::shared_ptr<fs::IStorage> m_pBaseStorageShared;
    int64_t m_BaseStorageSize;
    int m_DataAlignment;
};

}}
