﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <type_traits>

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/util/util_BinTypes.h>

#include <nn/fs/detail/fs_Newable.h>
#include <nn/fs/fsa/fs_IFile.h>
#include <nn/fs/fsa/fs_IFileSystem.h>
#include <nn/fssystem/fs_AesXtsStorage.h>

#include <nn/crypto/crypto_AesEncryptor.h>

namespace nn { namespace fs {
class IStorage;
}}

namespace nn { namespace fssystem {
class StorageFile;
class AesXtsFile;
}}

namespace nnt { namespace fs { namespace detail {
nn::fs::fsa::IFile* GetBaseFile(nn::fssystem::AesXtsFile* pAesXtsFile) NN_NOEXCEPT;
}}}

namespace nn { namespace fssystem {

struct AesXtsFileHeader
{
    static const size_t Size = 16 * 1024;
    static const size_t HeaderSignSize = 32;
    static const size_t KeySize = AesXtsStorage::KeySize;
    static const size_t Reserved2Size = 48;
    static const size_t EncryptionKeyGenerationKeySize = 16;
    static const size_t MacKeySize = 16;
    static const size_t CryptoKeySize = 16;
    static const int CryptoKeyCount = 2;

    enum class Signature : uint32_t
    {
        V0 = NN_UTIL_CREATE_SIGNATURE_4('N', 'A', 'X', '0'),
    };

    typedef void (*GenerateRandomFunction)(void* pOutData, size_t size);

    uint8_t headerSign[HeaderSignSize];
    uint32_t signature;
    uint32_t reserved1;
    uint8_t key1[KeySize];
    uint8_t key2[KeySize];
    int64_t actualFileSize;
    uint8_t reserved2[Reserved2Size];

public:
    void Create(
        const char* path,
        int64_t fileSize,
        const void* pEncryptionKeyGenerationKey, size_t encryptionKeyGenerationKeySize,
        const void* pMacKey, size_t macKeySize,
        GenerateRandomFunction pGenerateRandom
    ) NN_NOEXCEPT;

    void SignAndEncryptKeys(
        const char* path,
        const void* pEncryptionKeyGenerationKey, size_t encryptionKeyGenerationKeySize,
        const void* pMacKey, size_t macKeySize
    ) NN_NOEXCEPT;

    bool DecryptKeysAndVerify(
        void* pOutKeys, size_t keysSize,
        const char* path,
        const void* pEncryptionKeyGenerationKey, size_t encryptionKeyGenerationKeySize,
        const void* pMacKey, size_t macKeySize
    ) NN_NOEXCEPT;

    bool Update(int64_t fileSize, const void* pKeys, size_t keysSize, const void* pMacKey, size_t macKeySize) NN_NOEXCEPT;

public:
    bool HasValidSignature() const NN_NOEXCEPT
    {
        return AesXtsFileHeader::IsValidSignature(this->signature);
    }

    static bool IsValidSignature(uint32_t signature) NN_NOEXCEPT
    {
        return static_cast<Signature>(signature) == Signature::V0;
    }

private:
    void ComputeMac(void* pOutMac, size_t macSize, const void* pMacKey, size_t macKeySize) const NN_NOEXCEPT;

    void ComputeKeys(void* pOutKeys, size_t keysSize, const char* path, const void* pEncryptionKeyGenerationKey, size_t encryptionKeyGenerationKeySize) const NN_NOEXCEPT;

    void Encrypt(const void* pKeys, size_t keysSize) NN_NOEXCEPT;

    void Decrypt(const void* pKeys, size_t keysSize) NN_NOEXCEPT;
};

NN_STATIC_ASSERT(sizeof(AesXtsFileHeader) == 128);
NN_STATIC_ASSERT(std::is_pod<AesXtsFileHeader>::value);

class AesXtsFile : public fs::fsa::IFile, public nn::fs::detail::Newable
{
    NN_DISALLOW_COPY(AesXtsFile);

    friend nn::fs::fsa::IFile* nnt::fs::detail::GetBaseFile(nn::fssystem::AesXtsFile* pAesXtsFile) NN_NOEXCEPT;

public:
    static const size_t AesBlockSize = crypto::AesEncryptor128::BlockSize;
    static const size_t KeySize = AesBlockSize;
    static const size_t IvSize  = AesBlockSize;

public:
    AesXtsFile() NN_NOEXCEPT;
    virtual ~AesXtsFile() NN_NOEXCEPT NN_OVERRIDE;

    Result Initialize(
        fs::OpenMode mode,
        std::unique_ptr<fs::fsa::IFile>&& pBaseFile,
        const char* path,
        const void* pEncryptionKeyGenerationKey, size_t encryptionKeyGenerationKeySize,
        const void* pMacKey, size_t macKeySize,
        const void* pIv, size_t ivSize,
        size_t blockSize
    ) NN_NOEXCEPT;

private:
    virtual Result DoRead(size_t* outValue, int64_t offset, void* buffer, size_t size, const fs::ReadOption& option) NN_NOEXCEPT NN_OVERRIDE;
    virtual Result DoWrite(int64_t offset, const void* buffer, size_t size, const fs::WriteOption& option) NN_NOEXCEPT NN_OVERRIDE;
    virtual Result DoFlush() NN_NOEXCEPT NN_OVERRIDE;
    virtual Result DoSetSize(int64_t size) NN_NOEXCEPT NN_OVERRIDE;
    virtual Result DoGetSize(int64_t* outValue) NN_NOEXCEPT NN_OVERRIDE;
    virtual Result DoOperateRange(
        void* outBuffer,
        size_t outBufferSize,
        fs::OperationId operationId,
        int64_t offset,
        int64_t size,
        const void* inBuffer,
        size_t inBufferSize) NN_NOEXCEPT NN_OVERRIDE;

private:
    std::unique_ptr<fs::fsa::IFile> m_pBaseFile;
    std::unique_ptr<fs::IStorage> m_pFileStorage;
    std::unique_ptr<fs::IStorage> m_pFileSubStorage;
    std::unique_ptr<fs::IStorage> m_pAesXtsStorage;
    std::unique_ptr<fs::IStorage> m_pAlignmentMatchingStorage;
    std::unique_ptr<fs::fsa::IFile> m_pStorageFile;
    int64_t m_FileSize;
    uint8_t m_MacKey[AesXtsFileHeader::MacKeySize];
    uint8_t m_HeaderKeyEncryptionKeys[AesXtsFileHeader::CryptoKeySize * AesXtsFileHeader::CryptoKeyCount];
    fs::OpenMode m_Mode;
};

}}
