﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <cstring>

#include <nn/fssystem/dbm/fs_DbmParameters.h>
#include <nn/fssystem/dbm/fs_AllocationTable.h>
#include <nn/fssystem/dbm/fs_AllocationTableStorage.h>
#include <nn/fssystem/dbm/fs_HierarchicalFileTableTemplate.h>
#include <nn/fssystem/dbm/fs_FileSystemControlArea.h>
#include <nn/fssystem/dbm/fs_DirectoryObjectTemplate.h>
#include <nn/fssystem/dbm/fs_FileObjectTemplate.h>

class FileSystemObjectTest;

namespace nn { namespace fssystem { namespace dbm {

template <typename TDirectoryName, typename TFileName>
class FileObjectTemplate;

template <typename TDirectoryName, typename TFileName>
class DirectoryObjectTemplate;

/**
* @brief    階層構造のあるファイルシステム。
*
* @tparam   DirectoryNameClass
* @tparam   FileNameClass
*
* @details
*   階層構造のあるファイルシステム。
*/
template <typename TDirectoryName, typename TFileName>
class FileSystemObjectTemplate
    : private HierarchicalFileTableTemplate<
                 TDirectoryName,
                 TFileName,
                 detail::DirectoryInfo,
                 detail::FileInfo
             >
{
    NN_DISALLOW_COPY(FileSystemObjectTemplate);

private:
    typedef HierarchicalFileTableTemplate<
                 TDirectoryName,
                 TFileName,
                 detail::DirectoryInfo,
                 detail::FileInfo
             >
             BaseClass;

public:
    typedef typename BaseClass::StorageIndex StorageIndex;
    typedef typename BaseClass::DirectoryElement DirectoryElement;
    typedef typename BaseClass::FileElement FileElement;
    typedef FileObjectTemplate<TDirectoryName, TFileName> FileObjectTempl;
    typedef DirectoryObjectTemplate<TDirectoryName, TFileName> DirectoryObjectTempl;

public:
    /**
    * @brief        ファイルシステムとしてフォーマットします。
    *
    * @param[in]    pDirectoryEntryStorage      ディレクトリエントリー用ストレージ
    * @param[in]    pFileEntryStorage           ファイルエントリー用ストレージ
    *
    * @details      ファイルシステムとしてフォーマットします。
    * @see          nn::fssystem::dbm::HierarchicalFileTable::Format
    */
    using BaseClass::Format;

    /**
    * @brief        エントリー数から必要なストレージのサイズを求めます。
    *
    * @details      引数、返り値の詳細については以下の関数を参照。
    *
    * @details      ディレクトリエントリーストレージ、ファイルエントリーストレージは、
    *               可変長データベースで管理されているストレージに作成。
    * @see          nn::fssystem::dbm::HierarchicalFileTable::QueryDirectoryEntryStorageSize
    * @see          nn::fssystem::dbm::HierarchicalFileTable::QueryFileEntryStorageSize
    */
    using BaseClass::QueryDirectoryEntryStorageSize;
    using BaseClass::QueryFileEntryStorageSize;

    /**
    * @brief        アロケーションテーブルに必要なサイズを取得します。
    *
    * @param[in]    blockCount  ブロック数
    *
    * @return       アロケーションテーブルに必要なサイズ。
    *
    * @details      アロケーションテーブルに必要なサイズを取得します。
    */
    static inline int64_t QueryAllocationTableStorageSize(uint32_t blockCount) NN_NOEXCEPT
    {
        return AllocationTable::QuerySize(blockCount);
    }

    /**
    * @brief        コンストラクタです。
    *
    * @details      コンストラクタです。
    */
    FileSystemObjectTemplate() NN_NOEXCEPT;

    /**
    * @brief        ファイルシステムをマウントします。
    *
    * @param[in]    pAllocTable             アロケーションテーブル
    * @param[in]    blockSize               ブロックサイズ
    * @param[in]    pDirectoryEntryStorage  ディレクトリエントリー用ストレージ
    * @param[in]    pFileEntryStorage       ファイルエントリー用ストレージ
    *
    * @pre          pAllocTable != nullptr
    * @pre          pDirectoryEntryStorage != nullptr
    * @pre          pFileEntryStorage != nullptr
    *
    * @details      ファイルシステムをマウントします。
    */
    void Initialize(
             AllocationTable* pAllocTable,
             uint32_t blockSize,
             BufferedAllocationTableStorage* pDirectoryEntryStorage,
             BufferedAllocationTableStorage* pFileEntryStorage
         ) NN_NOEXCEPT;

    /**
    * @brief        ファイルシステムをアンマウントします。
    *
    * @details      ファイルシステムをアンマウントします。
    */
    void Finalize() NN_NOEXCEPT;

    /**
    * @brief        ディレクトリを作成します。
    *
    * @param[in]    pFullPath                       フルパス
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultSuccess                   正常に作成できました。
    * @retval       ResultNotInitialized            初期化されていません。
    * @retval       ResultInvalidPathFormat         pFullPath がルートディレクトリから始まっていません。
    * @retval       ResultTooLongPath               ディレクトリ名が長すぎます。
    * @retval       ResultDirectoryUnobtainable     ルートディレクトリの親ディレクトリは取得できません。
    * @retval       ResultIncompatiblePath          pFullPath がファイルでした。
    * @retval       ResultAlreadyExists             同名のファイルまたはディレクトリが存在します。
    * @retval       ResultAllocationTableFull       アロケーションテーブルに空きがありません。
    * @retval       ResultInvalidOffset             内部データに問題があります。
    * @retval       ResultDatabaseCorrupted         内部データに問題があります。
    * @retval       上記以外                        ストレージでの読み書きに失敗しました。
    *
    * @pre          pFullPath != nullptr
    *
    * @details      ディレクトリを作成します。
    */
    Result CreateDirectory(const PathChar* pFullPath) NN_NOEXCEPT;

    /**
    * @brief        ディレクトリを削除します。
    *
    * @details      ディレクトリを削除します。
    *               引数、返り値の詳細については以下を参照。
    * @see          nn::dbm::HierarchicalFileTableTemplate::DeleteDirectory
    */
    using BaseClass::DeleteDirectory;

    /**
    * @brief        ディレクトリの名前を変更します。
    *
    * @details      ディレクトリの名前を変更します。
    *               引数、返り値の詳細については以下を参照。
    * @see          nn::dbm::HierarchicalFileTableTemplate::RenameDirectory
    */
    using BaseClass::RenameDirectory;

    /**
    * @brief        ディレクトリを開きます。
    *
    * @param[out]   outIndex                        ディレクトリ ID
    * @param[out]   outDirectory                    ディレクトリオブジェクト
    * @param[in]    pFullPath                       フルパス
    *
    * @retval       ResultSuccess                   正常に開きました。
    * @retval       ResultNotInitialized            初期化されていません。
    * @retval       ResultInvalidPathFormat         pFullPath がルートディレクトリから始まっていません。
    * @retval       ResultTooLongPath               ディレクトリ名が長すぎます。
    * @retval       ResultDirectoryUnobtainable     ルートディレクトリの親ディレクトリは取得できません。
    * @retval       ResultIncompatiblePath          pFullPath がファイルでした。
    * @retval       ResultIncompatiblePath          内部データに問題があります。
    * @retval       ResultDirectoryNotFound         内部データに問題があります。
    * @retval       ResultInvalidOffset             内部データに問題があります。
    * @retval       ResultDatabaseCorrupted         内部データに問題があります。
    * @retval       上記以外                        ストレージからのデータ読み込みに失敗しました。
    *
    * @pre          outIndex != nullptr
    * @pre          outDirectory != nullptr
    * @pre          pFullPath != nullptr
    *
    * @details      ディレクトリを開きます。
    */
    Result OpenDirectory(
            StorageIndex* outIndex,
            DirectoryObjectTempl* outDirectory,
            const PathChar* pFullPath
        ) NN_NOEXCEPT;

    Result OpenDirectory(
            DirectoryObjectTempl* outDirectory,
            const PathChar* pFullPath
        ) NN_NOEXCEPT
    {
        StorageIndex tmp;
        return OpenDirectory(&tmp, outDirectory, pFullPath);
    }

    /**
    * @brief        ファイルを作成します。
    *
    * @param[out]   outIndex                        ファイル ID
    * @param[in]    pFullPath                       フルパス
    * @param[in]    fileOptionalInfo                ファイル操作オプション
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultSuccess                   正常に作成できました。
    * @retval       ResultNotInitialized            初期化されていません。
    * @retval       ResultAlreadyExists             同名のファイルまたはディレクトリが存在します。
    * @retval       ResultInvalidPathFormat         pFullPath がルートディレクトリから始まっていません。
    * @retval       ResultTooLongPath               ディレクトリ名が長すぎます。
    * @retval       ResultDirectoryUnobtainable     ルートディレクトリの親ディレクトリは取得できません。
    * @retval       ResultIncompatiblePath          pFullPath がディレクトリでした。
    * @retval       ResultAllocationTableFull       アロケーションテーブルに空きがありません。
    * @retval       ResultInvalidOffset             内部データに問題があります。
    * @retval       ResultDatabaseCorrupted         内部データに問題があります。
    * @retval       上記以外                        ストレージでの読み書きに失敗しました。
    *
    * @pre          outIndex != nullptr
    * @pre          pFullPath != nullptr
    *
    * @details      ファイルを作成します。
    */
    Result CreateFile(
               StorageIndex* outIndex,
               const PathChar* pFullPath,
               const FileOptionalInfo& fileOptionalInfo
           ) NN_NOEXCEPT;

    Result CreateFile(
               const PathChar* pFullPath,
               const FileOptionalInfo& fileOptionalInfo
           )
    {
        StorageIndex tmp;
        return CreateFile(&tmp, pFullPath, fileOptionalInfo);
    }

    /**
    * @brief        指定したファイルを削除します。
    *
    * @param[in]    pFullpath   ファイルパス
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultSuccess                   正常に削除できました。
    * @retval       ResultNotInitialized            初期化されていません。
    * @retval       ResultIncompatiblePath          pFullPath がディレクトリでした。
    * @retval       ResultTooLongPath               ディレクトリ名が長すぎます。
    * @retval       ResultDirectoryUnobtainable     ルートディレクトリの親ディレクトリは取得できません。
    * @retval       ResultInvalidPathFormat         pFullPath がルートディレクトリから始まっていません。
    * @retval       ResultIncompatiblePath          内部データに問題があります。
    * @retval       ResultFileNotFound              内部データに問題があります。
    * @retval       ResultDatabaseKeyNotFound       内部データに問題があります。
    * @retval       ResultInvalidOffset             内部データに問題があります。
    * @retval       ResultDatabaseCorrupted         内部データに問題があります。
    * @retval       上記以外                        ストレージでの読み書きに失敗しました。
    *
    * @pre          マウントしている。
    * @pre          pFullPath != nullptr
    *
    * @details      指定したファイルを削除します。
    */
    Result DeleteFile(const PathChar* pFullPath) NN_NOEXCEPT;

    /**
    * @brief        ファイルの名前を変更します。
    *
    * @details      指定したファイルを削除します。
    * @see          nn::dbm::HierarchicalFileTableTemplate::RenameFile を参照。
    *
    */
    using BaseClass::RenameFile;

    /**
    * @brief        ファイルを開きます。
    *
    * @param[out]   outIndex        ファイル ID
    * @param[out]   outFile         ファイルオブジェクト
    * @param[in]    pFullPath       フルパス
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultSuccess                   正常に開けました。
    * @retval       ResultNotInitialized            初期化されていません。
    * @retval       ResultInvalidPathFormat         pFullPath がルートディレクトリから始まっていません。
    * @retval       ResultTooLongPath               ディレクトリ名が長すぎます。
    * @retval       ResultDirectoryUnobtainable     ルートディレクトリの親ディレクトリは取得できません。
    * @retval       ResultIncompatiblePath          pFullPath がディレクトリでした。
    * @retval       ResultIncompatiblePath          内部データに問題があります。
    * @retval       ResultFileNotFound              内部データに問題があります。
    * @retval       ResultInvalidOffset             内部データに問題があります。
    * @retval       ResultDatabaseCorrupted         内部データに問題があります。
    * @retval       上記以外                        ストレージからのデータ読み込みに失敗しました。
    *
    * @pre          outIndex != nullptr
    * @pre          pFullPath != nullptr
    *
    * @details      ファイルを開きます。
    */
    Result OpenFile(
            StorageIndex* outIndex,
            FileObjectTempl* outFile,
            const PathChar* pFullPath
        ) NN_NOEXCEPT;

    Result OpenFile(
               FileObjectTempl* outFile,
               const PathChar* pFullPath
           ) NN_NOEXCEPT
    {
        StorageIndex tmp;
        return OpenFile(&tmp, outFile, pFullPath);
    }

    /**
    * @brief        アロケーションテーブルの空きブロック数を取得します。
    *
    * @param[out]   outCount       アロケーションテーブルの空きブロック数
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultSuccess   正常に取得できました。
    * @retval       上記以外        ストレージからの読み込みに失敗しました。
    *
    * @pre          マウントしている。
    * @pre          outCount != nullptr
    *
    * @details      アロケーションテーブルの空きブロック数を取得します。
    */
    inline Result CalcFreeListLength(uint32_t* outCount) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(m_pAllocationTable);
        NN_SDK_REQUIRES_NOT_NULL(outCount);
        return m_pAllocationTable->CalcFreeListLength(outCount);
    }

    /**
    * @brief        指定したファイルのアロケーションテーブルのインデックスを取得します。
    *
    * @param[out]   outIndex                        アロケーションテーブルのインデックス
    * @param[in]    pFullPath                       フルパス
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultSuccess                   正常に取得できました。
    * @retval       ResultNotInitialized            初期化されていません。
    * @retval       ResultInvalidPathFormat         pFullPath がルートディレクトリから始まっていません。
    * @retval       ResultTooLongPath               ディレクトリ名が長すぎます。
    * @retval       ResultDirectoryUnobtainable     ルートディレクトリの親ディレクトリは取得できません。
    * @retval       ResultIncompatiblePath          pFullPath がディレクトリでした。
    * @retval       ResultIncompatiblePath          内部データに問題があります。
    * @retval       ResultFileNotFound              内部データに問題があります。
    * @retval       ResultInvalidOffset             内部データに問題があります。
    * @retval       ResultDatabaseCorrupted         内部データに問題があります。
    * @retval       上記以外                        ストレージからのデータ読み込みに失敗しました。
    *
    * @pre          outIndex != nullptr
    * @pre          pFullPath != nullptr
    *
    * @details      指定したファイルのアロケーションテーブルのインデックスを取得します。
    *               ファイルサイズを利用しない特殊なファイルシステム用の機能です。
    */
    Result GetAllocationTableIndex(uint32_t* outIndex, const char* pFullPath) NN_NOEXCEPT;

    /**
    * @brief        指定したファイルのアロケーションテーブルのインデックスを変更します。
    *
    * @param[in]    index                           アロケーションテーブルのインデックス
    * @param[in]    pFullPath                       フルパス
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultSuccess                   正常に変更できました。
    * @retval       ResultNotInitialized            初期化されていません。
    * @retval       ResultInvalidPathFormat         pFullPath がルートディレクトリから始まっていません。
    * @retval       ResultTooLongPath               ディレクトリ名が長すぎます。
    * @retval       ResultDirectoryUnobtainable     ルートディレクトリの親ディレクトリは取得できません。
    * @retval       ResultIncompatiblePath          pFullPath がディレクトリでした。
    * @retval       ResultIncompatiblePath          内部データに問題があります。
    * @retval       ResultFileNotFound              内部データに問題があります。
    * @retval       ResultInvalidOffset             内部データに問題があります。
    * @retval       ResultDatabaseCorrupted         内部データに問題があります。
    * @retval       上記以外                        ストレージでのデータ読み書きに失敗しました。
    *
    * @pre          pFullPath != nullptr
    *
    * @details      ファイルサイズを利用しない特殊なファイルシステム用の機能です。
    */
    Result SetAllocationTableIndex(uint32_t index, const char* pFullPath) NN_NOEXCEPT;

    /**
    * @brief        指定したファイルの識別 ID を取得します。
    *
    * @param[out]   outId                           ファイル ID
    * @param[in]    pFullPath                       フルパス
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultSuccess                   正常に取得できました。
    * @retval       ResultNotInitialized            初期化されていません。
    * @retval       ResultInvalidPathFormat         pFullPath がルートディレクトリから始まっていません。
    * @retval       ResultTooLongPath               ディレクトリ名が長すぎます。
    * @retval       ResultDirectoryUnobtainable     ルートディレクトリの親ディレクトリは取得できません。
    * @retval       ResultIncompatiblePath          pFullPath がディレクトリでした。
    * @retval       ResultIncompatiblePath          内部データに問題があります。
    * @retval       ResultFileNotFound              内部データに問題があります。
    * @retval       ResultInvalidOffset             内部データに問題があります。
    * @retval       ResultDatabaseCorrupted         内部データに問題があります。
    * @retval       上記以外                        ストレージからのデータ読み込みに失敗しました。
    *
    * @pre          outId != nullptr
    * @pre          pFullPath != nullptr
    *
    * @details      ファイルサイズを利用しない特殊なファイルシステム用の機能です。
    */
    Result GetSessionId(int64_t* outId, const char* pFullPath) NN_NOEXCEPT;

    /**
    * @brief        指定したファイルに識別 ID を設定します。
    *
    * @param[in]    id                              ファイル ID
    * @param[in]    pFullPath                       フルパス
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultSuccess                   正常に設定できました。
    * @retval       ResultNotInitialized            初期化されていません。
    * @retval       ResultInvalidPathFormat         pFullPath がルートディレクトリから始まっていません。
    * @retval       ResultTooLongPath               ディレクトリ名が長すぎます。
    * @retval       ResultDirectoryUnobtainable     ルートディレクトリの親ディレクトリは取得できません。
    * @retval       ResultIncompatiblePath          pFullPath がディレクトリでした。
    * @retval       ResultIncompatiblePath          内部データに問題があります。
    * @retval       ResultFileNotFound              内部データに問題があります。
    * @retval       ResultInvalidOffset             内部データに問題があります。
    * @retval       ResultDatabaseCorrupted         内部データに問題があります。
    * @retval       上記以外                        ストレージでのデータ読み書きに失敗しました。
    *
    * @pre          pFullPath != nullptr
    *
    * @details      ファイルサイズを利用しない特殊なファイルシステム用の機能です。
    */
    Result SetSessionId(int64_t id, const char* pFullPath) NN_NOEXCEPT;

    using BaseClass::CheckSubEntry;

private:
    /**
    * @brief        指定されたサイズを抱合できるブロック数を取得します。
    *
    * @param[in]    size        サイズ
    *
    * @return       指定されたサイズを抱合できるブロック数。
    *
    * @pre          size >= 0
    *
    * @details      指定されたサイズを抱合できるブロック数を取得します。
    */
    uint32_t RoundUpBlockSize(int64_t size) NN_NOEXCEPT;

    /**
    * @brief        ファイルサイズを変更します。
    *
    * @param[in]    pFileInfo                       ファイル情報
    * @param[in]    fileKey                         ファイルキー
    * @param[in]    newSize                         変更後のファイルサイズ
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultSuccess                   ファイルサイズが変更できました。
    * @retval       ResultNotInitialized            初期化されていません。
    * @retval       ResultIncompatiblePath          fileKey はファイルではなくディレクトリです。
    * @retval       ResultFileNotFound              fileKey に紐づくファイルエントリーがありません。
    * @retval       ResultOutOfRange                newSize が範囲外です。
    * @retval       ResultInvalidOffset             内部データに問題があります。
    * @retval       ResultDatabaseCorrupted         内部データに問題があります。
    * @retval       ResultAllocationTableFull       アロケーションテーブルに空きが無い。
    * @retval       上記以外                        ストレージでの読み書きに失敗しました。
    *
    * @pre          マウントしている。
    * @pre          pFileInfo != nullptr
    * @pre          0 <= newSize < 4GB * m_BlockSize
    *
    * @details      ファイルサイズを変更します。
    */
    Result ResizeFile(
               detail::FileInfo* pFileInfo,
               const typename BaseClass::FileKey& fileKey,
               int64_t newSize
           ) NN_NOEXCEPT;

    /**
    * @brief 指定したデータ領域の次のデータ領域を取得します。
    *
    * @param[out]   outNextIndex        次の連続ブロックチェインの先頭インデックス。
    *                                   最後の連続ブロックチェインであれば INDEX_EMPTY。
    * @param[out]   outBlockCount       現在の連続ブロックチェインのデータ領域数
    * @param[in]    index               現在の連続ブロックチェインの先頭インデックス
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultSuccess       正常に取得できました。
    * @retval       上記以外            ストレージからのデータ読み込みに失敗しました。
    *
    * @pre          マウントしている。
    * @pre          outNextIndex != nullptr
    * @pre          outBlockCount != nullptr
    *
    * @details      指定したデータ領域の次のデータ領域を取得します。
    *               連続ブロックチェイン単位でイテレーションします。
    */
    Result GetNextAllocationBlock(
               uint32_t* outNextIndex,
               uint32_t* outBlockCount,
               uint32_t index
           ) NN_NOEXCEPT;

    /**
    * @brief        指定したデータ領域の手前のデータ領域を取得します。
    *
    * @param[out]   outPreviousIndex    手前の連続ブロックチェインの先頭インデックス。
    *                                   最初の連続ブロックチェインであれば INDEX_EMPTY。
    * @param[out]   outBlockCount       現在の連続ブロックチェインのデータ領域数
    * @param[in]    index               現在の連続ブロックチェインの先頭インデックス
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultSuccess       正常に取得できました。
    * @retval       上記以外            ストレージからのデータ読み込みに失敗しました。
    *
    * @pre          マウントしている。
    * @pre          outPreviousIndex != nullptr
    * @pre          outBlockCount != nullptr
    *
    * @details      指定したデータ領域の手前のデータ領域を取得します。
    *               連続ブロックチェイン単位でイテレーションします。
    */
    Result GetPrevAllocationBlock(
               uint32_t* outPreviousIndex,
               uint32_t* outBlockCount,
               uint32_t index
           ) NN_NOEXCEPT;

private:
    //! HierarchicalFileTable の以下のメソッドを private 化します。
    using BaseClass::CreateDirectory;
    using BaseClass::CreateFile;
    using BaseClass::Initialize;
    using BaseClass::Finalize;
    using BaseClass::FindOpen;
    using BaseClass::FindNextDirectory;
    using BaseClass::FindNextFile;
    using BaseClass::Notify;

private:
    AllocationTable* m_pAllocationTable;    //! アロケーションテーブル
    uint32_t m_BlockSize;                   //! ブロックサイズ
    uint32_t m_BlockSizeShift;              //! ブロック数を求めるためのシフト数

private:
    // 以下のクラスに private な関数を参照させます。
    friend class FileObjectTemplate<TDirectoryName, TFileName>;
    friend class DirectoryObjectTemplate<TDirectoryName, TFileName>;

private:
    // テスト用に private を公開します。
    friend class FileSystemObjectTestSuite;
};

}}}
