﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/fs/fs_SubStorage.h>
#include <nn/fssystem/dbm/fs_DbmParameters.h>

namespace nn { namespace fssystem { namespace dbm {

/**
* @brief    ファイルシステムの管理情報領域クラスです。
*
* @details  ファイルシステムの構成情報を格納します。
*           この情報をもとに、Storageを生成するのは使用者の役目です。
*           (そもそも管理領域自体このレイヤーで扱うべき内容ではなさそうです)
*/
class FileSystemControlArea
{
    NN_DISALLOW_COPY(FileSystemControlArea);

private:
    /**
    * @brief        ストレージ上のデータを示します。
    *
    * @details      ストレージ上のデータを示します。
    */
    struct StorageInfo
    {
        int64_t offset;         //! オフセット
        uint32_t size;          //! サイズ
        char reserved[4];
    };
    NN_STATIC_ASSERT(std::is_pod<StorageInfo>::value);

    /**
    * @brief        可変長データ領域のデータを示します。
    *
    * @details      可変長データ領域のデータを示します。
    */
    struct AllocationInfo
    {
        uint32_t index;         //!< 開始インデックス
    };
    NN_STATIC_ASSERT(std::is_pod<AllocationInfo>::value);

public:
    /**
    * @brief        ストレージの先頭部分に以下のような情報を配置します。
    *
    * @details      ストレージの先頭部分に以下のような情報を配置します。
    */
    struct ControlArea
    {
        int64_t blockSize;              //!< ブロックサイズ
        StorageInfo allocationTable;    //!< アローケーションテーブル。非マウント時に拡張可能。
        StorageInfo data;               //!< データ本体。非マウント時に拡張可能。
        AllocationInfo directoryEntry;  //!< ディレクトリエントリーの位置。data に存在。
        AllocationInfo fileEntry;       //!< ファイルエントリーの位置。data に存在。
    };
    NN_STATIC_ASSERT(std::is_pod<ControlArea>::value);

public:
    /**
    * @brief        コンストラクタです。
    *
    * @details      コンストラクタです。
    */
    FileSystemControlArea() NN_NOEXCEPT;

    /**
    * @brief        管理領域をマウントします。
    *
    * @param[in]    storage         ストレージ
    *
    * @details      管理領域をマウントします。
    */
    void Initialize(
             fs::SubStorage storage
         ) NN_NOEXCEPT;

    /**
    * @brief        管理領域をアンマウントします。
    *
    * @details      管理領域をアンマウントします。
    */
    void Finalize() NN_NOEXCEPT;

    /**
    * @brief        ブロックサイズを設定します。
    *
    * @param[in]    sizeBlock       ブロックサイズ
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultSuccess   正常に設定できました。
    * @retval       上記以外        ストレージへの書き込みに失敗しました。
    *
    * @details      ブロックサイズを設定します。
    */
    inline Result WriteBlockSize(int64_t sizeBlock) NN_NOEXCEPT
    {
        return m_Storage.Write(
                   offsetof(ControlArea, blockSize),
                   &sizeBlock,
                   sizeof(int64_t)
               );
    }

    /**
    * @brief        ブロックサイズを取得します。
    *
    * @param[out]   outBlockSize    ブロックサイズ
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultSuccess   正常に取得できました。
    * @retval       上記以外        ストレージからの読み込みに失敗しました。
    *
    * @details      ブロックサイズを取得します。
    */
    inline Result ReadBlockSize(int64_t* outBlockSize) NN_NOEXCEPT
    {
        return m_Storage.Read(
                   offsetof(ControlArea, blockSize),
                   outBlockSize,
                   sizeof(int64_t)
               );
    }

    /**
    * @brief        アロケーションテーブル情報を設定します。
    *
    * @param[in]    offset          アロケーションテーブルの開始オフセット
    * @param[in]    count           アロケーションテーブルが管理するブロック数
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultSuccess   正常に設定できました。
    * @retval       上記以外        ストレージへの書き込みに失敗しました。
    *
    * @details      アロケーションテーブル情報を設定します。
    */
    inline Result WriteAllocationTableInfo(int64_t offset, uint32_t count) NN_NOEXCEPT
    {
        return WriteStorageInfo(
                   offset,
                   count,
                   offsetof(ControlArea, allocationTable)
               );
    }

    /**
    * @brief        アロケーションテーブル情報を取得します。
    *
    * @param[out]   outOffset       アロケーションテーブルの開始オフセット
    * @param[out]   outCount        アロケーションテーブルの個数
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultSuccess   正常に取得できました。
    * @retval       上記以外        ストレージからの読み込みに失敗しました。
    *
    * @details      アロケーションテーブル情報を取得します。
    */
    inline Result ReadAllocationTableInfo(
                      int64_t* outOffset,
                      uint32_t* outCount
                  ) NN_NOEXCEPT
    {
        return ReadStorageInfo(
                   outOffset,
                   outCount,
                   offsetof(ControlArea, allocationTable)
               );
    }

    /**
    * @brief        実データ領域情報を設定します。
    *
    * @param[in]    offset          実データ領域の開始オフセット
    * @param[in]    count           実データ領域の個数
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultSuccess   正常に設定できました。
    * @retval       上記以外        ストレージへの書き込みに失敗しました。
    *
    * @details      実データ領域情報を設定します。
    */
    inline Result WriteDataBodyInfo(
                      int64_t offset,
                      uint32_t count
                  ) NN_NOEXCEPT
    {
        return WriteStorageInfo(
                   offset,
                   count,
                   offsetof(ControlArea, data)
               );
    }

    /**
    * @brief        実データ領域情報を取得します。
    *
    * @param[out]   outOffset       実データ領域の開始オフセット
    * @param[out]   outSize         実データ領域の個数
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultSuccess   正常に取得できました。
    * @retval       上記以外        ストレージからの読み込みに失敗しました。
    *
    * @details      実データ領域情報を取得します。
    */
    inline Result ReadDataBodyInfo(
                      int64_t* outOffset,
                      uint32_t* outCount
                  ) NN_NOEXCEPT
    {
        return ReadStorageInfo(
                   outOffset,
                   outCount,
                   offsetof(ControlArea, data)
               );
    }

    /**
    * @brief        ディレクトリエントリー情報を設定します。
    *
    * @param[in]    index           ディレクトリエントリーの
    *                               アローケーションテーブル上の先頭インデックス
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultSuccess   正常に設定できました。
    * @retval       上記以外        ストレージへの書き込みに失敗しました。
    *
    * @details      ディレクトリエントリー情報を設定します。
    */
    inline Result WriteDirectoryEntryInfo(uint32_t index) NN_NOEXCEPT
    {
        return WriteAllocationInfo(
                   index,
                   offsetof(ControlArea, directoryEntry)
               );
    }

    /**
    * @brief        ディレクトリエントリー情報を取得します。
    *
    * @param[out]   outIndex        ディレクトリエントリーの
    *                               アローケーションテーブル上の先頭インデックス
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultSuccess   正常に取得できました。
    * @retval       上記以外        ストレージからの読み込みに失敗しました。
    *
    * @details      ディレクトリエントリー情報を取得します。
    */
    inline Result ReadDirectoryEntryInfo(uint32_t* outIndex) NN_NOEXCEPT
    {
        return ReadAllocationInfo(
                   outIndex,
                   offsetof(ControlArea, directoryEntry)
               );
    }

    /**
    * @brief        ファイルエントリー情報を設定します。
    *
    * @param[in]    index           ファイルエントリーの
    *                               アロケーションテーブル上での先頭インデックス
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultSuccess   正常に設定できました。
    * @retval       上記以外        ストレージへの書き込みに失敗しました。
    *
    * @details      ファイルエントリー情報を設定します。
    */
    inline Result WriteFileEntryInfo(uint32_t index) NN_NOEXCEPT
    {
        return WriteAllocationInfo(
                   index,
                   offsetof(ControlArea, fileEntry)
               );
    }

    /**
    * @brief        ファイルエントリー情報を取得します。
    *
    * @param[out]   outIndex        ファイルエントリーの
    *                               アロケーションテーブル上での先頭インデックス
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultSuccess   正常に取得できました。
    * @retval       上記以外        ストレージからの読み込みに失敗しました。
    *
    * @details      ファイルエントリー情報を取得します。
    */
    inline Result ReadFileEntryInfo(uint32_t* outIndex) NN_NOEXCEPT
    {
        return ReadAllocationInfo(
                   outIndex,
                   offsetof(ControlArea, fileEntry)
               );
    }

    /**
    * @brief        拡張したアロケーションテーブルの情報を設定します。
    *
    * @param[in]    count           拡張後のアロケーションテーブルの個数
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultSuccess   正常に拡張できました。
    * @retval       上記以外        ストレージでの読み書きに失敗しました。
    *
    * @details      拡張したアロケーションテーブルの情報を設定します。
    */
    Result ExpandAllocationTableInfo(uint32_t count) NN_NOEXCEPT;

private:
    /**
    * @brief        ストレージ上のデータ情報を更新します。

    * @param[in]    offset          ストレージ上のデータのオフセット
    * @param[in]    size            ストレージ上のデータのサイズ
    * @param[in]    accessOffset    管理領域の先頭からのオフセット
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultSuccess   正常に更新できました。
    * @retval       上記以外        ストレージへの書き込みに失敗しました。
    *
    * @pre          マウントしている。
    *
    * @details      ストレージ上のデータ情報を更新します。
    */
    Result WriteStorageInfo(
               int64_t offset,
               uint32_t size,
               uint32_t accessOffset
           ) NN_NOEXCEPT;

    /**
    * @brief        ストレージ上のデータ情報を取得します。
    *
    * @param[out]   outOffset       ストレージ上のデータのオフセット
    * @param[out]   outSize         ストレージ上のデータのサイズ
    * @param[in]    accessOffset    管理領域の先頭からのオフセット
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultSuccess   正常に取得できました。
    * @retval       上記以外        ストレージからの読み込みに失敗しました。
    *
    * @pre          マウントしている。
    * @pre          outOffset != nullptr
    * @pre          outSize != nullptr
    *
    * @details      ストレージ上のデータ情報を取得します。
    */
    Result ReadStorageInfo(
               int64_t* outOffset,
               uint32_t* outSize,
               uint32_t accessOffset
           ) NN_NOEXCEPT;

    /**
    * @brief        ストレージ上の可変長データ情報を更新します。
    *
    * @param[in]    index           ストレージ上のデータの開始インデックス
    * @param[in]    accessOffset    管理領域の先頭からのオフセット
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultSuccess   正常に更新できました。
    * @retval       上記以外        ストレージへの書き込みに失敗しました。
    *
    * @pre          マウントしている。
    *
    * @details      ストレージ上の可変長データ情報を更新します。
    */
    Result WriteAllocationInfo(
               uint32_t index,
               uint32_t accessOffset
           ) NN_NOEXCEPT;

    /**
    * @brief        ストレージ上の可変長データ情報を取得します。
    *
    * @param[out]   outIndex        ストレージ上のデータの開始インデックス
    * @param[in]    accessOffset    管理領域の先頭からのオフセット
    *
    * @return       関数の処理結果を返します。
    * @retval       ResultSuccess   正常に取得できました。
    * @retval       上記以外        ストレージからの読み込みに失敗しました。
    *
    * @pre          マウントしている。
    * @pre          outIndex != nullptr
    *
    * @details      ストレージ上の可変長データ情報を取得します。
    */
    Result ReadAllocationInfo(
               uint32_t* outIndex,
               uint32_t accessOffset
           ) NN_NOEXCEPT;

private:
    fs::SubStorage m_Storage;           //! 管理領域用ストレージ
};

}}}

