﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <cstring>
#include <string>

#include <nn/nn_StaticAssert.h>
#include <nn/nn_SdkAssert.h>

#include <nn/fssystem/fs_PathTool.h>
#include <nn/fssystem/dbm/fs_DbmParameters.h>

namespace nn { namespace fssystem { namespace dbm {

/**
* @brief        ファイルパス操作関数群。
*
* @details      ファイルパス操作関数群。
*/
class PathTool
{
    NN_DISALLOW_COPY(PathTool);

public:
    /**
    * @brief        カレントディレクトリを示す文字かどうかを取得します。
    *
    * @param[in]    pDirectoryName  ディレクトリ名
    * @param[in]    length          ディレクトリ名の長さ
    *
    * @return       カレントディレクトリかどうか。
    *
    * @details      カレントディレクトリを示す文字かどうかを取得します。
    */
    static bool IsCurrentDirectory(
                    const DirectoryName* pDirectoryName,
                    size_t length
                ) NN_NOEXCEPT
    {
        return ((length == 1) && (pDirectoryName->name[0] == StringTraits::Dot));
    }

    /**
    * @brief        親ディレクトリを示す文字かどうかを取得します。
    *
    * @param[in]    pDirectoryName  ディレクトリ名
    * @param[in]    length          ディレクトリ名の長さ
    *
    * @return       カレントディレクトリかどうか。
    *
    * @details 親ディレクトリを示す文字かどうかを取得します。
    */
    static bool IsParentDirectory(
                    const DirectoryName* pDirectoryName,
                    size_t length
                ) NN_NOEXCEPT
    {
        if( length != 2 )
        {
            return false;
        }
        if( pDirectoryName->name[0] != StringTraits::Dot )
        {
            return false;
        }
        if( pDirectoryName->name[1] != StringTraits::Dot )
        {
            return false;
        }
        return true;
    }

    /**
    * @brief        ディレクトリ名が等しいかどうかを取得します。
    *
    * @param[in]    pDirectoryName1 ディレクトリ名
    * @param[in]    pDirectoryName2 ディレクトリ名
    *
    * @return       ディレクトリ名が等しいかどうか。
    *
    * @pre          pDirectoryName1 != nullptr
    * @pre          pDirectoryName2 != nullptr
    *
    * @details      ディレクトリ名が等しいかどうかを取得します。
    */
    static inline bool IsEqualDirectoryName(
                           const DirectoryName* pDirectoryName1,
                           const DirectoryName* pDirectoryName2
                       ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pDirectoryName1);
        NN_SDK_REQUIRES_NOT_NULL(pDirectoryName2);
        return (std::memcmp(pDirectoryName1, pDirectoryName2, sizeof(DirectoryName)) == 0);
    }

    /**
    * @brief        指定したディレクトリ名の長さを取得します。
    *
    * @param[in]    pName   ディレクトリ名
    *
    * @return       ディレクトリ名の長さ。
    *
    * @pre          pName != nullptr
    *
    * @details      指定したディレクトリ名の長さを取得します。
    */
    static size_t GetDirectoryNameLength(const DirectoryName* pName) NN_NOEXCEPT;

    /**
    * @brief        バイトデータをディレクトリ名に変換します。
    *
    * @param[out]   outName 変換後のディレクトリ名
    * @param[in]    pSrc    文字列を含んだバイトデータ
    * @param[in]    length  バイトデータの長さ
    *
    * @return       ディレクトリ名への変換に成功した場合は true、失敗した場合は false。
    *
    * @pre          outName != nullptr
    * @pre          pSrc != nullptr
    *
    * @details      バイトデータをディレクトリ名に変換します。
    */
    static bool GetDirectoryName(
                    DirectoryName* outName,
                    const char* pSrc,
                    size_t length
                ) NN_NOEXCEPT;

    /**
    * @brief        ファイル名が等しいかどうかを取得します。
    *
    * @param[in]    pFileName1  ファイル名
    * @param[in]    pFileName2  ファイル名
    *
    * @return       ファイル名が等しいかどうか。
    *
    * @pre          pFileName1 != nullptr
    * @pre          pFileName2 != nullptr
    *
    * @details      ファイル名が等しいかどうかを取得します。
    */
    static inline bool IsEqualFileName(
                           const FileName* pFileName1,
                           const FileName* pFileName2
                       ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pFileName1);
        NN_SDK_REQUIRES_NOT_NULL(pFileName2);
        return (std::memcmp(pFileName1, pFileName2, sizeof(FileName)) == 0);
    }

    /**
    * @brief        指定したファイル名の長さを取得します。
    *
    * @param[in]    pName   ファイル名
    *
    * @return       ファイル名の長さ。
    *
    * @pre          pName != nullptr
    *
    * @details      指定したファイル名の長さを取得します。
    */
    static size_t GetFileNameLength(const FileName* pName) NN_NOEXCEPT;

    /**
    * @brief        バイトデータをファイル名に変換します。
    *
    * @param[out]   outName 変換後のファイル名
    * @param[in]    pSrc    文字列を含んだバイトデータ
    * @param[in]    length  バイトデータの長さ
    *
    * @return       ファイル名への変換に成功した場合は true、失敗した場合は false。
    *
    * @pre          outName != nullptr
    * @pre          pSrc != nullptr
    *
    * @details      バイトデータをファイル名に変換します。
    */
    static bool GetFileName(
                    FileName* outName,
                    const char* pSrc,
                    size_t length
                ) NN_NOEXCEPT;

    /**
    * @brief        ディレクトリ名データをファイル名データに変換します。
    *
    * @param[out]   outFileName     変換後のファイル名
    * @param[in]    pDirectoryName  ディレクトリ名
    *
    * @return       変換に成功した場合は true、そうでなければ false。
    *
    * @pre          outFileName != nullptr
    * @pre          pDirectoryName != nullptr
    *
    * @details      ディレクトリ名データをファイル名データに変換します。
    */
    static bool ConvertDirectoryNameToFileName(
                    FileName* outFileName,
                    const DirectoryName* pDirectoryName
                ) NN_NOEXCEPT;

    /**
    * @brief        ファイル名データをディレクトリ名データに変換します。
    *
    * @param[out]   outDirectoryName 変換後のディレクトリ名
    * @param[in]    pFileName ファイル名
    *
    * @return       変換に成功した場合は true、そうでなければ false。
    *
    * @pre          outDirectoryName != nullptr
    * @pre          pFileName != nullptr
    *
    * @details      ファイル名データをディレクトリ名データに変換します。
    */
    static bool ConvertFileNameToDirectoryName(
                    DirectoryName* outDirectoryName,
                    const FileName* pFileName
                ) NN_NOEXCEPT;

    /**
    * @brief フルパスを解析し、イテレーションするためのクラスです。
    *
    * @details フルパスを解析し、イテレーションするためのクラスです。
    */
    class PathParser
    {
         NN_DISALLOW_COPY(PathParser);

    public:
        /**
        * @brief        コンストラクタです。
        *
        * @details      コンストラクタです。
        */
        PathParser() NN_NOEXCEPT;

        /**
        * @brief        パーサを初期化します。
        *
        * @param[in]    pFullPath   フルパス
        *
        * @return       関数の処理結果を返します。
        * @retval       ResultSuccess              処理が正常に終了しました。
        * @retval       ResultInvalidPathFormat     フルパスがルートディレクトリから始まっていません。
        *
        * @pre          pFullPath != nullptr
        *
        * @details      パーサを初期化します。
        */
        Result Initialize(const PathChar* pFullPath) NN_NOEXCEPT;

        /**
        * @brief        パーサを破棄します。
        *
        * @details      パーサを破棄します。
        */
        void Finalize() NN_NOEXCEPT;

        /**
        * @brief        パスのパースが完了したかどうかを取得します。
        *
        * @return       パスのパースが完了したかどうか。
        *
        * @details      パスのパースが完了したかどうかを取得します。
        */
        bool IsParseFinished() const NN_NOEXCEPT;

        /**
        * @brief        パスがディレクトリかどうかを取得します。
        *
        * @return       パスがディレクトリかどうか。
        *
        * @pre          初期化している。
        * @pre          IsParseFinished() == true
        *
        * @details      パスがディレクトリかどうかを取得します。
        */
        bool IsDirectoryPath() const NN_NOEXCEPT;

        /**
        * @brief        最上位ディレクトリ部分を取得します。
        *
        * @param[out]   outDirectoryName        ディレクトリ名
        * @param[out]   outDirectoryNameLength  ディレクトリ名の長さ
        *
        * @return       関数の処理結果を返します。
        * @retval       ResultSuccess                  処理が正常に終了しました。
        * @retval       ResultTooLongPath           ディレクトリ名が長すぎます。
        *
        * @pre          初期化している。
        * @pre          outDirectoryName != nullptr
        * @pre          outDirectoryNameLength != nullptr
        *
        * @details      最上位ディレクトリ部分を取得します。
        */
        Result GetNextDirectoryName(
                   DirectoryName* outDirectoryName,
                   size_t* outDirectoryNameLength
               ) NN_NOEXCEPT;

        /**
        * @brief        文字列の先頭から次の１節をディレクトリ名として取得します。
        *
        * @param[out]   outDirectoryName        ディレクトリ名
        * @param[out]   outDirectoryNameLength  ディレクトリ名の長さ
        *
        * @return       関数の処理結果を返します。
        * @retval       ResultSuccess                  処理が正常に終了しました。
        * @retval       ResultTooLongPath              ディレクトリ名が長すぎます。
        *
        * @pre          初期化している。
        * @pre          outDirectoryName != nullptr
        * @pre          outDirectoryNameLength != nullptr
        *
        * @details      文字列の先頭から次の１節をディレクトリ名として取得します。
        */
        Result GetAsDirectoryName(
                   DirectoryName* outDirectoryName,
                   size_t* outDirectoryNameLength
               ) const NN_NOEXCEPT;

        /**
        * @brief        文字列の先頭から次の１節をファイル名として取得します。
        *
        * @param[out]   outFileName         ファイル名
        * @param[out]   outFileNameLength   ファイル名の長さ
        *
        * @return       関数の処理結果を返します。
        * @retval       ResultSuccess                  処理が正常に終了しました。
        * @retval       ResultTooLongPath       ファイル名が長すぎます。
        * @retval       ResultInvalidPathFormat パスの最後が不正です。
        *
        * @pre          初期化している。
        * @pre          outFileName != nullptr
        * @pre          outFileNameLength != nullptr
        *
        * @details      文字列の先頭から次の１節をファイル名として取得します。
        */
        Result GetAsFileName(
                   FileName* outFileName,
                   size_t* outFileNameLength
               ) const NN_NOEXCEPT;

    private:
        const PathChar* m_PreviousStartPath;   //! 解析済みのパス始点へのポインタ
        const PathChar* m_PreviousEndPath;     //! 解析済みのパス終端へのポインタ
        const PathChar* m_NextPath;            //! 次回解析するパスへのポインタ
        bool m_IsParseFinished;                //! NUL 文字に到着したかどうか
    };
};

}}}

